/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    ecap_ex08_capture_three_phase_pwm.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "ecap_ex08_capture_three_phase_pwm.h"
#include "printf.h"
#include "stdlib.h"
#include "ecap_ex08_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define delta 100U
#define log_times 256

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

uint32_t myEPWM1_DIR;
uint32_t myEPWM2_DIR;
uint32_t myEPWM3_DIR;
vuint32_t pwm1_cmpaAVal;
vuint32_t pwm2_cmpaAVal;
vuint32_t pwm3_cmpaAVal;
vuint32_t pwm1_cmpaBVal;
vuint32_t pwm2_cmpaBVal;
vuint32_t pwm3_cmpaBVal;
vuint32_t INTRs = 0;
float real1[log_times] = {0};
float real2[log_times] = {0};
float measure1[log_times] = {0};
float measure2[log_times] = {0};
float Arr1[log_times] = {0};
float Arr2[log_times] = {0};
uint32_t size1;
uint32_t size2;

float duty;
uint32_t ecapLow;
uint32_t ecapHigh;

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

void myEPWM0_init(void);
void EPWM_init(void);
void SYNC_init(void);

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */


void addToStaticArray(float *arr, uint32_t size, float value) {
    if (size >= log_times) {
        size %= log_times;
    }
    arr[size] = value;
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void main(void){

	Device_init();

	myEPWM1_DIR = EPWM_TIMER_UP;
	myEPWM2_DIR = EPWM_TIMER_UP;
	myEPWM3_DIR = EPWM_TIMER_UP;
	size1 = 0;
	size2 = 0;

	// Board initialization
    //
	Board_init();

	EPWM_init();
	SYNC_init();

    // Loop forever. Suspend or place breakpoints to observe the buffers.
    while(1)
    {

    }
}

// EPWM Configurations
void EPWM_init()
{
	myEPWM0_init();
}

// EPWM Configurations for EPWM1~6
void myEPWM0_init()
{
	// Disable sync(Freeze clock to PWM as well)
	//
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

	// Configure ePWM
	// epwm1:epwm1A is always going one cycle faster than epwm1B because epwm1B misses the first falling edge at the beginning.
	EPWM_disableInterrupt(myEPWM1_BASE);
	EPWM_disableInterruptEventCountInit(myEPWM1_BASE);
	EPWM_disablePhaseShiftLoad(myEPWM1_BASE);
	EPWM_setTimeBaseCounterMode(myEPWM1_BASE, EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setTimeBasePeriod(myEPWM1_BASE, PWM_PRD);
	EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A,(PWM_CMP_MAX+PWM_CMP_MIN)/2); // 默认在ctr=0进行影子加载
	EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B,(PWM_CMP_MAX+PWM_CMP_MIN)/2); // 默认在ctr=0进行影子加载
	EPWM_setPhaseShift(myEPWM1_BASE, 0U);
	EPWM_setSyncInPulseSource(myEPWM1_BASE,EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP7);
	EPWM_enableSyncOutPulseSource(myEPWM1_BASE,EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
	EPWM_setActionQualifierAction(myEPWM1_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_HIGH,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(myEPWM1_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_LOW,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setActionQualifierAction(myEPWM1_BASE,
								  EPWM_AQ_OUTPUT_B,
								  EPWM_AQ_OUTPUT_LOW,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(myEPWM1_BASE,
								  EPWM_AQ_OUTPUT_B,
								  EPWM_AQ_OUTPUT_HIGH,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setClockPrescaler(myEPWM1_BASE,
						   EPWM_CLOCK_DIVIDER_1,
						   EPWM_HSCLOCK_DIVIDER_1);
	EPWM_setCountModeAfterSync(myEPWM1_BASE,EPWM_COUNT_MODE_UP_AFTER_SYNC);
	EPWM_setInterruptSource(myEPWM1_BASE,EPWM_INT_TBCTR_ZERO);
	EPWM_setInterruptEventCountInitValue(myEPWM1_BASE,0U);
	EPWM_setInterruptEventCount(myEPWM1_BASE,1U);
	EPWM_enablePhaseShiftLoad(myEPWM1_BASE);
	EPWM_enableInterruptEventCountInit(myEPWM1_BASE);
	EPWM_enableInterrupt(myEPWM1_BASE);

	// epwm2
	EPWM_disableInterrupt(myEPWM2_BASE);
	EPWM_disableInterruptEventCountInit(myEPWM2_BASE);
	EPWM_disablePhaseShiftLoad(myEPWM2_BASE);
	EPWM_setTimeBaseCounterMode(myEPWM2_BASE, EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setTimeBasePeriod(myEPWM2_BASE, PWM_PRD);
	EPWM_setCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_A,(PWM_CMP_MAX+PWM_CMP_MIN)/2);
	EPWM_setCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_B,(PWM_CMP_MAX+PWM_CMP_MIN)/2);
	EPWM_setPhaseShift(myEPWM2_BASE, PHASE_SHIFT);
	EPWM_setSyncInPulseSource(myEPWM2_BASE,EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP7);
	EPWM_enableSyncOutPulseSource(myEPWM2_BASE,EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
	EPWM_setActionQualifierAction(myEPWM2_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_HIGH,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(myEPWM2_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_LOW,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setActionQualifierAction(myEPWM2_BASE,
								  EPWM_AQ_OUTPUT_B,
								  EPWM_AQ_OUTPUT_LOW,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(myEPWM2_BASE,
								  EPWM_AQ_OUTPUT_B,
								  EPWM_AQ_OUTPUT_HIGH,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setClockPrescaler(myEPWM2_BASE,
						   EPWM_CLOCK_DIVIDER_1,
						   EPWM_HSCLOCK_DIVIDER_1);
	EPWM_setCountModeAfterSync(myEPWM2_BASE,EPWM_COUNT_MODE_DOWN_AFTER_SYNC);
	EPWM_setInterruptSource(myEPWM2_BASE,EPWM_INT_TBCTR_ZERO);
	EPWM_setInterruptEventCountInitValue(myEPWM2_BASE,0U);
	EPWM_setInterruptEventCount(myEPWM2_BASE,1U);
	EPWM_enablePhaseShiftLoad(myEPWM2_BASE);
	EPWM_enableInterruptEventCountInit(myEPWM2_BASE);
	EPWM_enableInterrupt(myEPWM2_BASE);

	// epwm3
	EPWM_disableInterrupt(myEPWM3_BASE);
	EPWM_disableInterruptEventCountInit(myEPWM3_BASE);
	EPWM_disablePhaseShiftLoad(myEPWM3_BASE);
	EPWM_setTimeBaseCounterMode(myEPWM3_BASE, EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setTimeBasePeriod(myEPWM3_BASE, PWM_PRD);
	EPWM_setCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_A,(PWM_CMP_MAX+PWM_CMP_MIN)/2);
	EPWM_setCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_B,(PWM_CMP_MAX+PWM_CMP_MIN)/2);
	EPWM_setPhaseShift(myEPWM3_BASE, PHASE_SHIFT);
	EPWM_setSyncInPulseSource(myEPWM3_BASE,EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP7);
	EPWM_enableSyncOutPulseSource(myEPWM3_BASE,EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
	EPWM_setActionQualifierAction(myEPWM3_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_HIGH,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(myEPWM3_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_LOW,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setActionQualifierAction(myEPWM3_BASE,
								  EPWM_AQ_OUTPUT_B,
								  EPWM_AQ_OUTPUT_LOW,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(myEPWM3_BASE,
								  EPWM_AQ_OUTPUT_B,
								  EPWM_AQ_OUTPUT_HIGH,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setClockPrescaler(myEPWM3_BASE,
						   EPWM_CLOCK_DIVIDER_1,
						   EPWM_HSCLOCK_DIVIDER_1);
	EPWM_setCountModeAfterSync(myEPWM3_BASE,EPWM_COUNT_MODE_UP_AFTER_SYNC);
	EPWM_setInterruptSource(myEPWM3_BASE,EPWM_INT_TBCTR_ZERO);
	EPWM_setInterruptEventCountInitValue(myEPWM3_BASE,0U);
	EPWM_setInterruptEventCount(myEPWM3_BASE,1U);
	EPWM_enablePhaseShiftLoad(myEPWM3_BASE);
	EPWM_enableInterruptEventCountInit(myEPWM3_BASE);
	EPWM_enableInterrupt(myEPWM3_BASE);

	// Enable sync and clock to PWM
	//
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

void SYNC_init(void)
{
	// treat ecap sync-out as sync-source
	ECAP_stopCounter(SYNC_BASE);

	ECAP_enableAPWMMode(SYNC_BASE);

	ECAP_setAPWMPeriod(SYNC_BASE,9000U);

	ECAP_setAPWMCompare(SYNC_BASE,4500U);

	ECAP_setAPWMPolarity(SYNC_BASE,ECAP_APWM_ACTIVE_LOW);

	ECAP_setPhaseShiftCount(SYNC_BASE,0U);

	ECAP_enableLoadCounter(SYNC_BASE);

	ECAP_setSyncOutMode(SYNC_BASE,ECAP_SYNC_OUT_SYNCI);

	ECAP_setEmulationMode(SYNC_BASE,ECAP_EMULATION_STOP);

	ECAP_setSyncInPulseSource(SYNC_BASE,ECAP_SYNC_IN_PULSE_SRC_DISABLE);

	ECAP_startCounter(SYNC_BASE);

	ECAP_loadCounter(SYNC_BASE);

	GPIO_setPinConfig(GPIO_9_OUTPUTXBAR6);
	XBAR_enableOutputMux(OUTPUTXBAR1_BASE,XBAR_OUTPUT6, XBAR_MUX12);
	XBAR_setOutputMuxConfig(OUTPUTXBAR1_BASE,XBAR_OUTPUT6, XBAR_OUT_MUX12_ECAP7_OUT);
}

void Ecap_ex08_IrqHandler1(void)
{
	++INTRs;

	// duty record
	if ((ECAP_getInterruptSource(myECAP1_BASE) & ECAP_ISR_SOURCE_CAPTURE_EVENT_2))
	{
		ecapHigh = ECAP_getEventTimeStamp(myECAP1_BASE,ECAP_EVENT_2);
		ecapLow = ECAP_getEventTimeStamp(myECAP1_BASE,ECAP_EVENT_1);
		duty = (float)ecapHigh/(2*PWM_PRD);
		addToStaticArray(measure1,size1,ecapHigh);
		addToStaticArray(real1,size1,2*PWM_PRD-pwm1_cmpaAVal-pwm1_cmpaBVal);
		addToStaticArray(Arr1,size1++,duty);
		ECAP_clearInterrupt(myECAP1_BASE,0xFEU);
		ECAP_clearGlobalInterrupt(myECAP1_BASE);
		ECAP_reArm(myECAP1_BASE);
	}
	if ((ECAP_getInterruptSource(myECAP2_BASE) & ECAP_ISR_SOURCE_CAPTURE_EVENT_2))
	{
		ecapHigh = ECAP_getEventTimeStamp(myECAP2_BASE,ECAP_EVENT_1);
		ecapLow = ECAP_getEventTimeStamp(myECAP2_BASE,ECAP_EVENT_2);
		duty = 1-(float)ecapLow/(2*PWM_PRD);
		addToStaticArray(measure2,size2,2*PWM_PRD-ecapLow);
		addToStaticArray(real2,size2,pwm1_cmpaAVal+pwm1_cmpaBVal);
		addToStaticArray(Arr2,size2++,duty);
		ECAP_clearInterrupt(myECAP2_BASE,0xFEU);
		ECAP_clearGlobalInterrupt(myECAP2_BASE);
		ECAP_reArm(myECAP2_BASE);
	}

	// epwm1
	pwm1_cmpaAVal = EPWM_getCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A);
	pwm1_cmpaBVal = EPWM_getCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B);
	if (pwm1_cmpaAVal >= PWM_CMP_MAX)
	{
		myEPWM1_DIR = EPWM_TIMER_DOWN;
	}
	else if (pwm1_cmpaAVal <= PWM_CMP_MIN)
	{
		myEPWM1_DIR = EPWM_TIMER_UP;
	}
	if (myEPWM1_DIR == EPWM_TIMER_DOWN)
	{
		EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, pwm1_cmpaAVal-delta);
		EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B, pwm1_cmpaBVal-delta);
	}
	else if (myEPWM1_DIR == EPWM_TIMER_UP)
	{
		EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, pwm1_cmpaAVal+delta);
		EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B, pwm1_cmpaBVal+delta);
	}

	EPWM_clearEventTriggerInterruptFlag(myEPWM1_BASE);
}

void Ecap_ex08_IrqHandler2(void)
{
	++INTRs;
	// duty record
	if ((ECAP_getInterruptSource(myECAP3_BASE) & ECAP_ISR_SOURCE_CAPTURE_EVENT_2))
	{
		ecapHigh = ECAP_getEventTimeStamp(myECAP3_BASE,ECAP_EVENT_2);
		ecapLow = ECAP_getEventTimeStamp(myECAP3_BASE,ECAP_EVENT_1);
		ECAP_clearInterrupt(myECAP3_BASE,0xFEU);
		ECAP_clearGlobalInterrupt(myECAP3_BASE);
		ECAP_reArm(myECAP3_BASE);
	}
	if ((ECAP_getInterruptSource(myECAP4_BASE) & ECAP_ISR_SOURCE_CAPTURE_EVENT_2))
	{
		ecapHigh = ECAP_getEventTimeStamp(myECAP4_BASE,ECAP_EVENT_2);
		ecapLow = ECAP_getEventTimeStamp(myECAP4_BASE,ECAP_EVENT_1);
		ECAP_clearInterrupt(myECAP4_BASE,0xFEU);
		ECAP_clearGlobalInterrupt(myECAP4_BASE);
		ECAP_reArm(myECAP4_BASE);
	}

	// epwm2
	pwm2_cmpaAVal = EPWM_getCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_A);
	pwm2_cmpaBVal = EPWM_getCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_B);
	if (pwm2_cmpaAVal >= PWM_CMP_MAX)
	{
		myEPWM2_DIR = EPWM_TIMER_DOWN;
	}
	else if (pwm2_cmpaAVal <= PWM_CMP_MIN)
	{
		myEPWM2_DIR = EPWM_TIMER_UP;
	}
	if (myEPWM2_DIR == EPWM_TIMER_DOWN)
	{
		EPWM_setCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_A, pwm2_cmpaAVal-delta);
		EPWM_setCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_B, pwm2_cmpaBVal-delta);
	}
	else if (myEPWM2_DIR == EPWM_TIMER_UP)
	{
		EPWM_setCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_A, pwm2_cmpaAVal+delta);
		EPWM_setCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_B, pwm2_cmpaBVal+delta);
	}

	EPWM_clearEventTriggerInterruptFlag(myEPWM2_BASE);
}

void Ecap_ex08_IrqHandler3(void)
{
	++INTRs;
	// duty record
	if ((ECAP_getInterruptSource(myECAP5_BASE) & ECAP_ISR_SOURCE_CAPTURE_EVENT_2))
	{
		ecapHigh = ECAP_getEventTimeStamp(myECAP5_BASE,ECAP_EVENT_2);
		ecapLow = ECAP_getEventTimeStamp(myECAP5_BASE,ECAP_EVENT_1);
		ECAP_clearInterrupt(myECAP5_BASE,0xFEU);
		ECAP_clearGlobalInterrupt(myECAP5_BASE);
		ECAP_reArm(myECAP5_BASE);
	}
	if ((ECAP_getInterruptSource(myECAP6_BASE) & ECAP_ISR_SOURCE_CAPTURE_EVENT_2))
	{
		ecapHigh = ECAP_getEventTimeStamp(myECAP6_BASE,ECAP_EVENT_2);
		ecapLow = ECAP_getEventTimeStamp(myECAP6_BASE,ECAP_EVENT_1);
		ECAP_clearInterrupt(myECAP6_BASE,0xFEU);
		ECAP_clearGlobalInterrupt(myECAP6_BASE);
		ECAP_reArm(myECAP6_BASE);
	}

	// epwm3
	pwm3_cmpaAVal = EPWM_getCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_A);
	pwm3_cmpaBVal = EPWM_getCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_B);
	if (pwm3_cmpaAVal >= PWM_CMP_MAX)
	{
		myEPWM3_DIR = EPWM_TIMER_DOWN;
	}
	else if (pwm3_cmpaAVal <= PWM_CMP_MIN)
	{
		myEPWM3_DIR = EPWM_TIMER_UP;
	}

	if (myEPWM3_DIR == EPWM_TIMER_DOWN)
	{
		EPWM_setCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_A, pwm3_cmpaAVal-delta);
		EPWM_setCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_B, pwm3_cmpaBVal-delta);
	}
	else if (myEPWM3_DIR == EPWM_TIMER_UP)
	{
		EPWM_setCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_A, pwm3_cmpaAVal+delta);
		EPWM_setCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_B, pwm3_cmpaBVal+delta);
	}

	EPWM_clearEventTriggerInterruptFlag(myEPWM3_BASE);
}

#ifdef __cplusplus
}
#endif
