/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>
#include <string.h>
#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"
#include "fft.h"
#include "gs32_math.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define FFT_TW_REAL_BASE_ADDR    (0x4008C400)
#define FFT_TW_IMAG_BASE_ADDR    (0x4008CC00)
#define FFT_WIN_COEF_BASE_ADDR   (0x4008D400)

#define FFT_DATA_BASE_ADDR       (0x40088400)
#define FFT_RESULT_BASE_ADDR     (0x40088400)

#define FFT_POINTS               2048

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
__attribute__((aligned(32))) uint32_t fft_result[FFT_POINTS*2];

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

extern uint32_t tw_imag[512];
extern uint32_t tw_real[512];
extern uint32_t win_coef[1024];
extern const uint32_t G_R_golden[2048];
extern const uint32_t G_I_golden[2048];

extern uint32_t a_li_lr[FFT_POINTS];

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */


void fft_init(int dma_en)
{
    HWREG(SYSCTL_BASE + 0x404) = 1;

    FFT_SET_TYPE(FFT_BASE, FFT_TYPE_REAL);
    FFT_SET_LEN(FFT_BASE, REAL_FFT_LEN_2048);
}

void fft_load_factor_by_cpu(void)
{
	uint32_t i;

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_REAL_BASE_ADDR)[i] = tw_real[i];
	}

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_IMAG_BASE_ADDR)[i] = tw_imag[i];
	}

	for (i=0; i<1024; i+=1) {
		((volatile uint32_t *)FFT_WIN_COEF_BASE_ADDR)[i] = win_coef[i];
	}
}

void fft_load_data_by_cpu(void)
{
    for (uint32_t i=0; i<sizeof(a_li_lr)/sizeof(uint32_t); i+=1) {
        ((volatile uint32_t *)FFT_DATA_BASE_ADDR)[i] = a_li_lr[i];	// ݷdata_ram
    }
}

void fft_dump_result_by_cpu(void)
{
    for (uint32_t i=0; i<sizeof(fft_result)/sizeof(uint32_t); i+=1) {
        fft_result[i] = ((volatile uint32_t *)FFT_RESULT_BASE_ADDR)[i];
    }
}

void fft_show_result(uint32_t *result, uint32_t points)
{
    log_debug("------- dump fft result start ----------\r\n");

    for (uint32_t i=0; i<points; i+=2) {
        log_debug("0x%08X:%08X %08X\n", FFT_RESULT_BASE_ADDR + i*4, result[i], result[i+1]);
    }

    log_debug("------- dump fft result end ----------\r\n");
}


void fft_dump_reg(void)
{
    log_debug("------- dump fft register start ----------\n");
    log_debug("           CFG_0x%02X: 0x%08X\n", FFT_O_FFT_CFG, HWREG(FFT_BASE + FFT_O_FFT_CFG));
    log_debug("         START_0x%02X: 0x%08X\n", FFT_O_FFT_START, HWREG(FFT_BASE + FFT_O_FFT_START));
    log_debug("DSPDMATRIG_CFG_0x%02X: 0x%08X\n", FFT_O_FFT_DSPDMATRIG_CFG, HWREG(FFT_BASE + FFT_O_FFT_DSPDMATRIG_CFG));
    log_debug("  AUTOSTART_EN_0x%02X: 0x%08X\n", FFT_O_FFT_AUTOSTART_EN, HWREG(FFT_BASE + FFT_O_FFT_AUTOSTART_EN));
    log_debug("        STATUS_0x%02X: 0x%08X\n", FFT_O_FFT_STATUS, HWREG(FFT_BASE + FFT_O_FFT_STATUS));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_FFT_INT_RAW, HWREG(FFT_BASE + FFT_O_FFT_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_FFT_INT_MASK, HWREG(FFT_BASE + FFT_O_FFT_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FLG, HWREG(FFT_BASE + FFT_O_FFT_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FRC, HWREG(FFT_BASE + FFT_O_FFT_INT_FRC));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_WIN_INT_RAW, HWREG(FFT_BASE + FFT_O_WIN_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_WIN_INT_MASK, HWREG(FFT_BASE + FFT_O_WIN_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FLG, HWREG(FFT_BASE + FFT_O_WIN_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FRC, HWREG(FFT_BASE + FFT_O_WIN_INT_FRC));
}


int main(void)
{

	Device_init();

	UartPrint_init(SCIA_BASE, 110000);

	log_set_level(LOG_DEBUG);

	log_info("Hello DSP!\r\n");
	log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);

    uint32_t i;
    uint32_t errCount = 0;
    uint32_t errIndex[2048];

    log_debug("------- %s start ----------\r\n", __FUNCTION__);

    fft_dump_reg();		// ӡĴϢ

    fft_init(0);

    fft_load_factor_by_cpu();						// תдram
    fft_load_data_by_cpu();							// a_li_lrdata_ram

    memset(fft_result, 0xAA, sizeof(fft_result));	// Ϊfft_resultٿռ

    log_debug("start windowing ...\r\n");
    FFT_WIN_SOFT_START(FFT_BASE);					// Ӵ
    fft_dump_reg();									// һδӡĴϢ
    while(FFT_IS_WIN_BUSY(FFT_BASE));				// жǷڼӴ
    log_debug("windowing finished.\r\n");

    log_debug("start calculating ...\r\n");
    FFT_SOFT_START(FFT_BASE);						//fft
    fft_dump_reg();									//ӡĴϢ
    while(FFT_IS_BUSY(FFT_BASE));
    log_debug("fft calculate finished.\r\n");

    FFT_CLEAR_WIN_INTERRUPT(FFT_BASE);					// ж
    FFT_CLEAR_INTERRUPT_FLAG(FFT_BASE, FFT_INT_CLR_OVF_INT_CLR | FFT_INT_CLR_DONE_INT_CLR);

    fft_dump_result_by_cpu();						// ת͵
    fft_dump_reg();									// ӡĴ

    fft_show_result(fft_result, FFT_POINTS*2);

    for (i=0; i<FFT_POINTS; i+=1) {
        if (abs(G_R_golden[i] - fft_result[i*2]) > 0x10 || abs(G_I_golden[i] - fft_result[i*2+1]) > 0x10) {
        	errIndex[errCount++] = i;	// golden.cԱȣ0x10һ
        }
    }

    for(i=0;i<errCount;i++) {
        log_debug("fft result check fail, real[%d]: %08X/%08X, imag[%d]: %08X/%08X\r\n",
        		errIndex[i], G_R_golden[errIndex[i]], fft_result[errIndex[i]*2], errIndex[i], G_I_golden[errIndex[i]], fft_result[errIndex[i]*2+1]);
    }
    if(errCount==0) {
        log_info("fft result check pass.\r\n");
    }
    else {
    	log_info("There are %d error fft result.\r\n ", errCount);
    }

    log_debug("------- %s end ----------\r\n", __FUNCTION__);

	/* We should never get here as control is now taken by the scheduler */
	for(;;);
}

