/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <string.h>
#include <stdlib.h>
#include "device.h"
#include "board_cfg.h"
#include "log.h"
#include "fft.h"
#include "dma.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define FFT_TW_REAL_BASE_ADDR       (0x4008C400)
#define FFT_TW_IMAG_BASE_ADDR       (0x4008CC00)
#define FFT_WIN_COEF_BASE_ADDR      (0x4008D400)

#define FFT_DATA_BASE_ADDR          (0x40088400)
#define FFT_RESULT_BASE_ADDR        (0x40088400)

#define FFT_POINTS                  2048
#define FFT_DMA_CHANNEL_NUM         0                       //from 0
#define FFT_DMA_CHANNEL_BASE        DMA_CH1_BASE            //from 0
#define FFT_RESULT_DMA_IRQn         INT_DMA_CH1

#define FFT_DMA_REQ_ID              (48)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
volatile uint32_t dspTicks = 0;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
extern const uint32_t tw_imag[512];
extern const uint32_t tw_real[512];
extern const uint32_t win_coef[1024];
extern const uint32_t G_R_golden[2048];
extern const uint32_t G_I_golden[2048];

extern uint32_t a_li_lr[FFT_POINTS];

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

void fft_dump_result_by_dma(void);

__attribute__((aligned(32))) uint32_t fft_result[FFT_POINTS*2];

volatile uint32_t testStep = 0;

void FFT_IRQHandler(void)//ɲж
{
    FFT_CLEAR_INTERRUPT_FLAG(FFT_BASE, FFT_INT_CLR_DONE_INT_CLR);

    fft_dump_result_by_dma();

    testStep += 1;
}

void DMA1_CH1_IRQHandler(void)//ӦIRQ96
{
    DMA_clearInterrupt(FFT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);

    if (testStep == 0) {
        //loading fft data finished
    	FFT_DSPDMA_TRIG_ENABLE(FFT_BASE, FFT_DMA_CHANNEL_NUM);
    } else {
        //dumping fft result finished

        fft_result[4095] = ((volatile uint32_t*)FFT_RESULT_BASE_ADDR)[4095];
        testStep += 1;
    }
}

void fft_init(int trigByDMA)
{
    HWREG(SYSCTL_BASE + 0x404) = 1;
    //memset((void *)FFT_DATA_BASE_ADDR, 0, 24*1024);
    FFT_SET_TYPE(FFT_BASE, FFT_TYPE_REAL);
    FFT_SET_LEN(FFT_BASE, REAL_FFT_LEN_2048);

    if (trigByDMA) {
        FFT_WIN_AUTOSTART_ENABLE(FFT_BASE);
        FFT_AUTOSTART_ENABLE(FFT_BASE);

        FFT_DSPDMA_TRIG_ENABLE(FFT_BASE, FFT_DMA_CHANNEL_NUM);

        //enable fft irq
        FFT_CLEAR_WIN_INTERRUPT(FFT_BASE);
        FFT_CLEAR_INTERRUPT_FLAG(FFT_BASE, FFT_INT_CLR_OVF_INT_CLR | FFT_INT_CLR_DONE_INT_CLR);

        FFT_INTERRUPT_ENABLE(FFT_BASE, FFT_DONE_INT_MASK);
        ECLIC_Register_IRQ(INT_FFT, ECLIC_NON_VECTOR_INTERRUPT, ECLIC_POSTIVE_EDGE_TRIGGER, 1, 0, FFT_IRQHandler);
        //NVIC_EnableIRQ(INT_FFT);
    }
}

void fft_load_factor_by_cpu(void)
{
	uint32_t i;

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_REAL_BASE_ADDR)[i] = tw_real[i];
	}

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_IMAG_BASE_ADDR)[i] = tw_imag[i];
	}

	for (i=0; i<1024; i+=1) {
		((volatile uint32_t *)FFT_WIN_COEF_BASE_ADDR)[i] = win_coef[i];
	}
}

void fft_load_data_by_dma(void)
{
    DMA_ConfigParams dmaCfg;

    memset(&dmaCfg, 0, sizeof(dmaCfg));

    DMA_stopChannel(FFT_DMA_CHANNEL_BASE);

    dmaCfg.enableInterrupt = 1;
    dmaCfg.srcAddr = (uint32_t)a_li_lr;
    dmaCfg.destAddr = (uint32_t)FFT_DATA_BASE_ADDR;
    dmaCfg.blockTS = FFT_POINTS;
    dmaCfg.ttfc    = DMA_TT_FC_0_M2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_32;
    dmaCfg.destBtl = DMA_BTL_32;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes = DMA_TR_WIDTH_BYTE_4;

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_max + 15;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_max + 15;
    DMA_configChannel(FFT_DMA_CHANNEL_BASE, &dmaCfg);

    DMA_clearInterrupt(FFT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);
    DMA_maskInterrupt(FFT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);
    DMA_unMaskInterrupt(FFT_DMA_CHANNEL_BASE, DMA_INT_TFR);
   // NVIC_EnableIRQ(FFT_RESULT_DMA_IRQn);

    DMA_startChannel(FFT_DMA_CHANNEL_BASE);
}

void fft_dump_result_by_dma(void)
{
    DMA_ConfigParams dmaCfg;

    memset(&dmaCfg, 0, sizeof(dmaCfg));

    DMA_stopChannel(FFT_DMA_CHANNEL_BASE);

    dmaCfg.enableInterrupt = 1;
    dmaCfg.srcAddr = (uint32_t)FFT_RESULT_BASE_ADDR;
    dmaCfg.destAddr = (uint32_t)fft_result;
    dmaCfg.blockTS = FFT_POINTS*2-1;    //max 4095 -> link list
    dmaCfg.ttfc    = DMA_TT_FC_0_M2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_32;
    dmaCfg.destBtl = DMA_BTL_32;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes = DMA_TR_WIDTH_BYTE_4;

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_max + 15;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_max + 15;
    DMA_configChannel(FFT_DMA_CHANNEL_BASE, &dmaCfg);

    DMA_clearInterrupt(FFT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);
    DMA_maskInterrupt(FFT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);

    DMA_unMaskInterrupt(FFT_DMA_CHANNEL_BASE, DMA_INT_TFR);//жϴԴDMAɣж1
//    NVIC_EnableIRQ(FFT_RESULT_DMA_IRQn);

    DMA_startChannel(FFT_DMA_CHANNEL_BASE);
}

void fft_show_result(uint32_t *result, uint32_t points)
{
    log_debug("------- dump fft result start ----------\r\n");

    for (uint32_t i=0;i<points; i+=2) {
        log_debug("0x%08X:%08X %08X\n", FFT_RESULT_BASE_ADDR + i*4, result[i], result[i+1]);
    }

    log_debug("------- dump fft result end ----------\r\n");
}

void fft_dump_reg(void)
{
    log_debug("------- dump fft register start ----------\n");
    log_debug("           CFG_0x%02X: 0x%08X\n", FFT_O_FFT_CFG, HWREG(FFT_BASE + FFT_O_FFT_CFG));
    log_debug("         START_0x%02X: 0x%08X\n", FFT_O_FFT_START, HWREG(FFT_BASE + FFT_O_FFT_START));
    log_debug("DSPDMATRIG_CFG_0x%02X: 0x%08X\n", FFT_O_FFT_DSPDMATRIG_CFG, HWREG(FFT_BASE + FFT_O_FFT_DSPDMATRIG_CFG));
    log_debug("  AUTOSTART_EN_0x%02X: 0x%08X\n", FFT_O_FFT_AUTOSTART_EN, HWREG(FFT_BASE + FFT_O_FFT_AUTOSTART_EN));
    log_debug("        STATUS_0x%02X: 0x%08X\n", FFT_O_FFT_STATUS, HWREG(FFT_BASE + FFT_O_FFT_STATUS));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_FFT_INT_RAW, HWREG(FFT_BASE + FFT_O_FFT_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_FFT_INT_MASK, HWREG(FFT_BASE + FFT_O_FFT_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FLG, HWREG(FFT_BASE + FFT_O_FFT_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FRC, HWREG(FFT_BASE + FFT_O_FFT_INT_FRC));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_WIN_INT_RAW, HWREG(FFT_BASE + FFT_O_WIN_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_WIN_INT_MASK, HWREG(FFT_BASE + FFT_O_WIN_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FLG, HWREG(FFT_BASE + FFT_O_WIN_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FRC, HWREG(FFT_BASE + FFT_O_WIN_INT_FRC));
}

int main(void)
{
	int ret;

	Device_init();

	UartPrint_init(LOG_SCI_BASE, 115200);

	log_set_level(LOG_DEBUG);

	log_info("Hello DSP!\r\n");
	log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);

    DMA_initController(DMA_BASE);

    uint32_t i;
	uint32_t errCount = 0;
	uint32_t errIndex[2048];

	log_debug("------- %s start ----------\r\n", __FUNCTION__);

	memset(fft_result, 0xAA, sizeof(fft_result));

	fft_dump_reg();

	__enable_irq();
	fft_init(0);
	fft_load_factor_by_cpu();
	fft_load_data_by_dma();     //start fft automatically


	log_debug("start windowing ...\r\n");
	FFT_WIN_AUTOSTART_ENABLE(FFT_BASE);
	while(FFT_IS_WIN_BUSY(FFT_BASE));
	log_debug("windowing finished.\r\n");


	log_debug("start calculating ...\r\n");
	FFT_AUTOSTART_ENABLE(FFT_BASE);
	while(FFT_IS_BUSY(FFT_BASE));
	log_debug("fft calculate finished.\r\n");


	FFT_CLEAR_WIN_INTERRUPT(FFT_BASE);
	FFT_CLEAR_INTERRUPT_FLAG(FFT_BASE, FFT_INT_CLR_OVF_INT_CLR | FFT_INT_CLR_DONE_INT_CLR);

	fft_dump_result_by_dma();
//    while(testStep < 2);

	fft_dump_reg();

	fft_show_result(fft_result, FFT_POINTS*2);

	for (i=0; i<FFT_POINTS; i+=1) {
		if (abs(G_R_golden[i] - fft_result[i*2]) > 0x10 || abs(G_I_golden[i] - fft_result[i*2+1]) > 0x10) {
			errIndex[errCount++] = i;
		}
	}

	for(i=0;i<errCount;i++) {
		log_debug("fft result check fail, real[%d]: %08X/%08X, imag[%d]: %08X/%08X\r\n",
				errIndex[i], G_R_golden[errIndex[i]], fft_result[errIndex[i]*2], errIndex[i], G_I_golden[errIndex[i]], fft_result[errIndex[i]*2+1]);
	}
	if(errCount==0) {
		log_info("fft result check pass.\r\n");
	}
	else {
		log_info("There are %d error fft result.\r\n ", errCount);
	}

	log_debug("------- %s end ----------\r\n", __FUNCTION__);

	/* We should never get here as control is now taken by the scheduler */
	for(;;);
}

