/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    HPEC_Source_Template.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <string.h>
#include <math.h>
#include "device.h"
#include "adc_pwm.h"
#include "gs32_math.h"

#define USING_DMA_LLI   0
#define VREFHI_VAL 		1.548
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

extern uint32_t a_li_lr[2048];

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

#define ADC_DMA_BASE    DMA_CH1_BASE
#define ADC_DMA_IRQn    INT_DMA1_CH1
/* ======================================================================= */
/*                            Global Connts                                */
/* ======================================================================= */

/* None */

/* ======================================================================= */
/*                            Global Varles                                */
/* ======================================================================= */

/* None */

/* ======================================================================= */
/*                          Local FunctiPrototypes                         */
/* ======================================================================= */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

static void ADC_DMA_init(uint32_t dmaChBase);


/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
 * \brief   "add brief description of this function" 
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */

volatile uint32_t cntAdc = 0;
volatile uint32_t adcStart = 0;
volatile uint32_t adcEnd = 0;
volatile uint32_t adcTicks = 0;

extern volatile uint32_t tickStart;

#define ADC_TEST_PIN    2

void ADC_DMA_IRQHandler()
{
    tickStart = __RV_CSR_READ(CSR_MCYCLE);      //fft convert start here
    adcTicks = __RV_CSR_READ(CSR_MCYCLE) - adcStart;

	DMA_clearInterrupt(ADC_DMA_BASE, DMA_INT_TFR | DMA_INT_BLOCK);

	GPIO_togglePin(ADC_TEST_PIN);

	ADC_DMA_init(ADC_DMA_BASE);
	cntAdc += 1;
}

static void ADC_init(uint32_t adcBase, ADC_Trigger trigger)
{
    //
    // ADC Initialization: Write ADC configurations and power up the ADC
    //
    // Configures the ADC module's offset trim
    //
    ADC_setVREF(adcBase,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);

    ADC_setINLTrim(adcBase);
    //
    // Configures the analog-to-digital converter module prescaler.
    //
    ADC_setPrescaler(adcBase, ADC_CLK_DIV_3_0);
    //
    // Sets the timing of the end-of-conversion pulse
    //
    ADC_setInterruptPulseMode(adcBase, ADC_PULSE_END_OF_CONV);
    //
    // Powers up the analog-to-digital converter core.
    //
    ADC_enableConverter(adcBase);
    //
    // Delay for 1ms to allow ADC time to power up
    //
    DEVICE_DELAY_US(5000);

    //
    // SOC Configuration: Setup ADC EPWM channel and trigger settings
    //
    // Disables SOC burst mode.
    //
    ADC_disableBurstMode(adcBase);
    //
    // Sets the priority mode of the SOCs.
    //
    ADC_setSOCPriority(adcBase, ADC_PRI_ALL_ROUND_ROBIN);

    //
    // Configures Repeater Modules
    //
    //
    //      Repeater module 1 parameters
    //      base:           myADC0_BASE
    //      RepInstance:    ADC_TRIGGER_REPEATER1
    //      ADC_RepeaterConfig : {
    //      mode            : ADC_REPMODE_OVERSAMPLING
    //      Trigger         : ADC_TRIGGER_EPWM1_SOCA
    //      SyncInput       : ADC_SYNCIN_DISABLE
    //      Count           : 3
    //      phase           : 0
    //      spread          : 0
    //
    ADC_triggerRepeaterSelect(adcBase, ADC_REPINST1, trigger);
    ADC_triggerRepeaterMode(adcBase, ADC_REPINST1, ADC_REPMODE_OVERSAMPLING);
    ADC_triggerRepeaterSyncIn(adcBase, ADC_REPINST1, ADC_SYNCIN_DISABLE);
    ADC_triggerRepeaterCount(adcBase, ADC_REPINST1, 2048);
    ADC_triggerRepeaterPhase(adcBase, ADC_REPINST1, 0);
    ADC_triggerRepeaterSpread(adcBase, ADC_REPINST1, 0);

    //
    // Start of Conversion 0 Configuration
    //
    //
    // Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
    //           SOC number        : 0
    //          Trigger            : ADC_TRIGGER_EPWM2_SOCA
    //          Channel            : ADC_CH_ADCIN3
    //         Sample Window    : 9 SYSCLK cycles
    //        Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
    //
    ADC_setupSOC(adcBase, ADC_SOC_NUMBER0, ADC_TRIGGER_REPEATER1, ADC_CH_ADCIN0, 9U);
    ADC_setInterruptSOCTrigger(adcBase, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);

    //
    // ADC Interrupt 1 Configuration
    //         SOC/EOC number    : 0
    //         Interrupt Source: enabled
    //        Continuous Mode    : enabled
    //
    ADC_setInterruptSource(adcBase, ADC_INT_NUMBER1, ADC_SOC_NUMBER0);
    ADC_enableInterrupt(adcBase, ADC_INT_NUMBER1);
    ADC_clearInterruptStatus(adcBase, ADC_INT_NUMBER1);
    ADC_enableContinuousMode(adcBase, ADC_INT_NUMBER1);

    ADC_setupPPB(adcBase+0x400, ADC_PPB_NUMBER2, ADC_SOC_NUMBER0);

    ADC_enableMeanGroup(adcBase+0x400,ADC_SOC_NUMBER0, ADC_MEAN_GROUP0);

    ADC_setupPPB(adcBase+0x400, ADC_PPB_NUMBER1, ADC_SOC_NUMBER0);

    ADC_setEPPBScale(adcBase+0x400,ADC_SOC_NUMBER0, 1.0f);
    ADC_setEPPBOffset(adcBase+0x400,ADC_SOC_NUMBER0, 0.0f);

    ADC_clearInterruptStatus(adcBase, ADC_INT_NUMBER1);
    //ADC_setInterruptSOCTrigger(adcBase, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_ADCINT1);
}

static void ADC_DMA_init(uint32_t dmaChBase)
{
    DMA_ConfigParams dmaCfg;

    DMA_stopChannel(dmaChBase);

    memset(&dmaCfg, 0, sizeof(dmaCfg));

    dmaCfg.enableInterrupt = 1;

    dmaCfg.srcAddr = (uint32_t)(ADCARESULT_BASE + 0x100 + ADC_E_RESULT0);
    dmaCfg.destAddr = (uint32_t)a_li_lr;
    dmaCfg.blockTS = sizeof(a_li_lr)/sizeof(a_li_lr[0]);
    dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_1;
    dmaCfg.destBtl = DMA_BTL_1;
    dmaCfg.srcAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_4;
    //dmaCfg.reloadDst = true;

    dmaCfg.chPriority = DMA_CH_PRIORITY_7;

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_Trig_ADCA_1int;

    DMA_configChannel(dmaChBase, &dmaCfg);
    DMA_clearInterrupt(dmaChBase, DMA_INT_TFR);
    DMA_unMaskInterrupt(dmaChBase, DMA_INT_TFR);

    Interrupt_register(ADC_DMA_IRQn, ADC_DMA_IRQHandler);
    Interrupt_enable(ADC_DMA_IRQn);

	DMA_startChannel(dmaChBase);
}

uint16_t g_dac_sin_array[128];

void DAC_CreateSinArray(double maxVoltage, uint16_t samples)		// 
{
    int maxValue = maxVoltage * 4095 / VREFHI_VAL;
    uint16_t outputData = 0;                                          // ż
    double increment = (2*3.141592653589793) / samples;                 // x

    for (uint8_t i = 0; i < samples; i++)
    {
        // Ҳʽy=Asin(x+)+b
        outputData = (maxValue * __sin(i * increment)+4095)/2;
        outputData = (outputData > 4095) ? 4095 : outputData;
        g_dac_sin_array[i] = outputData;
    }
}

void myDAC0_init(void)
{
	//
	// Set DAC reference voltage.
	//
	//DAC_setReferenceVoltage(DACA_BASE, DAC_REF_VDAC);
	DAC_setReferenceVoltage(DACA_BASE, DAC_REF_INTERNAL);
	//
	// Set DAC load mode.
	//
	DAC_setLoadMode(DACA_BASE, DAC_LOAD_SYSCLK);
	DAC_setGainMode(DACA_BASE, DAC_GAIN_ONE);
	//DAC_setGainMode(DACA_BASE, DAC_GAIN_TWO);
	//
	// Enable the DAC output
	//
	DAC_enableOutput(DACA_BASE);
	DAC_enable(DACA_BASE);
	//
	// Set the DAC shadow output
	//
	DAC_setShadowValue(DACA_BASE, 4000U);
	//
	// Delay for buffered DAC to power up.
	//
	DEVICE_DELAY_US(500);
}

void EPWM_ADC_init(void)
{
    GPIO_enableWritePin(ADC_TEST_PIN);

    myDAC0_init();

    ADC_DMA_init(ADC_DMA_BASE);

    ADC_init(ADCA_BASE, ADC_TRIGGER_CPU1_TINT3);        //timer3 10ms
}



#ifdef __cplusplus
}
#endif

