#include <stdint.h>
#include <string.h>

#include "log.h"
#include "device.h"
#include "fft.h"
#include "dma.h"

#ifdef __DEBUG
#undef __DEBUG
#endif

#define __DEBUG		1

#if __DEBUG == 1
#define debug(...)	printf(__VA_ARGS__)
#else
#define debug(...)
#endif

#define FFT_TW_REAL_BASE_ADDR    (0x4008C400)
#define FFT_TW_IMAG_BASE_ADDR    (0x4008CC00)
#define FFT_WIN_COEF_BASE_ADDR   (0x4008D400)

#define FFT_DATA_BASE_ADDR       (0x40088400)
#define FFT_RESULT_BASE_ADDR     (0x40088400)

#define FFT_POINTS               2048

extern const uint32_t tw_imag[512];
extern const uint32_t tw_real[512];
extern const uint32_t win_coef[1024];

#define FFT_RESULT_DMA_CHANNEL_BASE      DMA_CH2_BASE
#define FFT_RESULT_DMA_IRQn              INT_DMA1_CH2

void fft_dump_result_by_dma(void);

__attribute__((aligned(32))) uint32_t fft_result[FFT_POINTS*2];

typedef enum {
    FFT_LOAD_START = 0,
    FFT_LOAD_END = 1,
    FFT_CALC_END = 2,
    FFT_DUMP_END = 3,
} fftStep_t;

volatile fftStep_t testStep;
volatile uint32_t tickStart = 0;
volatile uint32_t tickFFTEnd = 0;

volatile uint32_t tickDumpStart = 0;
volatile uint32_t tickDumpEnd = 0;

volatile uint32_t tickFFT = 0;
volatile uint32_t tickDump = 0;


volatile uint32_t cntFFT = 0;

#define GetCurrTick()   __RV_CSR_READ(CSR_MCYCLE)

#define FFT_TEST_GPIO   3

void FFT_IRQHandler(void)
{
    tickFFTEnd = GetCurrTick();
    tickFFT = tickFFTEnd - tickStart;

    FFT_CLEAR_WIN_INTERRUPT(FFT_BASE);
    FFT_CLEAR_INTERRUPT_FLAG(FFT_BASE, FFT_INT_CLR_OVF_INT_CLR | FFT_INT_CLR_DONE_INT_CLR);

    fft_dump_result_by_dma();

    testStep = FFT_CALC_END;

    cntFFT += 1;

    GPIO_togglePin(FFT_TEST_GPIO);
}

void FFT_RESULT_DMA_IRQHandler(void)
{
    //dumping fft result finished
    fft_result[4095] = ((volatile uint32_t*)FFT_RESULT_BASE_ADDR)[4095];
    tickDumpEnd = GetCurrTick();
    tickDump = tickDumpEnd - tickDumpStart;

    DMA_clearInterrupt(FFT_RESULT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);

    testStep = FFT_DUMP_END;
    MInvalDCache();
    GPIO_togglePin(FFT_TEST_GPIO);
}

void fft_init(int trigByDMA)
{
    HWREG(SYSCTL_BASE + 0x404) = 1;

    FFT_SET_TYPE(FFT_BASE, FFT_TYPE_REAL);
    FFT_SET_LEN(FFT_BASE, REAL_FFT_LEN_2048);

    if (trigByDMA) {
        FFT_WIN_AUTOSTART_ENABLE(FFT_BASE);
        FFT_AUTOSTART_ENABLE(FFT_BASE);

        FFT_WIN_INTERRUPT_ENABLE(FFT_BASE);

        FFT_DSPDMA_TRIG_ENABLE(FFT_BASE, FFT_DMA_TRIG_CHAN1);

        //enable fft irq
        FFT_CLEAR_WIN_INTERRUPT(FFT_BASE);
        FFT_CLEAR_INTERRUPT_FLAG(FFT_BASE, FFT_INT_CLR_OVF_INT_CLR | FFT_INT_CLR_DONE_INT_CLR);

        FFT_INTERRUPT_ENABLE(FFT_BASE, FFT_DONE_INT_MASK);

        Interrupt_register(INT_FFT, FFT_IRQHandler);
        Interrupt_enable(INT_FFT);
    }
}

void fft_load_factor_by_cpu(void)
{
	uint32_t i;

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_REAL_BASE_ADDR)[i] = tw_real[i];
	}

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_IMAG_BASE_ADDR)[i] = tw_imag[i];
	}

	for (i=0; i<1024; i+=1) {
		((volatile uint32_t *)FFT_WIN_COEF_BASE_ADDR)[i] = win_coef[i];
	}
}

void fft_dump_result_by_dma(void)
{
    DMA_ConfigParams dmaCfg;

    memset(&dmaCfg, 0, sizeof(dmaCfg));

    DMA_stopChannel(FFT_RESULT_DMA_CHANNEL_BASE);

    dmaCfg.enableInterrupt = 1;
    dmaCfg.srcAddr = (uint32_t)FFT_RESULT_BASE_ADDR;
    dmaCfg.destAddr = (uint32_t)fft_result;
    dmaCfg.blockTS = FFT_POINTS*2-1;    //max 4095 -> link list
    dmaCfg.ttfc    = DMA_TT_FC_0_M2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_4;
    dmaCfg.destBtl = DMA_BTL_4;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes = DMA_TR_WIDTH_BYTE_4;

    dmaCfg.chPriority = DMA_CH_PRIORITY_0;

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_max + 15;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_max + 15;
    DMA_configChannel(FFT_RESULT_DMA_CHANNEL_BASE, &dmaCfg);

    DMA_clearInterrupt(FFT_RESULT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);
    DMA_maskInterrupt(FFT_RESULT_DMA_CHANNEL_BASE, DMA_INT_ERR | DMA_INT_DSTTRAN | DMA_INT_SRCTRAN | DMA_INT_BLOCK | DMA_INT_TFR);

    DMA_unMaskInterrupt(FFT_RESULT_DMA_CHANNEL_BASE, DMA_INT_TFR);

    DMA_startChannel(FFT_RESULT_DMA_CHANNEL_BASE);

    tickDumpStart = GetCurrTick();
}

void fft_dump_reg(void)
{
    log_debug("------- dump fft register start ----------\n");
    log_debug("           CFG_0x%02X: 0x%08X\n", FFT_O_FFT_CFG, HWREG(FFT_BASE + FFT_O_FFT_CFG));
    log_debug("         START_0x%02X: 0x%08X\n", FFT_O_FFT_START, HWREG(FFT_BASE + FFT_O_FFT_START));
    log_debug("DSPDMATRIG_CFG_0x%02X: 0x%08X\n", FFT_O_FFT_DSPDMATRIG_CFG, HWREG(FFT_BASE + FFT_O_FFT_DSPDMATRIG_CFG));
    log_debug("  AUTOSTART_EN_0x%02X: 0x%08X\n", FFT_O_FFT_AUTOSTART_EN, HWREG(FFT_BASE + FFT_O_FFT_AUTOSTART_EN));
    log_debug("        STATUS_0x%02X: 0x%08X\n", FFT_O_FFT_STATUS, HWREG(FFT_BASE + FFT_O_FFT_STATUS));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_FFT_INT_RAW, HWREG(FFT_BASE + FFT_O_FFT_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_FFT_INT_MASK, HWREG(FFT_BASE + FFT_O_FFT_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FLG, HWREG(FFT_BASE + FFT_O_FFT_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FRC, HWREG(FFT_BASE + FFT_O_FFT_INT_FRC));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_WIN_INT_RAW, HWREG(FFT_BASE + FFT_O_WIN_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_WIN_INT_MASK, HWREG(FFT_BASE + FFT_O_WIN_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FLG, HWREG(FFT_BASE + FFT_O_WIN_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FRC, HWREG(FFT_BASE + FFT_O_WIN_INT_FRC));
}

void FFT_Init(void)
{
    uint32_t i;

    GPIO_enableWritePin(FFT_TEST_GPIO);

    fft_dump_reg();

    fft_init(1);

    fft_dump_reg();

    fft_load_factor_by_cpu();

    Interrupt_register(FFT_RESULT_DMA_IRQn, FFT_RESULT_DMA_IRQHandler);
    Interrupt_enable(FFT_RESULT_DMA_IRQn);

    MFlushDCache();

    testStep = FFT_DUMP_END;
}


