/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


#ifdef __cplusplus
extern "C"{
#endif



#include "device.h"
#include "printf.h"
#include "eeprom_polling.h"
#include "stdio.h"


uint32_t eeprom_addr_len = 0;
uint32_t eeprom_status = 0;

uint32_t g_i2c_base = I2CB_BASE;

uint8_t wdata[W_DATA_LEN] = {0};
uint8_t rdata[W_DATA_LEN] = {0};

uint32_t i2c_port_num = 1;


uint16_t eeprom_DataAddrWrite(uint32_t base, uint32_t data_addr)
{
	uint8_t addr[2] = {0};

	if (eeprom_addr_len == 2)
	{
		addr[0] = (uint8_t)((data_addr >> 8) & 0xff);
		addr[1] = (uint8_t)(data_addr & 0xff);
	}
	else if (eeprom_addr_len == 1)
	{
		addr[0] = (uint8_t)(data_addr & 0xff);
	}
	else
	{
		return -1;
	}

	for(uint32_t i = 0; i < eeprom_addr_len; i++)
	{
		I2C_pollTxEmpty(base);
		I2C_putData(base, addr[i]);
	}
	return 0;
}


uint16_t eeprom_WriteData(uint32_t base, uint32_t data_addr, uint8_t *data_cmd, uint32_t data_cmd_len, I2C_NEXT_CONDITION next_cond)
{
    int32_t err = 0;
    //send addr
	eeprom_DataAddrWrite(base,data_addr);


	//send data
	if (I2C_getIcMode(base) == I2C_ICMODE_MASTER)
	{
	   for (uint32_t i = 0; i < data_cmd_len; i++)
	   {
		   I2C_pollTxEmpty(base);

		   if ((next_cond & I2C_NEXT_CONDITION_RESTART) && (i == 0))
		   {
			   I2C_putData(base, data_cmd[i]);
		   }
		   else if ((next_cond & I2C_NEXT_CONDITION_STOP) && (i == (data_cmd_len - 1)))
		   {
			   I2C_sendCmdWriteAndStop(base, data_cmd[i]);
		   }
		   else
		   {
			   I2C_putData(base, data_cmd[i]);
		   }
	   }
	}
	else
	{
	   ASSERT(0);
	}
	return err;

}

uint16_t eeprom_ReadData(uint32_t base, uint32_t data_addr, uint8_t *data_cmd, uint32_t data_cmd_len, I2C_NEXT_CONDITION next_cond)
{
	int32_t err = 0;
	uint32_t retry_cnt = 0;
	//send addr
	eeprom_DataAddrWrite(base,data_addr);

	//recv data
	if (I2C_getIcMode(base) == I2C_ICMODE_MASTER)
	{
		for (uint32_t i = 0; i < data_cmd_len; i++)
		{
			I2C_pollTxEmpty(base);

			if ((next_cond & I2C_NEXT_CONDITION_RESTART) && (i == 0))
			{
				I2C_sendCmdRestartAndRead(base);
			}
			else if ((next_cond & I2C_NEXT_CONDITION_STOP) && (i == (data_cmd_len - 1)))
			{
				I2C_sendCmdReadAndStop(base);
			}
			else
			{
				I2C_sendCmdRead(base);
			}

			I2C_pollRxFull(base);

			*data_cmd = I2C_getData(base);
			data_cmd++;
		}
	}
	else
	{
		ASSERT(0);
	}
	return 0;

}


uint16_t checkBusStatus(uint32_t base)
{

    if(I2C_isBusBusy(base))
    {
        return ERROR_BUS_BUSY;
    }

    if(I2C_getStopConditionStatus(base))
    {
        return ERROR_STOP_NOT_READY;
    }

    return I2C_LINK_SUCCESS;
}


uint16_t handleNACK(uint32_t base)
{
    if(I2C_getInterruptRawStatus(base) & I2C_INT_NO_ACK)
    {
    	I2C_clearInterruptTxAbrt(base);

        return ERROR_NACK_RECEIVED;
    }

    return I2C_LINK_SUCCESS;
}


void eeprom_init(uint32_t addr_len)
{
	eeprom_addr_len = addr_len;
}

void i2c_err_irq_check(uint32_t irq_status)
{
    if (irq_status & I2C_INT_TX_ABRT)
    {
        I2C_clearInterruptTxAbrt(g_i2c_base);
    }

    if (irq_status & I2C_INT_TX_OVER)
    {
        I2C_clearInterruptTxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_OVER)
    {
        I2C_clearInterruptRxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_UNDER)
    {
        I2C_clearInterruptRxUnder(g_i2c_base);
    }
}

void I2CX_INTR_IRQHandler(void)
{
    uint32_t irq_status = I2C_getInterruptStatus(g_i2c_base);

    i2c_err_irq_check(irq_status);

    I2C_clearAllInterruptStatus(g_i2c_base);
}

void i2c_eeprom_operate(uint32_t i2c_base)
{
	uint32_t i = 0;

    for (i = 0; i < W_DATA_LEN; i++)
    {
        wdata[i] = i+1;
    }

    //start to send data
    for (i = 0; i < W_DATA_LEN / EEPROM_PAGE_LEN; i++)//Only one page of EEPROM_PAGE_LEN byte data can be written at a time.
    {
    	eeprom_WriteData(i2c_base, W_ADDR+EEPROM_PAGE_LEN*i, wdata+EEPROM_PAGE_LEN*i, EEPROM_PAGE_LEN, I2C_NEXT_CONDITION_STOP);
        DEVICE_DELAY_US(3000); //delay 3ms
    }

    if (W_DATA_LEN % EEPROM_PAGE_LEN != 0)
    {
        uint32_t remain_data_len = W_DATA_LEN % EEPROM_PAGE_LEN;
        eeprom_WriteData(i2c_base, W_ADDR+EEPROM_PAGE_LEN*(W_DATA_LEN/EEPROM_PAGE_LEN), wdata+EEPROM_PAGE_LEN*(W_DATA_LEN/EEPROM_PAGE_LEN), remain_data_len, I2C_NEXT_CONDITION_STOP);

        DEVICE_DELAY_US(3000); //delay 3ms
    }

    //start to read data
    eeprom_ReadData(i2c_base, W_ADDR, rdata, W_DATA_LEN, I2C_NEXT_CONDITION_STOP);
}

uint16_t verifyEEPROMRead(uint32_t i2c_base)
{
    uint16_t i;
    uint16_t status = 0;

	for (i = 0; i < W_DATA_LEN; i++)
	{
		if (wdata[i] != rdata[i])
		{
			return 0;
		}
	}
	return status;
}


 void I2C_GPIO_init(void)
 {

	 GPIO_setPinConfig(GPIO_2_I2CB_SDA);
	 GPIO_setPadConfig(2, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	 GPIO_setQualificationMode(2, GPIO_QUAL_ASYNC);

	 GPIO_setPinConfig(GPIO_3_I2CB_SCL);
	 GPIO_setPadConfig(3, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	 GPIO_setQualificationMode(3, GPIO_QUAL_ASYNC);


 }

 void I2Cinit(uint32_t i2c_base)
 {
#if(IS_GS32F3xx(0x22))
	uint32_t i2c_port_irqn = INT_I2CA;
#else
	uint32_t i2c_port_irqn = INT_I2C0;
#endif

	 g_i2c_base = i2c_base;

	 if (i2c_base == I2CA_BASE)
	 {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CA;
#else
    	i2c_port_irqn = INT_I2C0;
#endif
		i2c_port_num = 0;
	 }
	 else if (i2c_base == I2CB_BASE)
	 {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CB;
#else
    	i2c_port_irqn = INT_I2C1;
#endif
		i2c_port_num = 1;
	 }
	 else
	 {
		return;
	 }

	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setTargetAddress(i2c_base,EEPROM_SLAVE_ADDRESS);
	I2C_setConfig(i2c_base, I2C_MASTER_MODE | I2C_TX_EMPTY_CTRL | I2C_RESTART_EN);//
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);

	//Enable error interrupt
	Interrupt_register(i2c_port_irqn, I2CX_INTR_IRQHandler);
	I2C_enableInterrupt(i2c_base, I2C_INT_RX_OVER | I2C_INT_RX_UNDER | I2C_INT_TX_ABRT | I2C_INT_TX_OVER);
	Interrupt_enable(i2c_port_irqn);
 }

#ifdef __cplusplus
}
#endif

