/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "eeprom_polling_register.h"
#include <stdio.h>

uint32_t g_i2c_base = I2CB_BASE;

Uint16 wdata[W_DATA_LEN] = {0};
Uint16 rdata[W_DATA_LEN] = {0};

uint32_t i2c_port_num = 1;

void i2c_err_irq_check(uint32_t irq_status)
{
    if (irq_status & I2C_INT_TX_ABRT)
    {
        I2C_clearInterruptTxAbrt(g_i2c_base);
    }

    if (irq_status & I2C_INT_TX_OVER)
    {
        I2C_clearInterruptTxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_OVER)
    {
        I2C_clearInterruptRxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_UNDER)
    {
        I2C_clearInterruptRxUnder(g_i2c_base);
    }
}

void I2CX_INTR_IRQHandler(void)
{
    uint32_t irq_status = I2C_getInterruptStatus(g_i2c_base);

    i2c_err_irq_check(irq_status);

    I2C_clearAllInterruptStatus(g_i2c_base);
}

void i2c_eeprom_test(void)
{
	uint32_t i = 0;
    for (i = 0; i < W_DATA_LEN; i++)
    {
        wdata[i] = i;
    }

    //start to send data
    for (i = 0; i < W_DATA_LEN / EEPROM_PAGE_LEN; i++)	//Only one page of EEPROM_PAGE_LEN byte data can be written at a time.
    {
    	WriteData(wdata+EEPROM_PAGE_LEN*i, W_ADDR+EEPROM_PAGE_LEN*i, EEPROM_PAGE_LEN);
        DEVICE_DELAY_US(7000); //delay 7ms
    }

    if (W_DATA_LEN % EEPROM_PAGE_LEN != 0)
    {
        uint32_t remain_data_len = W_DATA_LEN % EEPROM_PAGE_LEN;
        WriteData(wdata+EEPROM_PAGE_LEN*(W_DATA_LEN/EEPROM_PAGE_LEN), W_ADDR+EEPROM_PAGE_LEN*(W_DATA_LEN/EEPROM_PAGE_LEN), remain_data_len);
        DEVICE_DELAY_US(7000);  //delay 7ms
    }

    //start to read data
	ReadData(rdata, W_ADDR,W_DATA_LEN);
	verifyEEPROMRead(g_i2c_base);
}





uint16_t verifyEEPROMRead(uint32_t i2c_base)
{
    uint16_t i;
    uint16_t status = 0;


	for (i = 0; i < W_DATA_LEN; i++)
	{
		if (wdata[i] != rdata[i])
		{
			return 0;
		}
	}
	return status;
}


 void I2C_GPIO_init(void)
 {

	 GPIO_setPinConfig(GPIO_2_I2CB_SDA);
	 GPIO_setPadConfig(2, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	 GPIO_setQualificationMode(2, GPIO_QUAL_ASYNC);

	 GPIO_setPinConfig(GPIO_3_I2CB_SCL);
	 GPIO_setPadConfig(3, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	 GPIO_setQualificationMode(3, GPIO_QUAL_ASYNC);
 }



 void I2Cinit(uint32_t i2c_base)
 {
	 I2C_INIT_PARAM init_param = {0};
#if(IS_GS32F3xx(0x22))
	uint32_t i2c_port_irqn = INT_I2CA;
#else
	uint32_t i2c_port_irqn = INT_I2C0;
#endif

	 g_i2c_base = i2c_base;

	 if (i2c_base == I2CA_BASE)
	 {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CA;
#else
    	i2c_port_irqn = INT_I2C0;
#endif
		i2c_port_num = 0;
	 }
	 else if (i2c_base == I2CB_BASE)
	 {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CB;
#else
    	i2c_port_irqn = INT_I2C1;
#endif
		i2c_port_num = 1;
	 }
	 else if (i2c_base == PMBUSA_BASE)
	 {
		i2c_port_irqn = INT_PMBUSA;
		i2c_port_num = 2;
	 }
	 else
	 {
		return;
	 }

	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setTargetAddress(i2c_base,EEPROM_SLAVE_ADDRESS);
	I2C_setConfig(i2c_base, I2C_MASTER_MODE | I2C_TX_EMPTY_CTRL | I2C_RESTART_EN);//
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);

	//Enable error interrupt
	Interrupt_register(i2c_port_irqn, I2CX_INTR_IRQHandler);
	I2C_enableInterrupt(i2c_base, I2C_IC_INTR_STAT_TX_EMPTY | I2C_INT_RX_OVER | I2C_INT_RX_UNDER | I2C_INT_TX_ABRT | I2C_INT_TX_OVER);
	Interrupt_enable(i2c_port_irqn);
 }










 /*
  *Get I2c Transmit Status
  *Returns 0 --> the current data is being sent to the shift register and is not writable
  *Returns 1 --> current data has been sent to the shift register and can be written to
  */
 bool I2cGetTxReady(uint32_t i2c_base)
 {
	 if((I2C_getInterruptStatus(i2c_base) & I2C_IC_INTR_STAT_TX_EMPTY) == 0){
		 return false;
	 }else{
		 return true;
	 }
 }


/*
 *Queries whether the bus is busy
 *Return 0 --> the current bus is busy and is transmitting data
 *Return 1 --> current bus idle
 * */
 bool I2cGetBusStatus(uint32_t i2c_base)
 {
	 if(I2C_isBusBusy(i2c_base))
	 {
		 return false;
	 }else{
		 return true;
	 }
 }


 /*
  * eeprom data write
  * Wdata: pointer to write data cache
  * RomAddress: eeprom address
  * number: length of data written
  * return I2C_SUCCESS --- write data successfully.
  *
  **/
 Uint16 WriteData(Uint16 *Wdata, Uint16 RomAddress, Uint16 number)
 {
	 Uint16 i;
	 uint8_t addr[2] = {0};
	 uint8_t eeprom_addr_len = 0;
	 eeprom_addr_len = EEPROM_DATA_ADDRESS_LEN;
	 if (eeprom_addr_len == 2)
	 {
		 addr[0] = (uint8_t)((RomAddress >> 8) & 0xff);
		 addr[1] = (uint8_t)(RomAddress & 0xff);
	 }
	 else if (eeprom_addr_len == 1)
	 {
		 addr[0] = (uint8_t)(RomAddress & 0xff);
	 }

     //If the number of data is 0, return directly
     if(number == 0)
     	return I2C_NO_FLAGS;


     if(!I2cGetBusStatus(g_i2c_base))
     {
        return I2C_BUS_BUSY_ERROR;
     }

	while(I2C_isMasterBusy(g_i2c_base)) ;
    //Send EEPROM address
    for(i = 0; i < eeprom_addr_len; i++)
	{

		I2C_putData(g_i2c_base, addr[i]);

		while(!I2cGetTxReady(g_i2c_base));
		//check ack
		if(I2C_getInterruptStatus(g_i2c_base) & I2C_IC_INTR_STAT_TX_ABRT)
		{
			I2C_clearInterruptTxAbrt(g_i2c_base);
			return    I2C_NACK_ERROR;
		}
	}

     //send data
     for (i=0; i<number; i++)
     {
 	   if(i<(number-1))
 		   I2C_putData(g_i2c_base, (uint32_t)(*Wdata));
 	   else 	        //general stop bit
 	   {
 		   I2C_sendCmdWriteAndStop(g_i2c_base, (*Wdata));
 	   }
 	   Wdata++;

 	   //wait to send finish
 	  while(!I2cGetTxReady(g_i2c_base));

 	   //check ack
 	   if(I2C_getInterruptStatus(g_i2c_base) & I2C_IC_INTR_STAT_TX_ABRT)
 	   {
 		   I2C_clearInterruptTxAbrt(g_i2c_base);
 	   	   return    I2C_NACK_ERROR;
 	   }
     }
     return I2C_SUCCESS;
 }


 /*
  * eeprom data readout
  * Wdata: pointer to readout data cache
  * RomAddress: eeprom address
  * number: length of data read out
  * return I2C_SUCCESS --- read data successfully.
  */
 Uint16 ReadData(Uint16 *RamAddr, Uint16 RomAddress, Uint16 number)
 {
     Uint16  i=0,Temp=0;

     uint8_t addr[2] = {0};
	 uint8_t eeprom_addr_len = 0;
	 eeprom_addr_len = EEPROM_DATA_ADDRESS_LEN;
	 if (eeprom_addr_len == 2)
	 {
		 addr[0] = (uint8_t)((RomAddress >> 8) & 0xff);
		 addr[1] = (uint8_t)(RomAddress & 0xff);
	 }
	 else if (eeprom_addr_len == 1)
	 {
		 addr[0] = (uint8_t)(RomAddress & 0xff);
	 }

     //If the number of data is 0, return directly
     if(number == 0)
     	return I2C_NO_FLAGS;


     if(!I2cGetBusStatus(g_i2c_base))
     {
        return I2C_BUS_BUSY_ERROR;
     }

 	while(I2C_isMasterBusy(g_i2c_base)) ;

 	//Send EEPROM address
	for(i = 0; i < eeprom_addr_len; i++)
	{

		I2C_putData(g_i2c_base, addr[i]);

		while(!I2cGetTxReady(g_i2c_base));

		if(I2C_getInterruptStatus(g_i2c_base) & I2C_IC_INTR_STAT_TX_ABRT)
		{
			I2C_clearInterruptTxAbrt(g_i2c_base);
			return    I2C_NACK_ERROR;
		}
	}

    //start to read eeprom data
 	for(i=0;i<number;i++)
 	{
 		if(i==0)  //The first read command needs to be combined with Restart
 		{
 			if(number == 1)  //Only 1 Byte read, need to combine RESTART and STOP together
 			{
 				I2C_sendCmdRestartAndReadAndStop(g_i2c_base);
 			}
 			else  //send data with restart singal
 			{
 				I2C_sendCmdRestartAndRead(g_i2c_base);
 			}
 		}
 		else if(i==(number-1)) //The last read command needs to be combined with STOP
 			I2C_sendCmdReadAndStop(g_i2c_base);
 		else
 			I2C_sendCmdRead(g_i2c_base);

 		//Waiting for data to be received
 		if(!I2C_pollRxStatus(g_i2c_base, 6000))
 			return I2C_NACK_ERROR;

 		*RamAddr=I2C_getData(g_i2c_base);
 		RamAddr++;
 	}
 	return I2C_SUCCESS;
 }



