/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef __cplusplus
extern "C"{
#endif


#include "device.h"
#include "eeprom_state.h"

struct I2CMsg i2cMsgOut = { MSG_STATUS_SEND_WITHSTOP,
						    TARGET_ADDRESS,
                            NUM_BYTES,
                            EEPROM_HIGH_ADDR,
                            EEPROM_LOW_ADDR,
                            0x00,
                            0x01,
                            0x02,
                            0x03,
                            0x04,
                            0x05,
                            0x06,
						    0x07,
						    0x08};


struct I2CMsg i2cMsgIn  = {MSG_STATUS_SEND_NOSTOP,
                           TARGET_ADDRESS,
                           NUM_BYTES,
                           EEPROM_HIGH_ADDR,
                           EEPROM_LOW_ADDR};

struct I2CMsg *currentMsgPtr;

uint16_t passCount = 0;
uint16_t failCount = 0;
uint32_t i2c_port_num = 1;

void I2C_GPIO_init(void)
 {
	 GPIO_setPinConfig(GPIO_2_I2CB_SDA);
	 GPIO_setPadConfig(2, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	 GPIO_setQualificationMode(2, GPIO_QUAL_ASYNC);

	 GPIO_setPinConfig(GPIO_3_I2CB_SCL);
	 GPIO_setPadConfig(3, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	 GPIO_setQualificationMode(3, GPIO_QUAL_ASYNC);
 }

 void I2Cinit(uint32_t i2c_base)
 {

#if(IS_GS32F3xx(0x22))
	uint32_t i2c_port_irqn = INT_I2CA;
#else
	uint32_t i2c_port_irqn = INT_I2C0;
#endif

	 if (i2c_base == I2CA_BASE)
	 {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CA;
#else
    	i2c_port_irqn = INT_I2C0;
#endif
		i2c_port_num = 0;
	 }
	 else if (i2c_base == I2CB_BASE)
	 {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CB;
#else
    	i2c_port_irqn = INT_I2C1;
#endif
		i2c_port_num = 1;
	 }
	 else
	 {
		printf("i2c base error 0x%x \r\n", i2c_base);
		return;
	 }

	//Configuring IIC Parameters
	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setTargetAddress(i2c_base,TARGET_ADDRESS);
	I2C_setConfig(i2c_base, I2C_MASTER_MODE | I2C_TX_EMPTY_CTRL | I2C_RESTART_EN);//
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);
	I2C_disableFIFO(i2c_base);
	//enable error interrupt
	I2C_enableInterrupt(I2C_BASE, I2C_INT_STOP_DET | I2C_INT_NO_ACK);
	Interrupt_enable(i2c_port_irqn);
	//__enable_irq();
 }






 //
 // Function to send the data that is to be written to the EEPROM
 //
 uint16_t writeData(struct I2CMsg *msg)
 {
 	volatile uint16_t i;
 	volatile uint16_t len = 0;
 	len = msg->numBytes;
 	//
 	// Wait until the STP bit is cleared from any previous controller
 	// communication. Clearing of this bit by the module is delayed until after
 	// the SCD bit is set. If this bit is not checked prior to initiating a new
 	// message, the I2C could get confused.
 	//

 	if (!(I2C_getInterruptRawStatus(I2C_BASE)&I2C_TX_EMPT_ISRC)) //STP==1, there is no STOP condition set after data transmitted, STOP_DET is not set
	{
		return ERROR_STOP_NOT_READY;
	}

 	//
 	// Setup target address
 	//
 	I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);

 	//
 	// Check if bus busy
 	//
 	if ((I2C_getIcStatus(I2C_BASE)&0x1) && (msg->msgStatus != MSG_STATUS_RESTART))
	{
		return ERROR_BUS_BUSY;
	}
 	//
 	// Setup data to send
 	//
//	I2C_pollTxEmpty(I2C_BASE);
// 	I2C_putData(I2C_BASE, msg->memoryHighAddr);
 	I2C_pollTxEmpty(I2C_BASE);
 	I2C_putData(I2C_BASE, msg->memoryLowAddr);

 	for (i = 0; i < len; i++)
 	{
 		I2C_pollTxEmpty(I2C_BASE);
 		if(i == (len - 1))
	    {
 			I2C_sendCmdWriteAndStop(I2C_BASE, msg->msgBuffer[i]);
	    }else{
	    	I2C_putData(I2C_BASE, msg->msgBuffer[i]);
	    }

 	}
 		len = 0;

 	return(SUCCESS);
 }

 //
 // Function to prepare for the data that is to be read from the EEPROM
 //
 uint16_t readData(struct I2CMsg *msg)
 {

 	//
 	// Wait until the STP bit is cleared from any previous controller
 	// communication. Clearing of this bit by the module is delayed until after
 	// the SCD bit is set. If this bit is not checked prior to initiating a new
 	// message, the I2C could get confused.
 	//

	if (!(I2C_getInterruptRawStatus(I2C_BASE)&I2C_TX_EMPT_ISRC)) //STP==1, there is no STOP condition set after data transmitted, STOP_DET is not set
	{
		return ERROR_STOP_NOT_READY;
	}


 	//
 	// Setup target address
 	//
 	I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);

 	//
 	// If we are in the the address setup phase, send the address without a
 	// stop condition.
 	//
 	if(msg->msgStatus == MSG_STATUS_SEND_NOSTOP)
 	{
 		//
 		// Check if bus busy
 		//
		if (I2C_getIcStatus(I2C_BASE)&0x1)
		{
			return ERROR_BUS_BUSY;
		}

 		//
 		// Send data to setup EEPROM address
 		//
// 		I2C_putData(I2C_BASE, msg->memoryHighAddr);
 		I2C_sendCmdWriteAndStop(I2C_BASE, msg->memoryLowAddr);
 	}
 	return(SUCCESS);
 }


 //
 // I2C A ISR (non-FIFO)
 //
void Check_Status(void)
{
 	uint16_t i = 0;
 	i = 0;

 	if((I2C_getInterruptStatus(I2C_BASE) & I2C_IC_INTR_STAT_STOP_DET) != 0U)
 	{
 		//
 		// If completed message was writing data, reset msg to inactive state
 		//
 		if(currentMsgPtr->msgStatus == MSG_STATUS_WRITE_BUSY)
 		{
 			currentMsgPtr->msgStatus = MSG_STATUS_INACTIVE;
 		}
 		else
 		{
 			//
 			// If a message receives a NACK during the address setup portion of
 			// the EEPROM read, the code further below included in the register
 			// access ready interrupt source code will generate a stop
 			// condition. After the stop condition is received (here), set the
 			// message status to try again. User may want to limit the number
 			// of retries before generating an error.
 			//
 			if(currentMsgPtr->msgStatus == MSG_STATUS_SEND_NOSTOP_BUSY)
 			{
 				currentMsgPtr->msgStatus = MSG_STATUS_SEND_NOSTOP;
 			}
 			//
 			// If completed message was reading EEPROM data, reset message to
 			// inactive state and read data from FIFO.
 			//
 			else if(currentMsgPtr->msgStatus == MSG_STATUS_READ_BUSY)
 			{
 				currentMsgPtr->msgStatus = MSG_STATUS_INACTIVE;
 				I2C_pollTxEmpty(I2C_BASE);
 				for(i=0; i < NUM_BYTES; i++)
 				{
 					if(i == (NUM_BYTES - 1)){
 						I2C_sendCmdReadAndStop(I2C_BASE);
 					}else{
 						I2C_sendCmdRead(I2C_BASE);
 					}
 					I2C_pollRxFull(I2C_BASE);
 					currentMsgPtr->msgBuffer[i] = I2C_getData(I2C_BASE);

 				}
 				RecvDataCheck();
 			}
 		}
 		I2C_clearInterruptStatus(I2C_BASE,I2C_IC_INTR_STAT_STOP_DET);
 	}

 	DEVICE_DELAY_US(5);//

 	if((I2C_getInterruptStatus(I2C_BASE) & I2C_INT_NO_ACK) != 0U)
	{
		currentMsgPtr->msgStatus = MSG_STATUS_SEND_NOSTOP;
		I2C_clearInterruptStatus(I2C_BASE,I2C_INT_NO_ACK);
	}else{
		if(currentMsgPtr->msgStatus == MSG_STATUS_SEND_NOSTOP_BUSY)
		{
			currentMsgPtr->msgStatus = MSG_STATUS_RESTART;
		}
	}
}



void RecvDataCheck(void)
{
	uint16_t i = 0;

	failCount = 0;
	passCount = 0;
	for(i=0; i < NUM_BYTES; i++)
	{
		if(i2cMsgIn.msgBuffer[i] == i2cMsgOut.msgBuffer[i])
		{
			passCount++;
		}
		else
		{
			failCount++;
		}
	}


}




#ifdef __cplusplus
}
#endif
