/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


#ifdef __cplusplus
extern "C"{
#endif

#include "device.h"
#include "slave.h"

uint8_t slave_txdata[W_DATA_LEN] = {0};
uint8_t slave_rxdata[W_DATA_LEN] = {0};
uint32_t g_slave_i2c_base  = I2CB_BASE;
uint32_t i2c_slave_port = 0;

volatile uint8_t i2c_slave_rcv_data_len = 0;

void i2c_err_irq_check(uint32_t i2c_base, uint32_t i2c_port_num, uint32_t irq_status)
{
    if (irq_status & I2C_INT_TX_ABRT)
    {
        I2C_clearInterruptTxAbrt(i2c_base);
    }

    if (irq_status & I2C_INT_TX_OVER)
    {
        I2C_clearInterruptTxOver(i2c_base);
    }

    if (irq_status & I2C_INT_RX_OVER)
    {
        I2C_clearInterruptRxOver(i2c_base);
    }

    if (irq_status & I2C_INT_RX_UNDER)
    {
        I2C_clearInterruptRxUnder(i2c_base);
    }
}



void I2CX_SLAVE_INTR_IRQHandler(void)
{
	static uint8_t rx_finish = 0;
    uint16_t i = 0;
    uint32_t irq_status = I2C_getInterruptStatus(g_slave_i2c_base);

    i2c_err_irq_check(g_slave_i2c_base, i2c_slave_port, irq_status);

    if (irq_status & I2C_IC_INTR_STAT_RD_REQ)//interrupt is triggered when the master attempts to read data from slave.
    {
    	for(i = 0; i < W_DATA_LEN; i++){
			if ((I2C_getIcStatus(g_slave_i2c_base) & I2C_ICSTS_TFNF) == I2C_ICSTS_TFNF)
			{
				if(i == W_DATA_LEN - 1)
				{
					I2C_sendCmdWriteAndStop(g_slave_i2c_base, slave_txdata[i]);
				}
				else
				{
					I2C_putData(g_slave_i2c_base, slave_txdata[i]);
				}
				I2C_pollTxEmpty(g_slave_i2c_base);
			}
    	}
        I2C_clearInterruptRdReq(g_slave_i2c_base);
    }

    if (irq_status & I2C_IC_INTR_STAT_RX_FULL) // cannot use "I2C_IC_INTR_STAT_WR_REQ"
    {
    	rx_finish = 1;
        if ((I2C_getIcStatus(g_slave_i2c_base) & I2C_ICSTS_RFNE) == I2C_ICSTS_RFNE)
        {
        	slave_rxdata[i2c_slave_rcv_data_len] = I2C_getData(g_slave_i2c_base);

			i2c_slave_rcv_data_len++;
        }
    }

    if (irq_status & I2C_IC_INTR_STAT_STOP_DET)	//Received the stop signal
    {
    	if(rx_finish == 1){
    		rx_finish = 0;
    		i2c_slave_check_data_rcv();	//Check if the received data is the same as the sent data
    	}
		I2C_clearInterruptStopDet(g_slave_i2c_base);
		i2c_slave_rcv_data_len = 0;
    }
    I2C_clearAllInterruptStatus(g_slave_i2c_base);
}


void I2C_slave_init(uint32_t i2c_base)
{

	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setTargetAddress(i2c_base,I2C_SLAVE_ADDRESS);
	I2C_setConfig(i2c_base, I2C_SLAVE_MODE | I2C_TX_EMPTY_CTRL | I2C_RESTART_EN);//must be I2C_SLAVE_MODE
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);
}

void i2c_slave_verify(uint32_t i2c_base)
{
    uint32_t i = 0;
#if(IS_GS32F3xx(0x22))
	uint32_t i2c_port_irqn = INT_I2CB;
#else
	uint32_t i2c_port_irqn = INT_I2C1;
#endif

    for(i = 0; i < W_DATA_LEN; i++){
    	slave_txdata[i] = i;
    	slave_rxdata[i] = 0;
    }



    g_slave_i2c_base = i2c_base;


    I2C_slave_init(i2c_base);


    Interrupt_register(i2c_port_irqn, I2CX_SLAVE_INTR_IRQHandler);
    //enable error interrupt
    I2C_enableInterrupt(i2c_base, I2C_INT_STOP_CONDITION | I2C_INT_RX_OVER | I2C_INT_RX_UNDER | I2C_INT_TX_ABRT | I2C_INT_TX_OVER);
    Interrupt_enable(i2c_port_irqn);

    //enable transmit and receive interrupt
    I2C_enableInterrupt(i2c_base, I2C_INT_RXFF | I2C_INT_WR_REQ | I2C_INT_RD_REQ | I2C_INT_RX_DONE);
}

void i2c_slave_check_data_rcv(void)
{
    for (uint32_t i = 0; i < i2c_slave_rcv_data_len; i++)
    {
    	if (slave_rxdata[i] != slave_txdata[i])
		{
			return;
		}
    }
}


void i2c_test03_master_slave(void)
{

    GPIO_setPinConfig(GPIO_2_I2CB_SDA);
    GPIO_setPadConfig(0, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);//I2CA_SDA_GP0
    GPIO_setQualificationMode(0, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(GPIO_3_I2CB_SCL);
    GPIO_setPadConfig(1, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);//I2CA_SCL_GP1
    GPIO_setQualificationMode(1, GPIO_QUAL_ASYNC);


    __enable_irq();
    i2c_slave_verify(I2CB_BASE);
    i2c_slave_port = 1;

}


#ifdef __cplusplus
}
#endif
