/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "eeprom_state.h"

uint32_t i2c_state = I2C_IS_BUSY;
uint8_t check_rdata[PAGE_SIZE*PAGE_MAX_NUM] = {0x00};
uint8_t write_retry_num = 0;

void i2c_irq(void)
{
	I2C_InterruptSource intSource;
	intSource = I2C_getInterruptSource(I2C_BASE);

	if(intSource == I2C_INTSRC_ARB_LOST){
		i2c_state = I2C_IS_ARB_LOST;
		I2C_clearInterruptStatus(I2C_BASE,I2C_INTSRC_ARB_LOST);
	}else if(intSource == I2C_INTSRC_NO_ACK){
		i2c_state = I2C_IS_NACK;
		I2C_clearInterruptStatus(I2C_BASE,I2C_INTSRC_NO_ACK);
	}
}

void i2c_init()
{
#if (GS32_PART_NUM != 0x35)
	GPIO_setPinConfig(I2C_SDA_PINMUX);
    GPIO_setPadConfig(I2C_SDA_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SDA_PIN, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(I2C_SCL_PINMUX);
    GPIO_setPadConfig(I2C_SCL_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SCL_PIN, GPIO_QUAL_ASYNC);

#else

    I2C_SDA_PIN_MUX;
    I2C_SCL_PIN_MUX;

#endif

    I2C_disableModule(I2C_BASE);

    I2C_initController(I2C_BASE, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
    I2C_setAddressMode(I2C_BASE, I2C_ADDR_MODE_7BITS);
	I2C_setTargetAddress(I2C_BASE, EEPROM_ADDR);
    I2C_setBitCount(I2C_BASE, I2C_BITCOUNT_8);
    I2C_setEmulationMode(I2C_BASE, I2C_EMULATION_FREE_RUN);

    I2C_clearInterruptStatus(I2C_BASE,I2C_INT_ARB_LOST|I2C_INT_NO_ACK);
    I2C_enableInterrupt(I2C_BASE,I2C_INT_ARB_LOST|I2C_INT_NO_ACK);

    I2C_enableModule(I2C_BASE);

    Interrupt_register(INT_I2CB, &i2c_irq);
    Interrupt_enable(INT_I2CB);
}

static uint8_t eeprom_page_write(uint16_t target_page,uint16_t page_start_addr,uint8_t* pdata,uint16_t data_length)
{
	static uint16_t cnt;
	static uint8_t send_addr_flag = 1;

	if((i2c_state==I2C_IS_ARB_LOST)||(i2c_state==I2C_IS_NACK)){
		return -1;
	}

	switch(i2c_state){
		case I2C_IS_STOP:
		case I2C_IS_BUSY:
			if(I2C_isBusBusy(I2C_BASE)){
				i2c_state = I2C_IS_BUSY;
			}else{
				i2c_state = I2C_IS_IDLE;
			}
			break;

		case I2C_IS_IDLE:
			I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);
			I2C_setDataCount(I2C_BASE, data_length+REG_ADDR_SZIE);
			i2c_state = I2C_IS_READY;
			break;

		case I2C_IS_READY:
			I2C_sendStartCondition(I2C_BASE);
			cnt = 0;
			i2c_state = I2C_SEND_REGADDR;
			break;
		case I2C_SEND_REGADDR:
#if(REG_ADDR_SZIE == 1)

				if(I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY){
					I2C_putData(I2C_BASE,(target_page*PAGE_SIZE+page_start_addr));
					i2c_state = I2C_IS_TRANSFER;
				}

#elif(REG_ADDR_SZIE == 2)

				if((I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY) && (send_addr_flag == 1)){
					I2C_putData(I2C_BASE,((target_page*PAGE_SIZE+page_start_addr)>>8));
					send_addr_flag = 2;
				}else if((I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY) && (send_addr_flag == 2)){
					I2C_putData(I2C_BASE,((target_page*PAGE_SIZE+page_start_addr)&0xff));
					send_addr_flag = 1;
					i2c_state = I2C_IS_TRANSFER;
				}
#endif
			break;
		case I2C_IS_TRANSFER:
			if(cnt == data_length){
				I2C_sendStopCondition(I2C_BASE);
				DELAY_US(5500);
				i2c_state = I2C_IS_STOP;
				return 0;
			}else{
				if(I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY){
					I2C_putData(I2C_BASE,pdata[cnt++]);
				}
			}
			break;
	}
	return 1;
}

uint8_t eeprom_write(uint16_t reg_addr,uint8_t* pdata,uint16_t data_length)
{
	if((reg_addr+data_length)>(PAGE_SIZE*PAGE_MAX_NUM))
		return -1;

	uint16_t start_page = reg_addr/PAGE_SIZE;
	uint16_t start_page_addr = reg_addr%PAGE_SIZE;
	uint16_t page_num = 0;
	uint16_t remain_bytes = 0;
	uint16_t last_page_bytes = 0;
	static uint16_t num = 0;
	static uint16_t i = 0;
	static uint8_t write_over_flag = 0;
	uint16_t j = 0;

	page_num = (start_page_addr + data_length)/PAGE_SIZE;
	if(((start_page_addr + data_length)%PAGE_SIZE) != 0){
		page_num++;
	}

	if(page_num > 1){
		if(((start_page_addr + data_length)%PAGE_SIZE) == 0){
			last_page_bytes = PAGE_SIZE;
		}else{
			last_page_bytes = (start_page_addr+data_length-PAGE_SIZE)%PAGE_SIZE;
		}

	}
	if(page_num == 1){
		remain_bytes = data_length;
	}else{
		remain_bytes = PAGE_SIZE - start_page_addr;
	}

	if(i<page_num){
		if(i == 0){
			if(eeprom_page_write(i+start_page,start_page_addr,&pdata[num],remain_bytes)==0){
				num += remain_bytes;
				i++;
			}
		}else if(i == (page_num-1)){
			if(eeprom_page_write(i+start_page,0,&pdata[num],last_page_bytes)==0){
				num += last_page_bytes;
				i++;
			}
		}else{
			if(eeprom_page_write(i+start_page,0,&pdata[num],PAGE_SIZE)==0){
				num += PAGE_SIZE;
				i++;
			}
		}
		if(i == page_num){
			write_over_flag = 1;
		}
	}
	if(write_over_flag == 1){
		if(eeprom_read(reg_addr,check_rdata,data_length) == 0){
			i = 0;
			num = 0;
			write_over_flag = 0;
			for(j=0;j<data_length;j++){
				if(check_rdata[j]!=pdata[j]){
					write_retry_num++;
					break;
				}
			}
			if(write_retry_num == EEPROM_WRITE_RETRY_COUNT){
				write_retry_num = 0;
				return 2;
			}
			if(j==data_length){
				write_retry_num = 0;
				return 0;
			}
		}
	}
	return 1;
}

uint8_t eeprom_read(uint16_t reg_addr,uint8_t* pdata,uint16_t data_length)
{
	static uint16_t cnt;
	static uint8_t send_addr_flag = 1;

	if((i2c_state==I2C_IS_ARB_LOST)||(i2c_state==I2C_IS_NACK)){
		return -1;
	}

	switch(i2c_state){
		case I2C_IS_STOP:
		case I2C_IS_BUSY:
			if(I2C_isBusBusy(I2C_BASE)){
				i2c_state = I2C_IS_BUSY;
			}else{
				i2c_state = I2C_IS_IDLE;
			}
			break;
		case I2C_IS_IDLE:
			I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);
			I2C_setDataCount(I2C_BASE,REG_ADDR_SZIE);
			i2c_state = I2C_IS_READY;
			break;
		case I2C_IS_READY:
			I2C_sendStartCondition(I2C_BASE);
			cnt = 0;
			i2c_state = I2C_SEND_REGADDR;
			break;
		case (I2C_SEND_REGADDR):
#if(REG_ADDR_SZIE == 1)
				if(I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY){
					I2C_putData(I2C_BASE,(reg_addr & 0xff));
					i2c_state = I2C_IS_RESTART;
				}
#elif(REG_ADDR_SZIE == 2)
				if((I2C_getStatus(I2C_BASE)&I2C_STS_TX_DATA_RDY)&&(send_addr_flag == 1)){
					I2C_putData(I2C_BASE,((reg_addr>>8)&0xff));
					send_addr_flag = 2;
				}else if((I2C_getStatus(I2C_BASE)&I2C_STS_TX_DATA_RDY)&&(send_addr_flag == 2)){
					I2C_putData(I2C_BASE,(reg_addr&0xff));
					send_addr_flag = 1;
					i2c_state = I2C_IS_RESTART;
				}
#endif
			break;
		case I2C_IS_RESTART:
			if((I2C_getStatus(I2C_BASE) & I2C_STS_REG_ACCESS_RDY)){
				I2C_setConfig(I2C_BASE, I2C_CONTROLLER_RECEIVE_MODE);
				I2C_setDataCount(I2C_BASE,data_length);
				I2C_sendStartCondition(I2C_BASE);
				i2c_state = I2C_IS_TRANSFER;
			}
			break;

		case I2C_IS_TRANSFER:
		if(cnt == data_length){
			I2C_sendStopCondition(I2C_BASE);
			i2c_state = I2C_IS_STOP;
			return 0;
		}else{
			if(I2C_getStatus(I2C_BASE) & I2C_STS_RX_DATA_RDY){
				pdata[cnt++] = I2C_getData(I2C_BASE);
			}
		}
	}

	return 1;
}







