/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "device.h"
#include "lin_transfer.h"

uint8_t transfer_data[LIN_DATA_LENGTH];

LIN_LinMsg_t LinRxMsg={
		.id = 0x00,
		.data = transfer_data
};

void lin_irq_handler(void)
{
	LIN_LinModeNodeAction_t node_action;

	LIN_LinGetNodeAction(LIN_BASE, &node_action);
	if (node_action == LIN_LIN_OP_MODE_SUBSCRIBE) {
		LIN_LinSlaveReceiveFrameBlock(LIN_BASE, &LinRxMsg);
		/* Set node action to transmitter. */
		if (LIN_LinSetNodeAction(LIN_BASE, LIN_LIN_OP_MODE_PUBLISH))
			while (1);
	} else if (node_action == LIN_LIN_OP_MODE_PUBLISH) {
		if (LIN_LinSetNodeAction(LIN_BASE, LIN_LIN_OP_MODE_PUBLISH))
			while (1);
		LinRxMsg.data[0]++;
		LIN_LinSlaveSendFrameBlock(LIN_BASE, &LinRxMsg);
		/* Set node action to receiver. */
		if (LIN_LinSetNodeAction(LIN_BASE, LIN_LIN_OP_MODE_SUBSCRIBE))
			while (1);
	}
}

static void lin_module_lin_slave_mode_init(void)
{
	LIN_initParam_t LINinitParam = {0};
	LIN_LinModeInitParam_t LINModeInitParam = {0};

	LINinitParam.bit_order = LIN_TRANSFER_LSB;
	LINinitParam.char_length = LIN_CHAR_LENGTH_8BITS;
	LINinitParam.opmode = LIN_LIN_SLAVE_MODE;
	LINinitParam.chl_mode = LIN_CHAL_NORMAL_MODE;
	LINinitParam.stopbit = LIN_STOP_BIT_1;
	LINinitParam.parity = LIN_PARITY_NONE;

	LINModeInitParam.dma_en = false;
	LINModeInitParam.frame_slot_en = false;
	LINModeInitParam.checksum_en = true;
	LINModeInitParam.parity_en = true;
	LINModeInitParam.data_length_mode = LIN_LIN_DLM_BY_DLC;
	LINModeInitParam.iso_mode = LIN_LIN_ISO_MODE_1_3;
	LINModeInitParam.slave_sync_en = false;

	/* Initialize LIN module. */
	if (LIN_initModule(LIN_BASE, &LINinitParam))
		while(1);

	/* Set node action to receiver. */
	if (LIN_LinSetNodeAction(LIN_BASE, LIN_LIN_OP_MODE_PUBLISH))
		while (1);

	/* Initialize LIN mode. */
	if (LIN_LinModeInit(LIN_BASE, &LINModeInitParam))
		while (1);

	/* Set baud rate. */
#if (IS_GS32F00xx(0x30))

	uint32_t lin_clk_div = SysCtl_getLinClkDiv();
	lin_clk_div *= 2;

	if (LIN_setBaudRate(LIN_BASE,(DEVICE_PLLCLK_FREQ / lin_clk_div), 9600,
						LIN_OP_ASYNC_MODE, LIN_OVERSAMP_16))
		while (1);
#else
	if (LIN_setBaudRate(LIN_BASE, DEVICE_APBCLK_FREQ, 9600,
						LIN_OP_ASYNC_MODE, LIN_OVERSAMP_16))
		while (1);
#endif

	/* Set data length. */
	if (LIN_LinSetDataLength(LIN_BASE, LIN_DATA_LENGTH))
		while (1);

	/* Enable receiver. */
	if (LIN_enableReceiver(LIN_BASE, true))
		while (1);

	/* Enable transmitter. */
	if (LIN_enableTransmitter(LIN_BASE, true))
		while (1);
}

static void lin_clk_init(void)
{
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_LIN);

#if (IS_GS32F00xx(0x30))

	int Lin_defaultClkDiv = 2;

	SysCtl_setLinClkDiv(Lin_defaultClkDiv);

#endif

	SysCtl_resetLin();
}

void lin_init(void)
{
	/* Set PIN mux */
#if(GS32_PART_NUM == 0x35)

	LIN_TX_PIN_MUX;
	LIN_RX_PIN_MUX;

#else

	GPIO_setPinConfig(LIN_TX_PIN);
	GPIO_setPinConfig(LIN_RX_PIN);

#endif

	/* Initialize LIN clock */
	lin_clk_init();

	/* Initialize LIN to LIN slave mode. */
	lin_module_lin_slave_mode_init();

	if (LIN_LinSetNodeAction(LIN_BASE, LIN_LIN_OP_MODE_SUBSCRIBE))
		while (1);

	if (LIN_InterruptDisable(LIN_BASE, 0xffffffffU))
		while (1);

	if (LIN_InterruptEnable(LIN_BASE, LIN_LIN_MODULE_INTR_LINID))
		while (1);

	Interrupt_SetPriority(LIN_IRQ_NUMBER, 0, 0);
	Interrupt_register(LIN_IRQ_NUMBER, &lin_irq_handler);
	Interrupt_enable(LIN_IRQ_NUMBER);
}
