/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdbool.h>

#include "mcan_scheduled.h"
#include "mcan.h"
#include "device.h"
#define M_CAN_MSG_MAX_LENGTH			MCAN_ELEM_SIZE_8BYTES
#define M_CAN_MSG_BUFFER_SIZE			(MCAN_getMsgObjSize(M_CAN_MSG_MAX_LENGTH) + 8U)

#define M_CAN_RAM_BASE_ADDR				0U
#define M_CAN_STANDARD_FILTER_BASE_ADDR	M_CAN_RAM_BASE_ADDR
#define M_CAN_STANDARD_FILTER_NUM		(4U)
#define M_CAN_STANDARD_FILTER_SIZE		(4U)

/* Note: The size of all memory regions cannot exceed 4096 bytes. */
#define M_CAN_EXTENDED_FILTER_BASE_ADDR	(M_CAN_STANDARD_FILTER_BASE_ADDR + \
										(M_CAN_STANDARD_FILTER_NUM * \
										 M_CAN_STANDARD_FILTER_SIZE))
#define M_CAN_EXTENDED_FILTER_NUM		(2U)
#define M_CAN_EXTENDED_FILTER_SIZE		(8U)

#define M_CAN_RXFIFO0_BASE_ADDR			(M_CAN_EXTENDED_FILTER_BASE_ADDR + \
										(M_CAN_EXTENDED_FILTER_NUM * \
										 M_CAN_EXTENDED_FILTER_SIZE))
#define M_CAN_RXFIFO0_NUM				(5U)
#define M_CAN_RXFIFO0_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXFIFO1_BASE_ADDR			(M_CAN_RXFIFO0_BASE_ADDR + \
										(M_CAN_RXFIFO0_NUM * \
										 M_CAN_RXFIFO0_SIZE))
#define M_CAN_RXFIFO1_NUM				(5U)
#define M_CAN_RXFIFO1_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXBUFFER_BASE_ADDR		(M_CAN_RXFIFO1_BASE_ADDR + \
										(M_CAN_RXFIFO1_NUM * \
										 M_CAN_RXFIFO1_SIZE))
#define M_CAN_RXBUFFER_NUM				(8U)
#define M_CAN_RXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXEVENTFIFO_BASE_ADDR		(M_CAN_RXBUFFER_BASE_ADDR + \
										(M_CAN_RXBUFFER_NUM * \
										 M_CAN_RXBUFFER_SIZE))
#define M_CAN_TXEVENTFIFO_NUM			(10U)
#define M_CAN_TXEVENTFIFO_SIZE			(M_CAN_TX_EVENT_FIFO_SIZE)

#define M_CAN_TXBUFFER_BASE_ADDR		(M_CAN_TXEVENTFIFO_BASE_ADDR + \
										(M_CAN_TXEVENTFIFO_NUM * \
										 M_CAN_TXEVENTFIFO_SIZE))
#define M_CAN_TXBUFFER_NUM				(4U)
#define M_CAN_TXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXFIFOQUEUE_BASE_ADDR		(M_CAN_TXBUFFER_BASE_ADDR + \
										(M_CAN_TXBUFFER_NUM * \
										 M_CAN_TXBUFFER_SIZE))
#define M_CAN_TXFIFOQUEUE_NUM			(6U)
#define M_CAN_TXFIFOQUEUE_SIZE			(M_CAN_MSG_BUFFER_SIZE)

void mcan_config(void)
{
	MCAN_InitParams_t init_param;
	MCAN_ConfigParams_t configParams;
	MCAN_MsgRAMConfigParams_t RAMConfig;
	MCAN_BitTimingParams_t configBitrate;
	MCAN_StdMsgIDFilterElement_t std_filter;
	MCAN_ExtMsgIDFilterElement_t ext_filter;

	/* CAN mode configuration. */
	init_param.fdMode = false;
	init_param.fdFormat = MCAN_FD_ISO_11898_1;
	init_param.brsEnable = false;
	init_param.txpEnable = false;
	init_param.efbi = false;
	init_param.pxhddisable = false;
	/* Enable the auto retransmission. */
	init_param.darEnable = true;
	init_param.wkupReqEnable = false;
	init_param.autoWkupEnable = false;
	init_param.emulationEnable = true;
	init_param.wdcPreload = false;
	init_param.wmMarker = MCAN_WMM_8BIT_MODE;
	init_param.tdcEnable = false;
	init_param.tdcConfig.tdcf = 0U;
	init_param.tdcConfig.tdco = 0U;

	/* Set CAN extended feature. */
	configParams.asmEnable = false;
	configParams.monEnable = false;
	configParams.timeoutCntEnable = false;
	configParams.timeoutPreload = 1;
	configParams.timeoutSelect = MCAN_TIMEOUT_SELECT_CONT;

	configParams.tsClock = MCAN_INTERNAL_TIMESTAMP;
	configParams.tsSelect = MCAN_TSCNTVAL_ALWAYS0;
	configParams.tsPrescalar = 1;

	/* Set global configuration. */
	configParams.filterConfig.rrfs = true;
	configParams.filterConfig.rrfe = true;
	configParams.filterConfig.anfs = MCAN_NON_MATCH_REJECT_ACCEPT;
	configParams.filterConfig.anfe = MCAN_NON_MATCH_ACCEPT_IN_FIFO1;

	/* Set standard ID filter. */
	RAMConfig.lss = M_CAN_STANDARD_FILTER_NUM;
	RAMConfig.flssa = M_CAN_STANDARD_FILTER_BASE_ADDR;

	/* Set extended ID filter. */
	RAMConfig.lse = M_CAN_EXTENDED_FILTER_NUM;
	RAMConfig.flesa = M_CAN_EXTENDED_FILTER_BASE_ADDR;

	/* Set Rx fifo 0. */
	RAMConfig.rxFIFO0OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO0startAddr = M_CAN_RXFIFO0_BASE_ADDR;
	RAMConfig.rxFIFO0size = M_CAN_RXFIFO0_NUM;
	RAMConfig.rxFIFO0waterMark = 0;
	RAMConfig.rxFIFO0ElemSize  = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx fifo 1. */
	RAMConfig.rxFIFO1OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO1startAddr = M_CAN_RXFIFO1_BASE_ADDR;
	RAMConfig.rxFIFO1size = M_CAN_RXFIFO1_NUM;
	RAMConfig.rxFIFO1waterMark = 0;
	RAMConfig.rxFIFO1ElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx buffer. */
	RAMConfig.rxBufStartAddr = M_CAN_RXBUFFER_BASE_ADDR;
	RAMConfig.rxBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx buffer */
	RAMConfig.txBufMode = MCAN_TXBUF_OP_IN_FIFO_MODE;
	RAMConfig.txStartAddr = M_CAN_TXBUFFER_BASE_ADDR;
	RAMConfig.txFIFOSize = M_CAN_TXFIFOQUEUE_NUM;
	RAMConfig.txBufNum = M_CAN_TXBUFFER_NUM;
	RAMConfig.txBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx event fifo. */
	RAMConfig.txEventFIFOStartAddr = M_CAN_TXEVENTFIFO_BASE_ADDR;
	RAMConfig.txEventFIFOSize = M_CAN_TXEVENTFIFO_NUM;
	RAMConfig.txEventFIFOWaterMark = 1;

	/* Input clock source 40Mhz, normal phase bitrate 500kbps. */
	configBitrate.nomRatePrescalar = 1;
	configBitrate.nomTimeSeg1 = 30;
	configBitrate.nomTimeSeg2 = 7;
	configBitrate.nomSynchJumpWidth = 1;

	/* Reset the MCAN module. */
	while (MCAN_isInReset(CAN_BASE));

	/* Check if the MCAN RAM is ready. */
	while (!MCAN_isMemInitDone(CAN_BASE));

	/* Set the MCAN mode to init mode. */
	if (!MCAN_setOpMode(CAN_BASE, MCAN_OPERATION_SW_INIT_MODE))
		while (1);

	/* Initialize the MCAN. */
	if (!MCAN_init(CAN_BASE, &init_param))
		while (1);

	if (!MCAN_config(CAN_BASE, &configParams))
		while (1);

	if (!MCAN_setBitTime(CAN_BASE, &configBitrate))
		while (1);

	if (!MCAN_msgRAMConfig(CAN_BASE, &RAMConfig))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 0). */
	std_filter.sfid1 = 0x123;
	std_filter.sfid2 = 0x0;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 0, &std_filter))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 1). */
	std_filter.sfid1 = 0x234;
	std_filter.sfid2 = 0x1;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 1, &std_filter))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 2). */
	std_filter.sfid1 = 0x345;
	std_filter.sfid2 = 0x2;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 2, &std_filter))
		while (1);

	/* Add a new extended ID filter(Use Rx fifo 0). */
	ext_filter.efid1 = 0x11;
	ext_filter.efid2 = 0x1fffff;
	ext_filter.esync = MCAN_EXTIDF_SYNC_MSG_DISABLE;
	ext_filter.efec = MCAN_EXTIDF_ELE_STORE_IN_FO0_OF_MATCH_ID;
	ext_filter.eft = MCAN_EXTIDF_CLASSIC_ID_FILTER;
	if (!MCAN_addExtMsgIDFilter(CAN_BASE, 0, &ext_filter))
		while (1);

	ext_filter.efid1 = 0x22;
	ext_filter.efid2 = 0x1fffffff;
	ext_filter.esync = MCAN_EXTIDF_SYNC_MSG_DISABLE;
	ext_filter.efec = MCAN_EXTIDF_ELE_STORE_IN_FO1_OF_MATCH_ID;
	ext_filter.eft = MCAN_EXTIDF_CLASSIC_ID_FILTER;
	if (!MCAN_addExtMsgIDFilter(CAN_BASE, 1, &ext_filter))
		while (1);

	/* The received extension frame will be subjected to AND operation with Ext_mask. */
	if (!MCAN_setExtIDAndMask(CAN_BASE, 0x1FFFFFFF))
		while (1);

	/* Enable/Disable Loopback mode. */
	if (!MCAN_lpbkModeEnable(CAN_BASE, MCAN_LPBK_MODE_INTERNAL, false))
		while (1);

	/* Enable MCAN. */
	if (!MCAN_setOpMode(CAN_BASE, MCAN_OPERATION_NORMAL_MODE))
		while (1);

	 /* Configuration the external timestamp clock source. */
	 if (!MCAN_extTSCounterConfig(CAN_BASE, 0xffff))
	 	while (1);

	 /* Enable/Disable external timestamp clock source. */
	 if (!MCAN_extTSCounterEnable(CAN_BASE, true))
	 	while (1);

	 /* Enable/Disable external timestamp overflow interrupt. */
	 if (!MCAN_extTSEnableIntr(CAN_BASE, false))
	 	while (1);
}

void mcan_init(void)        // Initialize eCAN-A module
{
	MCAN_RxNewDataStatus_t rx_newdata;
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);
	SysCtl_setCanClkDiv(DEVICE_PLLCLK_FREQ / 40000000U);
	SysCtl_resetCana();
	GPIO_setPinConfig(CAN_TX_PIN);
	GPIO_setPinConfig(CAN_RX_PIN);
	mcan_config();
}

void scheduledreception(void)
{
	MCAN_TxMessage_t TxMessageBuf;

	TxMessageBuf.brs = 0;
	TxMessageBuf.dlc = MCAN_DATA_LENGTH_8;
	TxMessageBuf.efc = 0;
	TxMessageBuf.esi = 0;
	TxMessageBuf.fdf = 0;
	TxMessageBuf.id = M_CAN_STANDARD_ID_W(0x1);
	TxMessageBuf.mm = 0x0;
	TxMessageBuf.mm1 = 0x0;
	TxMessageBuf.rtr = 0;
	TxMessageBuf.tsce = 0;
	TxMessageBuf.xtd = 0;

	for (uint32_t i = 0; i < MCAN_getDataSize(TxMessageBuf.dlc); i++)
		TxMessageBuf.data[i] = i;
		/* Send standard frame. */
	TxMessageBuf.xtd = 0;
	TxMessageBuf.id_std = 0x7FF;
	while (!MCAN_transmitMsgBuffer(CAN_BASE, &TxMessageBuf, 0));

	TxMessageBuf.id_std = 0x4AE;
	while (!MCAN_transmitMsgBuffer(CAN_BASE, &TxMessageBuf, 1));

	TxMessageBuf.id_std = 0x3AE;
	while (!MCAN_transmitMsgBuffer(CAN_BASE, &TxMessageBuf, 2));

	/* Send extended frame. */
	TxMessageBuf.id_ext = 0x177777;
	TxMessageBuf.xtd = 1;
	while (!MCAN_transmitMsgBuffer(CAN_BASE, &TxMessageBuf, 3));
}
void scheduledsend()
{
	MCAN_RxFIFOStatus_t fifo0status,fifo1status;
	MCAN_RxMessage_t fifo0_data, fifo1_data,rxbuffer0,rxbuffer1,rxbuffer2;
	uint32_t interrupt_status;
	MCAN_RxNewDataStatus_t rx_newdata;
	MCAN_RxNewDataStatus_t new_dat;

	fifo0status.num = MCAN_RX_FIFO_NUM_0;
	fifo1status.num = MCAN_RX_FIFO_NUM_1;

	interrupt_status = MCAN_getIntrStatus(CAN_BASE);

	if (interrupt_status & MCAN_INT_SRC_RxFIFO0_NEW_MSG) {
		MCAN_receiveMsgFromFifo0(CAN_BASE, &fifo0_data);

		fifo0status.num = MCAN_RX_FIFO_NUM_0;
		MCAN_getRxFIFOStatus(CAN_BASE, &fifo0status);

		if (fifo0status.fillLvl == 0)
			MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_RxFIFO0_NEW_MSG);
	}
	if (interrupt_status & MCAN_INT_SRC_RxFIFO1_NEW_MSG) {
		MCAN_receiveMsgFromFifo1(CAN_BASE, &fifo1_data);

		fifo1status.num = MCAN_RX_FIFO_NUM_1;
		MCAN_getRxFIFOStatus(CAN_BASE, &fifo1status);

		if (fifo1status.fillLvl == 0)
			MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_RxFIFO1_NEW_MSG);
	}
	if (interrupt_status & MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF) {
		MCAN_getNewDataStatus(CAN_BASE, &new_dat);

		if (new_dat.statusLow & (1U << 0)) {
			MCAN_receiveMsgFromBuffer(CAN_BASE, &rxbuffer0, 0);
		}

		if (new_dat.statusLow & (1U << 1)) {
			MCAN_receiveMsgFromBuffer(CAN_BASE, &rxbuffer1, 1);
		}

		if (new_dat.statusLow & (1U << 2)) {
			MCAN_receiveMsgFromBuffer(CAN_BASE, &rxbuffer2, 2);
		}

		MCAN_getNewDataStatus(CAN_BASE, &new_dat);
		if (new_dat.statusHigh == 0 && new_dat.statusLow == 0)
			MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF);
	}
}
