/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File:	spi_ex4_external_loopback_fifo_interrupt.c
 *
 * @Title:	SPI Digital Loopback with FIFO Interrupts.
 *
 * @brief:
 * This program uses the external loopback between two SPI modules.
 * Both the SPI FIFOs and their interrupts are used.
 * SPIA is configured as a peripheral and receives data from
 * SPI B which is configured as a controller.
 * A stream of data is sent and then compared to the received stream.
 * The sent data looks like this:
 * 0000 0001
 * 0001 0002
 * ....
 * FFFE FFFF
 * etc..
 *
 * @External Connections:
 * Refer to SysConfig for external connections (GPIO pin numbers)
 * specific to each device.
 *
 * @Watch Variables
 * - sData - Data to send
 * - rData - Received data
 * - rDataPoint - Used to keep track of the last position in the receive
 * 				  stream for error checking.
 */


#include <spi_ex04_board.h>
#include "driverlib.h"
#include "device.h"

/*
 * Globals Variables
 */
uint16_t sData[2];                  /* Send data buffer */
uint16_t rData[2];                  /* Receive data buffer */
uint16_t rDataPoint = 0;            /* To keep track of where we are in the
                                     data stream to check received data */

void spiARxFIFOISR(void);
void spiBTxFIFOISR(void);

/*
 * Main
 */
int main(void)
{
    uint16_t i;

    /*
     * Initialize device clock and peripherals
     */
    Device_init();

    __disable_irq();

    Board_init();

    /*
     * Initialize the data buffers
     */
    for(i = 0; i < 2; i++)
    {
        sData[i] = i;
        rData[i]= 0;
    }

    __enable_irq();

    /*
     * Loop forever. Suspend or place breakpoints to observe the buffers.
     */
    while(1)
    {

    }

    return 0;
}

void spiARxFIFOISR(void)
{
	uint16_t i;
	uint16_t status;
	status = SPI_getInterruptStatus(SPIA_BASE);

	/*
	 * Read data
	 */
    for(i = 0; i < 2; i++)
    {
        rData[i] = SPI_readDataNonBlocking(SPIA_BASE);
    }

    /*
     * Read data
     */
    for(i = 0; i < 2; i++)
    {
        if(rData[i] != (uint16_t)(rDataPoint + i))
        {
        	/*
        	 * Something went wrong. rData doesn't contain expected data.
        	 */
            ESTOP0;
        }
    }

    rDataPoint++;

    /*
     * RX FIFO Threshold Level interrupt is clear by the hardware
     */
}

void spiBTxFIFOISR(void)
{
	uint16_t i;
	uint16_t status;
	status = SPI_getInterruptStatus(SPIB_BASE);

	/*
	 * Send data
	 */
    for(i = 0; i < 2; i++)
    {
       SPI_writeDataNonBlocking(SPIB_BASE, sData[i]);
    }

    /*
     * Increment data for next cycle
     */
    for(i = 0; i < 2; i++)
    {
       sData[i] = sData[i] + 1;
    }

    /*
     * TX FIFO Threshold Level interrupt is clear by the hardware
     */
}

