/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


#include "spi_loopback.h"
#include "device.h"
#include "log.h"
#include "spi.h"

void spi_init(void)
{

    log_info("01_spi_loopback start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();
    /** <\ @brief holed SPI in reset state before configure SPI. */
    SPI_disableModule(CONFIG_SPI_BASE);

    SPI_setConfig(CONFIG_SPI_BASE, DEVICE_APBCLK_FREQ,
                  SPI_PROT_POL0PHA0, SPI_MODE_CONTROLLER,
                  CONFIG_SPI_BITRATE, CONFIG_SPI_DATAWIDTH);

    SPI_setPTESignalPolarity(CONFIG_SPI_BASE, SPI_PTE_ACTIVE_LOW);

    SPI_disableFIFO(CONFIG_SPI_BASE);

    SPI_setEmulationMode(CONFIG_SPI_BASE, SPI_EMULATION_FREE_RUN);

    SPI_disableInterrupt(CONFIG_SPI_BASE,
                         SPI_INT_RX_OVERRUN |
                         SPI_INT_RX_DATA_TX_EMPTY |
                         SPI_INT_RXFF | /** <\ @brief also disables SPI_INT_RXFF_OVERFLOW.*/
                         SPI_INT_TXFF);

    SPI_enableLoopback(CONFIG_SPI_BASE);
    /** <\ @brief release SPI. */
    SPI_enableModule(CONFIG_SPI_BASE);
}

void spi_process(void)
{
    volatile uint16_t sdata = 1;
    volatile uint16_t rdata = 0;

    while(1)
    {
        /** <\ @brief block translation. */
        SPI_writeDataBlockingNonFIFO(CONFIG_SPI_BASE, sdata);
        /** <\ @brief block received. */
        rdata = SPI_readDataBlockingNonFIFO(CONFIG_SPI_BASE);
        /** <\ @brief compare rdata with sdata. */
        if(sdata != rdata)
        {
            log_error("transmission failed!\n");
            ESTOP0;
        }
        else
        {
            log_info("sdata: %d rdata: %d \n", sdata, rdata);
        }
        sdata++;
    }
}
