/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "spi_external_loopback_master.h"
#include "log.h"
#include "spi.h"
#include "device.h"

#define __DATA_SIZE (16U)

static volatile uint16_t sdata[__DATA_SIZE] = {0xFF, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F};
static volatile uint16_t rdata[__DATA_SIZE] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static
void __spi_pin_init(void)
{
#if IS_GS32F00xx(0x30)
    GPIO_setPinConfig(CONFIG_SPI_MASTER_CLK);
    GPIO_setPadConfig(CONFIG_SPI_MASTER_CLK_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_MASTER_CLK_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_MASTER_SIMO);
    GPIO_setPadConfig(CONFIG_SPI_MASTER_SIMO_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_MASTER_SIMO_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_MASTER_SOMI);
    GPIO_setPadConfig(CONFIG_SPI_MASTER_SOMI_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_MASTER_SOMI_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_MASTER_CS);
    GPIO_setPadConfig(CONFIG_SPI_MASTER_CS_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_MASTER_CS_GPIO, GPIO_QUAL_ASYNC);
#endif /* IS_GS32F00xx(0x30) */
}

void spi_init(void)
{

    log_info("03_spi_external_loopback_master start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();
    /** <\ @brief SPI pins initial. */
    __spi_pin_init();

    /** <\ @brief holed SPI in reset state before configure SPI. */
    SPI_disableModule(CONFIG_SPI_MASTER_BASE);

    SPI_setConfig(CONFIG_SPI_MASTER_BASE, DEVICE_APBCLK_FREQ,
                  SPI_PROT_POL0PHA0, SPI_MODE_CONTROLLER,
                  CONFIG_SPI_BITRATE, CONFIG_SPI_DATAWIDTH);

    SPI_disableInterrupt(CONFIG_SPI_MASTER_BASE,
                         SPI_INT_RX_OVERRUN |
                         SPI_INT_RX_DATA_TX_EMPTY |
                         SPI_INT_RXFF | /** <\ @brief also disables SPI_INT_RXFF_OVERFLOW.*/
                         SPI_INT_TXFF);
    SPI_disableLoopback(CONFIG_SPI_MASTER_BASE);
    /** <\ @brief release SPI. */
    SPI_enableModule(CONFIG_SPI_MASTER_BASE);

}

void spi_process(void)
{
    uint16_t i;

    for(i = 0; i < __DATA_SIZE; i++)
    {
        /**
         * SPI_writeDataBlockingNonFIFO(CONFIG_SPI_MASTER_BASE, sdata0[i]);
         * rdata0[i] = SPI_readDataBlockingNonFIFO(CONFIG_SPI_MASTER_BASE);
         */
        rdata[i] = SPI_pollingNonFIFOTransaction(CONFIG_SPI_MASTER_BASE, CONFIG_SPI_DATAWIDTH, sdata[i]);
        if(rdata[i] != sdata[i])
        {
            log_error("transmission failed!\n");
            ESTOP0;
        }
    }

    log_info("test ok!\n");

    while(1)
    {
        __NOP();
    }
}
