/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "spi_external_loopback_fifo_slave.h"
#include "device.h"
#include "spi.h"
#include "log.h"

static volatile uint16_t sdata[CONFIG_SPI_FIFO_LEVE];
static volatile uint16_t rdata[CONFIG_SPI_FIFO_LEVE];

#define __DATA_FIELD (0xAAU)

static
void __spi_rx_isr_handle(void)
{
    uint16_t status, i;
    /** <\ @brief get interrupt status. */
    status = SPI_getInterruptStatus(CONFIG_SPI_SLAVE_BASE);
    /** <\ @check if fifo overrun. */
    if(status & SPI_INT_RXFF_OVERFLOW)
    {
        log_error("received overflow!\n");
        ESTOP0;
        SPI_clearInterruptStatus(CONFIG_SPI_SLAVE_BASE, SPI_INT_RXFF_OVERFLOW);
    }
    /** <\ @brief rxfifo event. */
    if(status & SPI_INT_RXFF)
    {
        for(i = 0; i < CONFIG_SPI_FIFO_LEVE; i++)
        {
            rdata[i] = SPI_readDataNonBlocking(CONFIG_SPI_SLAVE_BASE);
            if(__DATA_FIELD != rdata[i])
            {
                log_error("transmission failed!\n");
                ESTOP0;
            }
        }
        log_info("SLAVE: communication successfully!\n");
        Interrupt_disable(CONFIG_SPI_SLAVE_RX_IRQn);
        SPI_clearInterruptStatus(CONFIG_SPI_SLAVE_BASE, SPI_INT_RXFF);
    }
}

static
void __spi_pin_init(void)
{
#if IS_GS32F00xx(0x30)
    GPIO_setPinConfig(CONFIG_SPI_SLAVE_CLK);
    GPIO_setPadConfig(CONFIG_SPI_SLAVE_CLK_GPIO, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(CONFIG_SPI_SLAVE_CLK_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_SLAVE_SIMO);
    GPIO_setPadConfig(CONFIG_SPI_SLAVE_SIMO_GPIO, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(CONFIG_SPI_SLAVE_SIMO_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_SLAVE_SOMI);
    GPIO_setPadConfig(CONFIG_SPI_SLAVE_SOMI_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_SLAVE_SOMI_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_SLAVE_CS);
    GPIO_setPadConfig(CONFIG_SPI_SLAVE_CS_GPIO, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(CONFIG_SPI_SLAVE_CS_GPIO, GPIO_QUAL_ASYNC);
#endif
}

void spi_init(void)
{
    uint16_t i;

    log_info("04_spi_external_loopback_fifo_slave start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();
    /** <\ @brief SPI pins initial. */
    __spi_pin_init();
    /** <\ @brief holed SPI in reset state before configure SPI. */
    SPI_disableModule(CONFIG_SPI_SLAVE_BASE);

    SPI_setConfig(CONFIG_SPI_SLAVE_BASE, DEVICE_APBCLK_FREQ,
                  SPI_PROT_POL0PHA1, SPI_MODE_PERIPHERAL,
                  CONFIG_SPI_BITRATE, CONFIG_SPI_DATAWIDTH);
    /** <\ @brief enable fifo mode. */
    SPI_enableFIFO(CONFIG_SPI_SLAVE_BASE);

    SPI_disableInterrupt(CONFIG_SPI_SLAVE_BASE,
                        SPI_INT_RXFF |
                        SPI_INT_RXFF_OVERFLOW |
                        SPI_INT_TXFF);
    /** <\ @brief reset rx/tx fifo pointer to zero. */
    SPI_resetTxFIFO(CONFIG_SPI_SLAVE_BASE);
    SPI_resetRxFIFO(CONFIG_SPI_SLAVE_BASE);
    SPI_reset(CONFIG_SPI_SLAVE_BASE);
    /** <\ @brief set fifo interrupt level. */
    SPI_setFIFOInterruptLevel(CONFIG_SPI_SLAVE_BASE,
                              CONFIG_SPI_FIFO_LEVE,
                              CONFIG_SPI_FIFO_LEVE);

    SPI_clearInterruptStatus(CONFIG_SPI_SLAVE_BASE,
                         SPI_INT_RX_OVERRUN |
                         SPI_INT_RX_DATA_TX_EMPTY |
                         SPI_INT_RXFF |
                         SPI_INT_RXFF_OVERFLOW |
                         SPI_INT_TXFF);

    SPI_enableInterrupt(CONFIG_SPI_SLAVE_BASE, SPI_INT_RXFF);
    /** <\ @brief release SPI. */
    SPI_enableModule(CONFIG_SPI_SLAVE_BASE);

    /** <\ @brief register system interrupt. */
    Interrupt_register(CONFIG_SPI_SLAVE_RX_IRQn, __spi_rx_isr_handle);
    Interrupt_enable(CONFIG_SPI_SLAVE_RX_IRQn);

    /** <\ @brief initial transmission data. */
    for(i = 0; i < CONFIG_SPI_FIFO_LEVE; i++)
    {
        sdata[i] = __DATA_FIELD;
        rdata[i] = 0;
    }

    for(i = 0; i < CONFIG_SPI_FIFO_LEVE; i++)
    {
        SPI_writeDataBlockingFIFO(CONFIG_SPI_SLAVE_BASE, sdata[i] << (16U - CONFIG_SPI_DATAWIDTH));
    }
}

void spi_process(void)
{
    uint16_t i;

    log_info("slave field data. wait master initiate transmission\n");
    while(1)
    {
        __NOP();
    }
}
