/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "w25qxx.h"

typedef enum
{
    w25qxx_WriteEnable       = 0x06,
    w25qxx_WriteDisable      = 0x04,
    w25qxx_ReadStatusReg     = 0x05,
    w25qxx_WriteStatusReg    = 0x01,
    w25qxx_ReadData          = 0x03,
    w25qxx_FastReadData      = 0x0B,
    w25qxx_FastReadDual      = 0x3B,
    w25qxx_PageProgram       = 0x02,
    w25qxx_BlockErase        = 0xD8,
    w25qxx_SectorErase       = 0x20,
    w25qxx_ChipErase         = 0xC7,
    w25qxx_PowerDown         = 0xB9,
    w25qxx_ReleasePowerDown  = 0xAB,
    w25qxx_DeviceID          = 0xAB,
    w25qxx_ManufactDeviceID  = 0x90,
    w25qxx_JedecDeviceID     = 0x9F
} __w25qxx_cmd_t;

static
void __w25qxx_write_enable(w25qxx_dev_t* p_dev)
{
    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_WriteEnable);
    p_dev->pfn_cs_high();
}

static
void __w25qxx_write_disable(w25qxx_dev_t* p_dev)
{
    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_WriteDisable);
    p_dev->pfn_cs_high();
}

static
void __w25qxx_wait_busy(w25qxx_dev_t* p_dev)
{
    while((w25qxx_read_sr(p_dev) & 0x01) == 0x01);
}

static
void __w25qxx_write_page(w25qxx_dev_t* p_dev, uint8_t *p_buffer,
                         uint32_t address, uint16_t size)
{
    uint16_t i = 0;

    __w25qxx_write_enable(p_dev);
    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_PageProgram);
    p_dev->pfn_send_and_recv_byte((uint8_t)((address) >> 16U));
    p_dev->pfn_send_and_recv_byte((uint8_t)((address) >> 8U));
    p_dev->pfn_send_and_recv_byte((uint8_t)address);
    for(i = 0; i < size; i++)
    {
    p_dev->pfn_send_and_recv_byte(p_buffer[i]);
    }
    p_dev->pfn_cs_high();
    __w25qxx_wait_busy(p_dev);
}

static
void __w25qxx_erase_sector(w25qxx_dev_t* p_dev, uint32_t address)
{
    address *= 4096;

    __w25qxx_write_enable(p_dev);
    __w25qxx_wait_busy(p_dev);
    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_SectorErase);
    p_dev->pfn_send_and_recv_byte((uint8_t)((address) >> 16));
    p_dev->pfn_send_and_recv_byte((uint8_t)((address) >> 8));
    p_dev->pfn_send_and_recv_byte((uint8_t)address);
    p_dev->pfn_cs_high();
    __w25qxx_wait_busy(p_dev);
}

static
void __w25qxx_write_nocheck(w25qxx_dev_t* p_dev, uint8_t *p_buffer,
    uint32_t address, uint16_t size)
{
    uint16_t page_remain;

    page_remain = 256 - address % 256;

    if(size <= page_remain)
    {
        page_remain = size;
    }

    while(1)
    {
        __w25qxx_write_page(p_dev, p_buffer, address, page_remain);
        if(size == page_remain)
        {
            break;
        }
        else
        {
            p_buffer += page_remain;
            address += page_remain;
            size -= page_remain;
            if(size > 256)
            {
                page_remain = 256;
            }
            else
            {
                page_remain = size;
            }
        }
    };
}

uint16_t w25qxx_read_sr(w25qxx_dev_t* p_dev)
{
    uint8_t reg_sr_val;

    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_WriteStatusReg);
    reg_sr_val = p_dev->pfn_send_and_recv_byte(0xFFU);
    p_dev->pfn_cs_high();

    return reg_sr_val;
}

void w25qxx_write_sr(w25qxx_dev_t* p_dev, uint8_t sr)
{
    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_WriteStatusReg);
    p_dev->pfn_send_and_recv_byte(sr);
    p_dev->pfn_cs_high();
}

uint16_t w25qxx_read_id(w25qxx_dev_t* p_dev)
{
    uint16_t id = 0;

    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(0x90U);
    p_dev->pfn_send_and_recv_byte(0x00U);
    p_dev->pfn_send_and_recv_byte(0x00U);
    p_dev->pfn_send_and_recv_byte(0x00U);
    id |= ((p_dev->pfn_send_and_recv_byte(0xFFU)) << 8U);
    id |= p_dev->pfn_send_and_recv_byte(0xFFU);
    p_dev->pfn_cs_high();

    return id;
}

void w25qxx_read(w25qxx_dev_t* p_dev, uint8_t* p_buffer,
                 uint32_t address, uint16_t size)
{
    uint16_t i = 0;

    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_ReadData);
    p_dev->pfn_send_and_recv_byte((uint8_t)((address) >> 16U));
    p_dev->pfn_send_and_recv_byte((uint8_t)((address) >> 8U));
    p_dev->pfn_send_and_recv_byte((uint8_t)address);
    for(i = 0; i < size; i++)
    {
        p_buffer[i] = p_dev->pfn_send_and_recv_byte(0xFFU);
    }
    p_dev->pfn_cs_high();
}

static uint8_t __g_w25qxx_buffer[4096];
void w25qxx_write(w25qxx_dev_t* p_dev, uint8_t* p_buffer,
                  uint32_t address, uint16_t size)
{
    uint32_t sec_pos;
    uint16_t sec_off;
    uint16_t sec_remain;
    uint16_t i;

    sec_pos = address / 4096;
    sec_off = address % 4096;
    sec_remain = 4096 - sec_off;

    if(size <= sec_remain)
    {
        sec_remain = size;
    }

    while(1)
    {
        w25qxx_read(p_dev, __g_w25qxx_buffer, sec_pos * 4096, 4096);
        for(i = 0; i < sec_remain; i++)
        {
            if(__g_w25qxx_buffer[sec_off+i] != 0XFF)
                break;
        }

        if(i < sec_remain)
        {
            __w25qxx_erase_sector(p_dev, sec_pos);
            for(i = 0; i < sec_remain; i++)
            {
                __g_w25qxx_buffer[i + sec_off] = p_buffer[i];
            }
            __w25qxx_write_nocheck(p_dev, __g_w25qxx_buffer, sec_pos * 4096, 4096);

        }else
        {
            __w25qxx_write_nocheck(p_dev, p_buffer, address, sec_remain);
        }

        if(size == sec_remain)
        {
            break;
        }
        else
        {
            sec_pos++;
            sec_off = 0;

            p_buffer += sec_remain;
            address += sec_remain;
            size -= sec_remain;
            if(size > 4096)
            {
                sec_remain = 4096;
            }
            else
            {
                sec_remain = size;
            }
        }
    };
}

void w25qxx_pwr_down(w25qxx_dev_t* p_dev)
{
    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_PowerDown);
    p_dev->pfn_cs_high();
}

void w25qxx_wakeup(w25qxx_dev_t* p_dev)
{
    p_dev->pfn_cs_low();
    p_dev->pfn_send_and_recv_byte(w25qxx_ReleasePowerDown);
    p_dev->pfn_cs_high();
}
