/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @example: Receive interrupt in FIFO mode
 * @Language:English
 * @encode:
 * @brief:
 * This routine demonstrates how the SCI module enables FIFO and uses interrupts to receive data.
 * After enabling the FIFO and initiating the receive interrupt,
 * two interrupts will be enabled: the TimeOut interrupt and the FIFO Level interrupt,
 * Therefore, attention should be paid when judging the interrupt state.
 * This routine only demonstrates the use of FIFO waterline to receive a fixed array of 8 bytes. (Acceptable fixed lengths: 1, 4, 8, 14)
 *
 * Program flow:
 * 		Initialize SCI enabled FIFO and interrupts, and set the FIFO water line length to 8.
 * 		In the interrupt service function, retrieve the length of data from the FIFO and read it, set the flag bit
 * 		The main loop detects the flag and will print the received data.
 *
 *	Note:
 *	1. The data that the receiving end can receive should be equal to the length of the waterline.
 *	2. When using TimeOut and FIFO level, attention should be paid to interrupt clearing.
 *		TimeOut and FIFO level interrupt clearing both read RBR,
 *		and FIFO level has a higher priority than TimeOut. Therefore,
 *		when receiving data, both flags should be checked simultaneously,
 *		otherwise it will loop into interrupts.
 *	3. When setting the SCI baud rate, it should be ensured that SCI is in an idle state.
 *		SCI can be made idle through the SCI_deset() function before setting the baud rate.
 * */

/*
 * @example: Receive interrupt in FIFO mode
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 *	ʾSCIģʹFIFOжϽݡ
 *	ʹFIFOжϺ󣬻ʹжϣʱж(TimeOut)FIFOˮж(FIFO Level)
 *	жж״̬ʱӦע⡣
 *	̽ʾFIFOˮ߽8ֽڵĹ̶顣ɽչ̶ȣ14814
 *
 *	̣
 *		ʼSCIʹFIFOжϣFIFOˮ߳Ϊ8
 *		жϷУȡFIFOݵĳȲȡλ־λ
 *		ѭ⵽־λӡյݡ
 *
 *	ע⣺
 *		1նܽյӦˮߵĳȡ
 *		2ʹTimeOutFIFO levelʱעжTimeOutFIFO levelжΪȡRBR
 *			FIFO levelűTimeOutߵȼڽʱӦͬʱж־λѭжϡ
 *		3SCIʱӦ֤SCIڿ״̬ͨSCI_reset()ʹSCIУٽвá
 * */

#include "device.h"
#include "sci_ex03_rx_irq_enableFIFO.h"

int main(void)
{
	/* initialize the Flash and System Clock, and enable the clocks for all peripherals. */
    Device_init();

    /* Enable global interrupt */
    __enable_irq();

    UART_Init(MY_SCI_BASE, 115200);

    UART_writeCharArray(MY_SCI_BASE, "SCI_ex03_rx_irq_enableFIFO start \r\n", 35);

    sci_ex03_rx_irq_enableFIFO_Example(MY_SCI_BASE);

	UART_writeCharArray(MY_SCI_BASE, "SCI_ex03_rx_irq_enableFIFO end \r\n", 33);

	for(;;);
    return 0;
}




