/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/*
 * @example: Transmit interrupt
 * @Language:English
 * @encode:
 * @brief:
 * 	This routine demonstrates the use of the SCI module's interrupt sending function.
 * 	After enabling the global interrupt (__enable_irq) and SCI send interrupt,
 * 	the SCI interrupt will immediately enter,
 * 	After clearing the SCI interrupt,
 * 	the condition for entering the send interrupt again is that the data transmission ends and
 * 		enters, that is, the THR register goes from non empty to empty.
 *
 * 	Program flow:
 * 	After initialization is complete, the main loop will wait for the relevant flag position bit.
 * 	When the flag position bit is reached, data will be sent and then the main loop will exit.
 * 	When sending an air interrupt, the SCI interrupt will be turned off first,
 * 	followed by the number of times the interrupt is entered, and then set as a flag.
 *
 *	Note: When configuring the SCI baud rate, ensure the SCI is in an idle state.
 *	You may call the SCI_reset function to reset the SCI before setting the baud rate.
 * */

/*
 * @example: Transmit interrupt
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 *	ʾSCIģ鷢жϹܵʹá
 *	ʹȫж(__enable_irq)SCIжϺ󣬻SCIжϣ
 *	SCIжϺٴν뷢жϵǷݽ룬THRĴɷǿյա
 *
 *	̣
 *		ʼɺѭлȴر־λλ־λλʱᷢݣ˳ѭ
 *		ڷͿжУȽSCIжϹرգͽжϵĴΪ־λ
 *
 *	ע⣺SCIʱӦ֤SCIڿ״̬ԵSCI_resetʹSCIλٽвá
 * */

#include "device.h"
#include "sci_ex06_rx_irq.h"


int main(void)
{
	/* initialize the Flash and System Clock, and enable the clocks for all peripherals. */
    Device_init();

    /* Enable global interrupt */
    __enable_irq();

    UART_Init(MY_SCI_BASE,115200);

    UART_writeCharArray(MY_SCI_BASE, "SCI_ex03_tx_rx_irq start \r\n", 27);

    sci_ex06_rx_irq_Example(MY_SCI_BASE);

    UART_writeCharArray(MY_SCI_BASE, "SCI_ex03_tx_rx_irq end \r\n", 25);

    for (;;);
    return 0;
}
