/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define LED_GPIO_PIN    GPIO_PIN_LED1
#define LED_GPIO_CFG    GPIO_CFG_LED1

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
volatile uint32_t dspTicks = 0;


/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */
void initSCIAEchoback(void);
void transmitSCIAChar(uint8_t a);
void transmitSCIAMessage(unsigned char* msg);
void initSCIAFIFO(void);
/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief  CPUTimer1 ISR
 */
#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER0_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    dspTicks += 1;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void TIMER0_IRQHandler(void)
{
	CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
	dspTicks += 1;
}
#endif


/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
	uint32_t loopCounter = 0;
	unsigned char *msg;
	uint8_t ReceivedChar;

	Device_init();

	GPIO_setPinConfig(SCIA_TX_PIN);
	GPIO_setPinConfig(SCIA_RX_PIN);

	GPIO_setPadConfig(28, GPIO_PIN_TYPE_PULLUP);
	GPIO_setPadConfig(29, GPIO_PIN_TYPE_PULLUP);


	initSCIAFIFO();
	initSCIAEchoback();

    msg = "\r\n\n\nHello World!\0";
    transmitSCIAMessage(msg);

    msg = "\r\nYou will enter a character, and the DSP will echo it back! \n\0";
    transmitSCIAMessage(msg);

    for(;;)
    {
        msg = "\r\nEnter a character: \0";
        transmitSCIAMessage(msg);


        /* Wait for character */
        while(SciaRegs.SCIFFRX.bit.RXFFST == 0)
        {

        } /* wait for empty state */


        /* Get character */
        ReceivedChar = SciaRegs.SCIRXBUF.all;


        /* Echo character back */

        msg = "  You sent: \0";
        transmitSCIAMessage(msg);
        transmitSCIAChar(ReceivedChar);

        loopCounter++;
    }
}

void initSCIAFIFO(void)
{
    SciaRegs.SCIFFTX.all = 0xE040;
    SciaRegs.SCIFFRX.all = 0x2044;
    SciaRegs.SCIFFCT.all = 0x0;
}

void initSCIAEchoback(void)
{

    /* Note: Clocks were turned on to the SCIA peripheral
     in the InitSysCtrl() function */

    SciaRegs.SCICCR.all = 0x0007;           /* 1 stop bit,  No loopback */
                                            /* No parity, 8 char bits, */
                                            /* async mode, idle-line protocol */
    SciaRegs.SCICTL1.all = 0x0003;          /* enable TX, RX, internal SCICLK, */
                                            /* Disable RX ERR, SLEEP, TXWAKE */
    SciaRegs.SCICTL2.all = 0x0003;
    SciaRegs.SCICTL2.bit.TXINTENA = 1;
    SciaRegs.SCICTL2.bit.RXBKINTENA = 1;

#if DEVICE_APBCLK_FREQ == 90*1000*1000
    /* SCIA at 115200 baud
     * baud = APBCLK/16/(((HBAUD+LBAUD+DLF/64)+1)/2)
     * @ APBCLK = 360M/4 = 60MHz  HBAUD = 0x00 and LBAUD = 0x5FDLF = 0x35
     **/
    SciaRegs.SCIHBAUD.all = 0x0000;
    SciaRegs.SCILBAUD.all = 0x005F;
    SciaRegs.SCIDLF.all = 0x35;
#elif DEVICE_APBCLK_FREQ == 60*1000*1000
    /* SCIA at 115200 baud
     * baud = APBCLK/16/(((HBAUD+LBAUD+DLF/64)+1)/2)
     * @ APBCLK = 240M/4 = 60MHz  HBAUD = 0x00 and LBAUD = 0x3FDLF = 0x23
     *  */
    SciaRegs.SCIHBAUD.all = 0x0000;
    SciaRegs.SCILBAUD.all = 0x003F;
    SciaRegs.SCIDLF.all = 0x23;
#elif DEVICE_APBCLK_FREQ == 40*1000*1000
    /* SCIA at 115200 baud
     * baud = APBCLK/16/(((HBAUD+LBAUD+DLF/64)+1)/2)
     * @ APBCLK = 160M/4 = 40MHz  HBAUD = 0x00 and LBAUD = 0xc1DLF = 0x2a
     *  */
    SciaRegs.SCIHBAUD.all = 0x0000;
    SciaRegs.SCILBAUD.all = 0x0029;
    SciaRegs.SCIDLF.all = 0x45;
#endif

    SciaRegs.SCICTL1.all = 0x0023;         /* Relinquish SCI from Reset */
}

void transmitSCIAChar(uint8_t a)
{
    while (SciaRegs.SCIFFTX.bit.TXFFST != 0)
    {

    }
    SciaRegs.SCITXBUF.all = a;
}

void transmitSCIAMessage(unsigned char * msg)
{
    int i;
    i = 0;
    while(msg[i] != '\0')
    {
        transmitSCIAChar(msg[i]);
        i++;
    }
}

