/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define FORCE_NMI_TRIGGER_TYPE 	SYSCTL_NMI_CLOCKFAIL /* Define the NMI trigger type*/
#define Delay_time 				200					 /* Increase delay to trigger NMI reset */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

volatile uint16_t nmi_count                = 0; // NMI count variable
volatile uint16_t NMI_WatchdogCounterValue = 0; // NMI watchdog counter value

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* NMI interrupt handler function */
void NMI_Handler(void)
{
	/* Delay nmi_count % 200 microseconds, adding wait time to trigger NMI_WDG reset (greater than 218us) */
    DELAY_US(nmi_count % Delay_time);
    if (SysCtl_isNMIFlagSet(FORCE_NMI_TRIGGER_TYPE)) {
        nmi_count++; /* If NMI is triggered, increment nmi_count */
    }
    /* Get the current watchdog counter value */
    NMI_WatchdogCounterValue = SysCtl_getNMIWatchdogCounter();
    /* Clear all NMI flags */
    SysCtl_clearAllNMIFlags();
}

/**
 * @brief  Initialize the NMI watchdog
 */
void NMI_Wdg_init(void)
{
    /* Set NMI watchdog period to 65535(MAX)
     * The counting frequency is the same as DEVICE_SYSCLK_FREQ
     * */
    SysCtl_setNMIWatchdogPeriod(65535);

    /* Enable NMI reset request */
    SysCtl_enableNMIResetRequest();

    /* Register the NMI interrupt handler */
    Exception_Register_EXC(NMI_EXCn, (unsigned long)NMI_Handler);

    /* Enable global NMI interrupts */
    SysCtl_enableNMIGlobalInterrupt();
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/**
 * @brief  Main function
 * @param  None
 * @return None
 */
int main(void)
{
    uint32_t cycles = __get_rv_cycle(); /* Get the current CPU cycle count */

    __disable_irq(); /* Disable global interrupts */

    Device_init(); /* Initialize the device */

    /* Initialize the UART */
    UartPrint_init(LOG_SCI_BASE, 115200);

    log_info("NMI Watchdog Reset Example Started!\r\n");
    log_info("Core Frequency: %d MHz\r\n", DEVICE_SYSCLK_FREQ / 1000 / 1000);
    log_info("Main Function Location: 0x%08X, NMI Index Location: 0x%08X\r\n", (uint32_t)main, (uint32_t)&nmi_count);

#if IS_GS32F00xx(0x12)
    log_info("Reset Record: %d, Cycles Count: %d\r\n", SysCtl_getCoreRstRecord(), cycles);
#elif IS_GS32F3xx(0x22)
    log_info("Reset Record: %d, Cycles Count: %d\r\n", SysCtl_getDspCpu1RstRecord(), cycles);
#endif

    /* Delay for 1 millisecond */
    DELAY_US(1000); 

    /* Initialize NMI watchdog */
    NMI_Wdg_init();

    while (1) {
        /* Delay for 100ms */
        DELAY_US(100*1000); 
        /* Force trigger the NMI */
        SysCtl_forceNMIFlags(FORCE_NMI_TRIGGER_TYPE);
        /* Waiting for register to be successfully written */
        __DSB();
        /* Print the related information about NMI triggering */
        log_info("NMI Triggered! nmi_count = %d, Watchdog Counter = %d\r\n", nmi_count, NMI_WatchdogCounterValue);
    }

    /* This point will never be reached */
    for (;;) {
    }

    return 0;
}
