/*
 * device_init.h
 *
 *  Created on: 2024
 *      Author: hpec
 */

#ifndef DEVICE_INIT_H_
#define DEVICE_INIT_H_

#if defined(GS32F00xx)
  #include "GS32F00xx.h"
#elif defined(GS32F3xx)
  #include "GS32F3xx.h"
#else
  #error Device type must be defined.
#endif
#include "driverlib.h"

#include "regs/regs_adc.h"

#define INT_ADCA_IRQn   			INT_ADCA2
#define ADC_INT_NUMBER				ADC_INT_NUMBER2
#define ECLIC_INT_ADCA_IRQn         50//SOC_INT50_IRQn  = 50

/* DAC loopback to ADC via 1-software mode, 2-internal DAC, 3-external I2C DAC */
#define DAC_ADC_LOOPBACK_MODE     2

//*****************************************************************************
//
// Adjust the modeled current so that it can be output by DAC and read from
// ADC correctly as the modeled current is between 1.0 and -1.0
//
//*****************************************************************************
#define Ia_ADJUST_FACTOR 1000
#define Ia_ADJUST_BASELINE 2700

#define ADJUST_Ia_OUTPUT(x) ( ((x) * Ia_ADJUST_FACTOR) + Ia_ADJUST_BASELINE )
#define ADJUST_Ia_INPUT(x)  ( (x - Ia_ADJUST_BASELINE) / Ia_ADJUST_FACTOR )

#define Ib_ADJUST_FACTOR 500
#define Ib_ADJUST_BASELINE 1500

#define ADJUST_Ib_OUTPUT(x) ( ((x) * Ib_ADJUST_FACTOR) + Ib_ADJUST_BASELINE )
#define ADJUST_Ib_INPUT(x)  ( (x - Ib_ADJUST_BASELINE) / Ib_ADJUST_FACTOR )

extern void Device_setup(void);
extern __interrupt void aci_ctrlLoop(void);
extern void disablePWM(void);
//
// Bitfield datastructure definition
//
#define AdcaResult (*(struct ADC_RESULT_REGS *)ADCARESULT_BASE)

extern uint16_t adcValIa, adcValIb;

//
// ADC read functions
//
static inline uint16_t ADC_getIa()
{
#if  DAC_ADC_LOOPBACK_MODE==1
	return adcValIa;
#else
	return ADC_readResult(ADCARESULT_BASE, ADC_SOC_NUMBER0); //AdcaResult.ADCRESULT0;//
#endif

}


static inline uint16_t ADC_getIb()
{
#if  DAC_ADC_LOOPBACK_MODE==1
	return adcValIb;
#else
	return ADC_readResult(ADCARESULT_BASE, ADC_SOC_NUMBER1);  //AdcaResult.ADCRESULT1;
#endif

}

//
// Interrupt handling functions
//
static inline void ADCInt_ack()
{
    //
    // Enable more interrupts from this timer
    //
    // *(uint16_t *)(ADCA_BASE+ADC_O_INTFLG) |= ADC_INTFLG_ADCINT2;

    //
    // Acknowledge interrupt to recieve more interrupts from PIE group 1
    //
    ADC_clearInterruptStatus(ADCA_BASE, ADC_INT_NUMBER);
    //Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP1);
}

static inline void ADCInt_enable()
{
    //
    // Enable ADC interrupt
    //
    ADC_enableInterrupt(ADCA_BASE, ADC_INT_NUMBER);

	Interrupt_register(INT_ADCA2, aci_ctrlLoop);
	Interrupt_enable(INT_ADCA2);

}

static inline void ADCInt_disable()
{
    //
    // Disable PWMs by disabling clocks
    //
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_EPWM);
#elif IS_GS32F00xx(0x30)
    SysCtl_disablePeripheral_TBCLKSYNC();
#else
//    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
#endif

    //
    // Disable ADC interrupt
    //
	ADC_disableInterrupt(ADCA_BASE, ADC_INT_NUMBER);
    Interrupt_disable(INT_ADCA_IRQn);
}

//*****************************************************************************
//
// PWM functions
//
//*****************************************************************************
__STATIC_INLINE void PWM_setUa(uint16_t Ua)
{
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, Ua);
}

 __STATIC_INLINE void PWM_setUb(uint16_t Ub)
{
    EPWM_setCounterCompareValue(EPWM2_BASE, EPWM_COUNTER_COMPARE_A, Ub);
}

 __STATIC_INLINE void PWM_setUc(uint16_t Uc)
{
    EPWM_setCounterCompareValue(EPWM3_BASE, EPWM_COUNTER_COMPARE_A, Uc);
}

//*****************************************************************************
//
// DAC functions
//
//*****************************************************************************
 __STATIC_INLINE void DAC_setIa(uint16_t value)
{
#if  DAC_ADC_LOOPBACK_MODE==1
	 adcValIa = value;
#elif DAC_ADC_LOOPBACK_MODE==2
    DAC_setShadowValue(DACA_BASE, value);
#elif DAC_ADC_LOOPBACK_MODE==3
    //DAC_setShadowValue(DACA_BASE, value);
#else
#endif
}

 __STATIC_INLINE void DAC_setIb(uint16_t value)
{
#if DAC_ADC_LOOPBACK_MODE==1
	 adcValIb = value;
#elif DAC_ADC_LOOPBACK_MODE==2
    DAC_setShadowValue(DACB_BASE, value);
#elif DAC_ADC_LOOPBACK_MODE==3
    //DAC_setShadowValue(DACB_BASE, value);
#else
#endif
}


#endif /* DEVICE_INIT_H_ */
//
// End of file
//
