/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    dma_ex11_sci_autoback_with_timeout_board.h
*   @brief   
*
*/

#ifndef DMA_EX11_SCI_AUTOBACK_WITH_TIMEOUT_BOARD_H_
#define DMA_EX11_SCI_AUTOBACK_WITH_TIMEOUT_BOARD_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
//
// Defines for SCI
//
#define SCI_BASE            SCIA_BASE
#define SCI_INT             INT_SCIA
#define DMAMUX_ReqId_TX 	DMAMUX_ReqId_scia_tx
#define DMAMUX_ReqId_RX     DMAMUX_ReqId_scia_rx
//
// Defines for DMA
//
#if __arm__
#define DMA_INT_CH4_IRQn        (INT_DMA_CH4)
#else
#define DMA_INT_CH4_IRQn        (INT_DMA1_CH4)
#endif
//
// Other MACRO 
//
#define COM_TIMEOUT  (3000U)
//*****************************************************************************
//
// PinMux Configurations
//
//*****************************************************************************

#define MAX_COM_BUF_LEN (2048)



/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef struct {
    uint32_t dlog_load_start;
    uint32_t dlog_load_stop;
    uint32_t dlog_loads;    
    uint32_t dlog_load_max;
    uint32_t dlog_load_cnt;
    uint32_t dlog_load_aver;
}DLOG_LOAD_COMP;

typedef struct {
__IO DLOG_LOAD_COMP    timer0_isr;
__IO DLOG_LOAD_COMP    dma0_isr;
__IO DLOG_LOAD_COMP    adc_cvt;
__IO DLOG_LOAD_COMP    adc_dmatfr;
}DLOAD_STRUCT;

typedef enum
{
    UART_RX = 1U,
    UART_TX = 2U,
}UartState_Typedef;

typedef struct
{
    uint8_t Error;
    uint8_t State;
    uint16_t TxLen;
    uint16_t pTx;
    uint16_t RxLen;
    uint16_t pRx;
    uint16_t RxDelay;
    uint8_t Buf[MAX_COM_BUF_LEN];
}ComTxRxType;
/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */


/**
 * \brief   "get flash cache Statistic Hit"
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  Counts the number of cache hits during cache look up.
 */
CCMFUNC_T static inline uint32_t cFlash_Cache_getStatisticHit(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    return HWREG(cacheBase + FLASH_CACHE_O_CSHR);
}

CCMFUNC_T static inline void cFlash_Cache_clearStatisticHit(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CSHR) = 0;
}

/**
 * \brief   "get flash cache Statistic Miss"
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  Counts the number of cache hits during cache look up.
 */
CCMFUNC_T static inline uint32_t cFlash_Cache_getStatisticMiss(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    return HWREG(cacheBase + FLASH_CACHE_O_CSMR);
}

CCMFUNC_T static inline void cFlash_Cache_clearStatisticMiss(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CSMR) = 0;
}

//*****************************************************************************
//
// Board Configurations
//
//*****************************************************************************
void Board_init();
void PinMux_init();
void DMA_CHN_Modeconfig(void);
CCMFUNC_T void sDMA_configChannel(uint32_t base, const DMA_ConfigParams *transfParams);
void InitCpuTimer(uint32_t base, uint32_t period);
CCMFUNC_T void sTimeBaseIsr(void);
void SCIx_INTR_IRQHandler(void);
void DMA_CH4_IRQHandler(void);
void Uart_Init(void);


#ifdef __cplusplus
}
#endif


#endif /* DMA_EX11_SCI_AUTOBACK_WITH_TIMEOUT_BOARD_H_ */
