/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @file main.c
 * This flash API example demonstrates:
 * 1) how to program the user option byte DLL clock divider;
 * 2) erase and program main flash area and measure the erase and program time.
 *    Erase one sector = 2.64mS (GS32F0039)
 *    Program 4KB data = 49.6mS
 *    memcmp 4KB data = 0.038mS
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"
#include "flash_programming_gs32.h"
#include "gs32_flashAPI_lib.h"
#include "F021.h"
#include "string.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define LED_GPIO_PIN    GPIO_PIN_LED1
#define LED_GPIO_CFG    GPIO_CFG_LED1

#define FLS_WRITE_DATA_ADDR  0x08010000
/* DATA_LENGTH is number of data in 32bit words */
#define PGM_DATA_LENGTH      (4096/4)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
volatile uint32_t dspTicks = 0;

__attribute__((section(".ilmbss")))
uint32_t fls_wdata[PGM_DATA_LENGTH];

volatile uint32_t erase_start=0;

volatile uint32_t dlog_fls_time[6];

volatile uint32_t dlog_pgm_progress;
volatile uint32_t dlog_erase_progress;

Fapi_FlashStatusWordType fls_status_0;

volatile uint32_t dlog_blank_rslt;
volatile uint32_t dlog_erase_rslt;
volatile uint32_t dlog_pgm_rslt;
volatile uint32_t dlog_verify_rslt;
volatile uint32_t dlog_data_size;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief  CPUTimer1 ISR
 */
#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER1_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    dspTicks += 1;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void TIMER1_IRQHandler(void)
{
	CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
	dspTicks += 1;
}
#endif

/*
 * @brief  CPUTimer1 initialization
 */
int32_t Timer1_init(uint32_t period)
{
	/* Initialize CPUTimer with interrupt enable */
	CPUTimer_init(CPUTIMER1_BASE, period);

	Interrupt_register(INT_TIMER1, TIMER1_IRQHandler);
	Interrupt_enable(INT_TIMER1);

	return 0;
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
    uint32_t oldTicks;
    uint32_t tickCnt;
    uint32_t ret;

    /*
     * User Option Byte6 bit0 is for DLL clock divider
     * set value 0 means DLL=PLL;
     */
//    if((Fapi_getUserOptByte(6)&0x01) == 1)
//    {
//    	/* set Option Byte6 bit0 to value 0 */
//    	GS32Fapi_OPT_SetClkRateSel(0);
//    	/* delay 1uS */
//    	DEVICE_DELAY_US(1);
//    	/* trigger a software reset */
//    	SysTimer_SoftwareReset();
//    }

    /*
     * User Option Byte6 bit0 is for DLL clock divider:
     * value 0 = div by 1, DLL = PLL = 480MHz
     * value 1 = div by 2, DLL = PLL/2 = 240MHz
     */
//    if((Fapi_getUserOptByte(6)&0x01) == 0)
//    {
//    	/* set Option Byte6 bit0 to value 1 */
//    	GS32Fapi_OPT_SetClkRateSel(1);
//    	/* delay 1uS */
//    	DEVICE_DELAY_US(1);
//    	/* trigger a software reset */
//    	SysTimer_SoftwareReset();
//    }

    Device_init();

    GPIO_setPinConfig(LED_GPIO_CFG);     //gpio
    GPIO_enableWritePin(LED_GPIO_PIN);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP template!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    log_info("Code @ 0x%08X, Data @ 0x%08X.\r\n", (uint32_t)main, (uint32_t)&oldTicks);

    //Timer1_init(DEVICE_APBCLK_FREQ/100);    //10ms

    //__enable_irq();

    oldTicks = dspTicks;
    dlog_data_size = PGM_DATA_LENGTH*4; //data size in bytes

    /*
     * FlashAPI initialization
     */
    ret = GS32Fapi_Init(DEVICE_SYSCLK_FREQ);

    /* prepare the program data */
    for(uint32_t i=0; i<sizeof(fls_wdata)/sizeof(fls_wdata[0]); i++)
    	fls_wdata[i] = (i+1);

    /*
     * check blank and erase of not blank
     */
    if(Fapi_Status_Success != Fapi_doBlankCheck(0, (uint32 *)FLS_WRITE_DATA_ADDR, sizeof(fls_wdata)/4, &fls_status_0))
    {
    	CPU_clearCycleCnt(); //clear timer value
    	/* erase region size should be multiple of sector size 8KB divided by 4byte */
    	dlog_erase_rslt = GS32Fapi_EraseRegion(FLS_WRITE_DATA_ADDR, 8*1024/4);
    	dlog_fls_time[0] = CPU_getCycleCnt();
    }
    /*
     * Before reading the target Flash sector,
     * invalidate the DCache to avoid reading back unexpected data in DCache.
     */
    MInvalDCacheLines(FLS_WRITE_DATA_ADDR, 8*1024/32);
    //MInvalDCacheLine(FLS_WRITE_DATA_ADDR);

    if(Fapi_Status_Success != dlog_erase_rslt)
    	while(erase_start != 3);

    /*
     * Program data
     */
    dlog_pgm_progress = 0;
    CPU_clearCycleCnt(); //clear timer value
    /* Program length value should be multiple of 4, namely 4 words or 16 bytes. */
    dlog_pgm_rslt = GS32Fapi_Program(FLS_WRITE_DATA_ADDR, (uint32 )fls_wdata, sizeof(fls_wdata)/4);
    dlog_fls_time[1] = CPU_getCycleCnt();

    /*
     * Compare the source data with the programmed target Flash region.
     */
    CPU_clearCycleCnt(); //clear timer value
    /* memcmp parameter size should be number of size. */
    dlog_verify_rslt = memcmp((void *)FLS_WRITE_DATA_ADDR, fls_wdata, sizeof(fls_wdata));
    dlog_fls_time[2] = CPU_getCycleCnt();

    /*
     * Print Flash program and erase time.
     */
    printf("Flash erase one sector time: %d us.\r\n", dlog_fls_time[0]/240);
    printf("Program data size is %d bytes.\r\n", dlog_data_size);
    printf("Flash program data time: %d us.\r\n", dlog_fls_time[1]/240);
    printf("Flash compare 4KB data time: %d us.\r\n", dlog_fls_time[2]/240);
    while(erase_start==0);

    /*
     * Erase Flash
     */
    dlog_erase_progress = 0;
    CPU_clearCycleCnt(); //clear timer value
    for(uint32_t i=0; i<PGM_DATA_LENGTH; i+=1024*2){
        ret = GS32Fapi_EraseSector(FLS_WRITE_DATA_ADDR+i*4);
        dlog_erase_progress++;
    }
    dlog_fls_time[3] = CPU_getCycleCnt();

    while (1) {
        if (dspTicks != oldTicks) {
            oldTicks = dspTicks;
            tickCnt += 1;

            if ((tickCnt % 100) == 0) {
                GPIO_togglePin(LED_GPIO_PIN);
                log_info("ticks: %d\r\n", dspTicks);
            }
        }
    }

    return 0;
}

