#include <stdlib.h>
#include <string.h>

#include "device.h"
#include "driverlib.h"

#define EMBEDDED_CLI_IMPL
#include "embedded_cli.h"

#include "printf.h"
#include "board_cfg.h"

#define LED_Pin  (50)

uint8_t rxData;
EmbeddedCli *cli = NULL;

UserOptionByteType *userOption = (UserOptionByteType *)(FLASH_INTERFACE_EFC_BASE + SET_USER_OPT0);

void SCIA_IRQHandler(void)
{
    uint32_t irq_status = SCI_getInterruptStatus(SCIA_BASE);

    if ((irq_status & SCI_RX_DATA_AVAILABLE) || (irq_status & SCI_CHARACTER_TIMEOUT)) {
        while (SCI_rxDataReady(SCIA_BASE)) {
            embeddedCliReceiveChar(cli, SCI_readData(SCIA_BASE));
        }
    } else {
        SCI_clearInterruptStatus(SCIA_BASE);
    }
}

static void cli_writeChar(EmbeddedCli *cli, char c)
{
	UART_writeCharArray(LOG_SCI_BASE, &c, 1);
}

__attribute__ ((section (".RamFunc"))) static void Reboot(EmbeddedCli *cli, char *args, void *context)
{
    printf("\r\n");
    DEVICE_DELAY_US(100000);

    SysCtl_resetDevice();
}

static __attribute__ ((section (".RamFunc"))) int32_t Flash_optUnprotect(void)
{
    uint32_t blkBase = FLASH_INTERFACE_EFC_BASE;

    //disable write protect
    HWREG(blkBase + FLASH_O_UNLOCK1) = FLASH_UNLOCK_CODE1;
    HWREG(blkBase + FLASH_O_UNLOCK2) = FLASH_UNLOCK_CODE2;
    HWREG(blkBase + FLASH_O_UNLOCK1) = FLASH_UNLOCK_CODE3;

    HWREG(blkBase + FLASH_O_WRITE_PROTECT) &= (~FLASH_CONTROL_WRITE_PROTECT);

    HWREG(blkBase + FLASH_O_STATUS) = FLASH_STATUS_ALL;

    //unlock opt
    HWREG(blkBase + OPT_UNLOCK) = 0x08192A3B;
    HWREG(blkBase + OPT_UNLOCK) = 0x4C5D6E7F;

    return 0;
}

static __attribute__ ((section (".RamFunc"))) int32_t Flash_optLaunch(void)
{
    HWREG(FLASH_INTERFACE_EFC_BASE + OTP_MOD_STATUS) = 1<<1;    //clear error
    HWREG(FLASH_INTERFACE_EFC_BASE + OTP_MOD_CTRL) = 1<<0;      //launch

    while (HWREG(FLASH_INTERFACE_EFC_BASE + OTP_MOD_STATUS) & 0x03);

    if (HWREG(FLASH_INTERFACE_EFC_BASE + OTP_MOD_STATUS) & (1<<1)) {
        printf("launch opt failed!\r\n");
    } else {
        printf("finished!\r\n");
    }

    return 0;
}

static void RDPSetting(EmbeddedCli *cli, char *args, void *context)
{
    if (embeddedCliGetTokenCount(args) < 1) {
        printf("setup RDP.\r\n");
        printf("usage: RDP [0/1/level2], level2 is unrecoverable\r\n");
        printf("example: RDP 0\r\n");
        printf("example: RDP 1\r\n");
        printf("example: RDP level2\r\n");
        return;
    }

    if (strcmp(embeddedCliGetToken(args, 1), "0") == 0) {
        Flash_optUnprotect();
        HWREG(FLASH_INTERFACE_EFC_BASE + SET_RDP) = RDP_LEVEL_0;
        printf("RDP, level0 no read protect.\r\n");
    } else if (strcmp(embeddedCliGetToken(args, 1), "1") == 0) {
        Flash_optUnprotect();
        HWREG(FLASH_INTERFACE_EFC_BASE + SET_RDP) = 0x00;
        printf("RDP, level1 mem read protect.\r\n");
    } else if (strcmp(embeddedCliGetToken(args, 1), "level2") == 0) {
        Flash_optUnprotect();
        HWREG(FLASH_INTERFACE_EFC_BASE + SET_RDP) = RDP_LEVEL_2;
        printf("RDP, level2 chip read protect.\r\n");
    } else {
        printf("invalid arg: %s\r\n", embeddedCliGetToken(args, 1));
        return;
    }

    //launch
    Flash_optLaunch();
}

static void BootLockSetting(EmbeddedCli *cli, char *args, void *context)
{
    if (embeddedCliGetTokenCount(args) < 1) {
        printf("setup bootLock.\r\n");
        printf("usage: bootLock [0/1]\r\n");
        printf("example: bootLock 0\r\n");
        printf("example: bootLock 1\r\n");
        return;
    }

    if (strcmp(embeddedCliGetToken(args, 1), "0") == 0) {
        Flash_optUnprotect();
        HWREG(FLASH_INTERFACE_EFC_BASE + SET_BOOT_LOCK) = 0;
        printf("bootLock, boot based on pad/optionbyte.\r\n");
    } else if (strcmp(embeddedCliGetToken(args, 1), "1") == 0) {
        Flash_optUnprotect();
        HWREG(FLASH_INTERFACE_EFC_BASE + SET_BOOT_LOCK) = 1;
        printf("bootLock, boot forced from main flash.\r\n");
    } else {
        printf("invalid arg: %s\r\n", embeddedCliGetToken(args, 1));
        return;
    }

    Flash_optLaunch();
}

static void SecDebugEnSetting(EmbeddedCli *cli, char *args, void *context)
{
    if (embeddedCliGetTokenCount(args) < 1) {
        printf("setup secDebugEn.\r\n");
        printf("usage: secDebugEn [0/1]\r\n");
        printf("example: secDebugEn 0\r\n");
        printf("example: secDebugEn 1\r\n");
        return;
    }

    if (strcmp(embeddedCliGetToken(args, 1), "0") == 0) {
        Flash_optUnprotect();
        HWREG(FLASH_INTERFACE_EFC_BASE + SET_SEC_DEBUG_EN) = 0;
        printf("secDebugEn, security debug is disabled.\r\n");
    } else if (strcmp(embeddedCliGetToken(args, 1), "1") == 0) {
        Flash_optUnprotect();
        HWREG(FLASH_INTERFACE_EFC_BASE + SET_SEC_DEBUG_EN) = 1;
        printf("secDebugEn, security debug is enabled.\r\n");
    } else {
        printf("invalid arg: %s\r\n", embeddedCliGetToken(args, 1));
        return;
    }

    Flash_optLaunch();
}

static void JtagAuthEnSetting(EmbeddedCli *cli, char *args, void *context)
{
    if (embeddedCliGetTokenCount(args) < 1) {
        printf("change jtag auth en.\r\n");
        printf("usage: authEn [on/off]\r\n");
        printf("example: authEn on\r\n");
        printf("example: authEn off\r\n");
        printf("auth val: 0x%02X\r\n", HWREG(AHB_COMM_PARA_BASE + 0x100));
        return;
    }

    if (strcmp(embeddedCliGetToken(args, 1), "on") == 0) {
        HWREG(AHB_COMM_PARA_BASE + 0x100) = (1<<8) | (1<<4) | (1<<0);
        HWREG(AHB_COMM_PARA_BASE + 0x100) = (1<<8);
        printf("jtag auth enabled, %X\r\n", HWREG(AHB_COMM_PARA_BASE + 0x100));
    } else if (strcmp(embeddedCliGetToken(args, 1), "off") == 0) {
        HWREG(AHB_COMM_PARA_BASE + 0x100) = (1<<4) | (1<<0);
        HWREG(AHB_COMM_PARA_BASE + 0x100) = 0;
        printf("jtag auth disabled, %X\r\n", HWREG(AHB_COMM_PARA_BASE + 0x100));
    } else {
        printf("invalid arg: %s\r\n", embeddedCliGetToken(args, 1));
        return;
    }
}

void ShowSettings(EmbeddedCli *cli, char *args, void *context)
{
    printf("\r\n");
    printf("       RDP : 0x%02X\r\n", HWREG(FLASH_INTERFACE_EFC_BASE + SET_RDP));
    printf("  BootLock : %d\r\n", HWREG(FLASH_INTERFACE_EFC_BASE + SET_BOOT_LOCK));
    //printf("  SecDebug : %d\r\n", HWREG(FLASH_INTERFACE_EFC_BASE + SET_SEC_DEBUG_EN));
    //printf("JtagAuthor : 0x%02X\r\n", HWREG(AHB_COMM_PARA_BASE + 0x100));
}

void ShowData(void *buf, uint32_t baseAddr, uint32_t width, uint32_t items)
{
    for (uint32_t i=0; i<items; i+=1) {
        if (i%8==0) {
            printf("%08X: ", baseAddr + i*width);
        }

        switch(width) {
            case 1: printf("%02X ", ((uint8_t *)buf)[i]); break;
            case 2: printf("%04X ", ((uint16_t *)buf)[i]); break;
            case 4: printf("%08X ", ((uint32_t *)buf)[i]); break;
        }

        if (i%8==7) {
            printf("\r\n");
        }
    }
}

REGISTER_CLI_COMMAND("show", "show settings", false, NULL, ShowSettings);
REGISTER_CLI_COMMAND("reboot", "reboot", false, NULL, Reboot);
REGISTER_CLI_COMMAND("RDP", "change RDP", TRUE, NULL, RDPSetting);
REGISTER_CLI_COMMAND("bootLock", "change bootLock", TRUE, NULL, BootLockSetting);
//REGISTER_CLI_COMMAND("secDebugEn", "change secDebugEn", TRUE, NULL, SecDebugEnSetting);
//REGISTER_CLI_COMMAND("authEn", "change jtag authEn", TRUE, NULL, JtagAuthEnSetting);


void onCommand(EmbeddedCli *embeddedCli, CliCommand *command)
{
	char str[16];

	printf("Received command: %s\r\n",command->name);

    embeddedCliTokenizeArgs(command->args);
    for (int i = 1; i <= embeddedCliGetTokenCount(command->args); ++i) {
    	printf("arg%d: %s\r\n", i, embeddedCliGetToken(command->args, i));
	}
}

void embedded_cli_init(void)
{
    cli = embeddedCliNewDefault();

    cli->onCommand = onCommand;
    cli->writeChar = cli_writeChar;

    CliCommandBinding *cli_cmds = (CliCommandBinding *)__cli_cmds_table_start__;

    for (;(uint32_t)cli_cmds < (uint32_t)__cli_cmds_table_end__; cli_cmds += 1) {
        if (!embeddedCliAddBinding(cli, *cli_cmds)) {
            printf("fail to binding cmd: %s, out of memory\r\n", cli_cmds->name);
        }
    }


    Interrupt_register(INT_SCIA, SCIA_IRQHandler);
    Interrupt_enable(INT_SCIA);
    SCI_enableInterrupt(SCIA_BASE, SCI_RX_DATA_AVAILABLE_INT | SCI_LINE_STATUS_INT);
}
