#include "Kernel.h"
#include "riscv_encoding.h"

//#pragma DATA_SECTION(knlTaskStack,"kernelStack");
knlStack_t knlTaskStack[KNL_TOTAL_STK_SIZE]; /* Stack of all tasks*/

//******************************************************************************
//*                 the constant definition(only use in local file)
//******************************************************************************
const kernel_t knlUnMapTbl[16] =
{
    KNL_TASK_LOWEST_PRIO, 0, 1, 0,
    2, 0, 1, 0,
    3, 0, 1, 0,
    2, 0, 1, 0
};

//******************************************************************************
//*                 the global variable define 
//******************************************************************************
knl_tcb_t knlTCBTable[KNL_TASK_NUM]; /* Table of TCB*/
knl_stack_t knlStackTable[KNL_TASK_NUM]; /* Table of Stack*/

//******************************************************************************
//*                 the local variable define 
//******************************************************************************
static uint16_t knlRdyMap;

kernel_t knlNextTaskPrio;     /* Priority of highest priority in ready table  */
kernel_t knlCurrentTaskPrio;  /* Priority of current task              */
knl_tcb_t *knlNextTaskTCB;    /* Pointer to highest priority TCB in ready table*/
knl_tcb_t *knlCurrentTaskTCB; /* Pointer to TCB currently running */
knlStack_t *knlStackPtr;

DSPstatic int32_t kernelTick = 0;
#if PRE_EMPTIVE_EN == 0
DSPstatic bool_t arm = false;
#endif
//******************************************************************************
//*                 the define for the os 
//******************************************************************************
//the Macro define for some operation
#define knlTaskReady(prio)            {knlRdyMap |= BIT(prio);}   //set task ready
#define knlTaskSuspend(prio)        {knlRdyMap &= ~BIT(prio);} //suspend a task
#ifndef timerTickHook
#define timerTickHook()
#else
extern void timerTickHook(void);
#endif
//******************************************************************************
//* Local function Prototypes
//******************************************************************************
DSPstatic kernel_t knlFindNextTaskPrio(void);
DSPstatic void knlSched(void);
void knlInitSys(void);
DSPstatic void knlInitRdyList(void);
DSPstatic void knlInitTCBList(void);

DSPstatic knlStack_t *knlTaskStkInit(void(*task)(void), knlStack_t *ptos);

#if STACK_CHECK_EN == 1
DSPstatic void knlCheckStack(void);
#endif

/***********************global function prototypes*****************************/
#ifndef __RAM_FUNC__
#define __RAM_FUNC__   __attribute__((section(".RamFunc")))
#endif

/*******************************************************************************
* Function Name  : knlInitKernel
* Description    : initialize the kernel
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void knlInitKernel(void)
{
    knlInitSys();
    knlInitRdyList();
    knlInitTCBList();
}


/*******************************************************************************
* Function Name  : knlStartKernel
* Description    : start the kernel os 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void knlStartKernel(void)
{
    knlNextTaskPrio = knlFindNextTaskPrio();
    knlCurrentTaskPrio = knlNextTaskPrio;
    knlNextTaskTCB = &knlTCBTable[knlNextTaskPrio];
    knlCurrentTaskTCB = knlNextTaskTCB;
    knlStartTask();
}


/*******************************************************************************
* Function Name  : knlCreateTask
* Description    : os create a task 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void knlCreateTask(void(*task)(void), kernel_t stksize, kernel_t prio)
{
    knl_tcb_t *ptcb;
    knlStack_t *psp;
    knlStack_t *ptos;

    ptos = knlStackPtr;
    stksize &= 0xFFFE; //must be even
    knlStackTable[prio].start = knlStackPtr;
    #if STACK_DIRECTION == DIR_UP
    knlStackPtr += stksize;
    #else
    knlStackPtr -= stksize;
    #endif
    knlStackTable[prio].end = knlStackPtr - 1;
    knlStackTable[prio].used = 0;
    psp = (knlStack_t*)knlTaskStkInit(task, ptos);
    ptcb = &knlTCBTable[prio];
    ptcb->tcbStackPtr = psp;
    ptcb->timerPeriod = 0;
    ptcb->targetTime = 0;
    ptcb->eventRx = 0;
    knlTaskReady(prio);
}


/*******************************************************************************
* Function Name  : knlWaitEvent
* Description    : wait specified events for current task.if no specified events
                   found, suspend current task and run next ready task with highest
                   priority
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
taskEvent_t knlWaitEvent(taskEvent_t eventWait)
{
    taskEvent_t event;

#if PRE_EMPTIVE_EN  == 0
    knlTimerTick();
#endif
#if STACK_CHECK_EN == 1
    knlCheckStack();
#endif

    
    PROTECT_EN();

    knlCurrentTaskTCB->eventWait = eventWait;

    event = knlCurrentTaskTCB->eventRx;

    if (checkEvent(event, eventWait) == 0)
    {
        knlTaskSuspend(knlCurrentTaskPrio);
        PROTECT_DIS();

        knlSched();

        PROTECT_EN();
        event = knlCurrentTaskTCB->eventRx;
    }
  
    knlCurrentTaskTCB->eventRx &= (~eventWait);
    PROTECT_DIS();
    return(event &eventWait);
}


/*******************************************************************************
* Function Name  : knlIsTimeOut
* Description    : This function get the current task event
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
taskEvent_t knlGetEvent(void)
{
    taskEvent_t event;
#if PRE_EMPTIVE_EN == 0
    knlTimerTick();
#endif
    PROTECT_EN();
    event = knlCurrentTaskTCB->eventRx;
    knlCurrentTaskTCB->eventRx = 0;
    PROTECT_DIS();
    return (event);
}


/*******************************************************************************
* Function Name  : knlSendEventISR
* Description    : send an event to specified task in ISR level
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ void knlSendEventISR(kernel_t prio, kernel_t eventId)
{
    knl_tcb_t *ptcb;
    
    ptcb = &knlTCBTable[prio];
    ptcb->eventRx |= EVENT(eventId);
    if (checkEvent(ptcb->eventRx, ptcb->eventWait))
    {
        knlTaskReady(prio);
    }  
}
/*******************************************************************************
* Function Name  : knlSendEventISR
* Description    : send an event to specified task in task level
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void knlSendEvent(kernel_t prio, kernel_t eventId)
{
    knl_tcb_t *ptcb;
    PROTECT_EN();
    ptcb = &knlTCBTable[prio];
    ptcb->eventRx |= EVENT(eventId);
    if (checkEvent(ptcb->eventRx, ptcb->eventWait))
    {
        knlTaskReady(prio);
    }
    PROTECT_DIS();
}


/*******************************************************************************
* Function Name  : knlGetTargetTime
* Description    : This function get the target time
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
int32_t knlGetTargetTime(uint32_t tick)
{
    int32_t tmp;
    tmp = kernelTick + tick;
    if(tmp == 0)tmp = 1;
    return tmp;
}


/*******************************************************************************
* Function Name  : knlIsTimeOut
* Description    : This function check the time out or not
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
bool_t knlChkTimeOut(int32_t* targetTime)
{
    bool_t res;
    if(*targetTime == 0)return true;
    res = (kernelTick - *targetTime) >= 0 ? true : false;
    if(res)*targetTime = 0;
    return res;
}


/*******************************************************************************
* Function Name  : knlDelayTask
* Description    : This function delays the task
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void knlDelayTask(kernel_t prio,uint32_t tick)
{
    knlTCBTable[prio].targetTime = kernelTick + tick;
}

/*******************the local function definition ******************************/
/*******************************************************************************
* Function Name  : knlFindNextTaskPrio
* Description    : This function is used to find the highest task next to execute.
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ DSPstatic kernel_t knlFindNextTaskPrio(void)
{
    if(knlRdyMap == 0)
    {
        return KNL_TASK_LOWEST_PRIO;
    }

    else if(knlRdyMap & 0x000F)
    {
        return(knlUnMapTbl[knlRdyMap & 0x000F]);
    }
    
    else if(knlRdyMap & 0x00F0)
    {
        return (knlUnMapTbl[(knlRdyMap >> 4) & 0x000F] + 4);
    }
    else if(knlRdyMap & 0x0F00)
    {
        return(knlUnMapTbl[(knlRdyMap >> 8) & 0x000F] + 8);
    }
    else
    {
        return(knlUnMapTbl[(knlRdyMap >> 12) & 0x000F] + 12);
    }
}


/*******************************************************************************
* Function Name  : knlSched
* Description    : This function is called by other OS services to determine whether a
                   new, high level code priority task has been made ready to run.This 
                   function is invoked by TASK and is not used to reschedule task from 
                   ISRs (see OSIntExit() for ISR rescheduling).
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
DSPstatic void knlSched(void)
{
    PROTECT_EN();
    knlNextTaskPrio = knlFindNextTaskPrio(); 
    knlCurrentTaskPrio = knlNextTaskPrio;
    knlNextTaskTCB = &knlTCBTable[knlNextTaskPrio];
    TASK_SW();
    PROTECT_DIS();
}


/*******************************************************************************
* Function Name  : knlInitSys
* Description    : This function do the initialize os system 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void knlInitSys(void)
{
    int i;
    for(i = 0; i < KNL_TOTAL_STK_SIZE; i++)
    {
        knlTaskStack[i] = 0x55AA55AA;
    }
  #if STACK_DIRECTION == DIR_UP
    knlStackPtr = knlTaskStack;
  #else
    knlStackPtr = knlTaskStack + KNL_TOTAL_STK_SIZE - 1;
  #endif
}


/*******************************************************************************
* Function Name  : knlInitRdyList
* Description    : This function do the initialize os system ready list 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
DSPstatic void knlInitRdyList(void)
{
    knlRdyMap = 0; //no task is ready at the beginning
    knlCurrentTaskPrio = 0;
    knlCurrentTaskTCB = (knl_tcb_t*)0;
    knlNextTaskPrio = 0;
    knlNextTaskTCB = (knl_tcb_t*)0;
}


/*******************************************************************************
* Function Name  : knlInitTCBList
* Description    : This function do the initialize os system task control block list 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
DSPstatic void knlInitTCBList(void)
{
    kernel_t i;
    knl_tcb_t *ptcb;

    ptcb = &knlTCBTable[0];
    
    for (i = 0; i < KNL_TASK_NUM; i++)
    {
        ptcb->tcbStackPtr = (void*)0;
        ptcb->timerPeriod = 0;
        ptcb->targetTime = 0;
        ptcb->eventRx = 0;
        ptcb->eventWait = 0;
        ptcb++;
    }
}


/*******************************************************************************
* Function Name  : knlTimerTick
* Description    : This function do the the time tick  
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ void knlTimerTick(void)
{
    kernel_t prio = 0;
    knl_tcb_t* pTcb;
#if PRE_EMPTIVE_EN == 0
    if(arm == false)return;
    arm = false;
#else
    kernelTick++;
#endif

//    timerTickHook();
  
    while (prio < KNL_TASK_NUM)
    {
        pTcb = &knlTCBTable[prio];
        if (pTcb->timerPeriod != 0)
        {
            if((kernelTick - pTcb->targetTime) >= 0)
            {
                pTcb->targetTime = kernelTick + pTcb->timerPeriod;
#if PRE_EMPTIVE_EN == 0
                knlSendEvent(prio, KNL_EVENT_TIMER);
#else
                knlSendEventISR(prio, KNL_EVENT_TIMER);
#endif
            }
        }
        
        prio++;
    }
}


/*******************************************************************************
* Function Name  : knlExitInt
* Description    : function used to check if a task switch is needed when exiting 
                   from an ISR
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ void knlExitInt(void)
{
    knlNextTaskPrio = knlFindNextTaskPrio();
    if (knlNextTaskPrio != knlCurrentTaskPrio)
    {
        knlCurrentTaskPrio = knlNextTaskPrio;
        knlNextTaskTCB = &knlTCBTable[knlNextTaskPrio];
        knlIntCtxSw();
    }
}


/**************************************NOTE*************************************
* for the different chip, must be refer to the interrupt instruction or data 
* sheet modify the initialize the register
*** ****************************************************************************/
/*******************************************************************************
* Function Name  : knlTaskStkInit
* Description    : initialization of task stack                  
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/

#define DEFAULT_MSTATUS		( MSTATUS_XS | MSTATUS_MPP | MSTATUS_MPIE | MSTATUS_FS_INITIAL | MSTATUS_VS_INITIAL)

DSPstatic knlStack_t *knlTaskStkInit(void(*task)(void), knlStack_t *ptos)
{
    knlStack_t *p_stk;

    p_stk = ptos + 1u;                        /* Load stack pointer and align it to 8bytes            */
    p_stk = (knlStack_t *)((knlStack_t)(p_stk) & 0xFFFFFFF8u);

    *(--p_stk) = (knlStack_t) DEFAULT_MSTATUS;     /* mstatus											   */

    *(--p_stk) = (knlStack_t) 0x31313131uL;        /* t6                                                   */
    *(--p_stk) = (knlStack_t) 0x30303030uL;        /* t5                                                   */
    *(--p_stk) = (knlStack_t) 0x29292929uL;        /* t4                                                   */
    *(--p_stk) = (knlStack_t) 0x28282828uL;        /* t3                                                   */
                                               /* Saved Registers                                      */
    *(--p_stk) = (knlStack_t) 0x27272727uL;        /* s11                                                  */
    *(--p_stk) = (knlStack_t) 0x26262626uL;        /* s10                                                  */
    *(--p_stk) = (knlStack_t) 0x25252525uL;        /* s9                                                   */
    *(--p_stk) = (knlStack_t) 0x24242424uL;        /* s8                                                   */
    *(--p_stk) = (knlStack_t) 0x23232323uL;        /* s7                                                   */
    *(--p_stk) = (knlStack_t) 0x22222222uL;        /* s6                                                   */
    *(--p_stk) = (knlStack_t) 0x21212121uL;        /* s5                                                   */
    *(--p_stk) = (knlStack_t) 0x20202020uL;        /* s4                                                   */
    *(--p_stk) = (knlStack_t) 0x19191919uL;        /* s3                                                   */
    *(--p_stk) = (knlStack_t) 0x18181818uL;        /* s2                                                   */
                                               /* Function Arguments                                   */
    *(--p_stk) = (knlStack_t) 0x17171717uL;        /* a7                                                   */
    *(--p_stk) = (knlStack_t) 0x16161616uL;        /* a6                                                   */
    *(--p_stk) = (knlStack_t) 0x15151515uL;        /* a5                                                   */
    *(--p_stk) = (knlStack_t) 0x14141414uL;        /* a4                                                   */
    *(--p_stk) = (knlStack_t) 0x13131313uL;        /* a3                                                   */
    *(--p_stk) = (knlStack_t) 0x12121212uL;        /* a2                                                   */
                                               /* Function Arguments/return values                     */
    *(--p_stk) = (knlStack_t) 0x11111111uL;        /* a1                                                   */
    *(--p_stk) = (knlStack_t) 0x10101010uL;               /* a0                                                   */
    *(--p_stk) = (knlStack_t) 0x09090909uL;        /* s1   : Saved register                                */
    *(--p_stk) = (knlStack_t) 0x08080808uL;        /* s0/fp: Saved register/Frame pointer                  */
                                               /* Temporary registers                                  */
    *(--p_stk) = (knlStack_t) 0x07070707uL;        /* t2                                                   */
    *(--p_stk) = (knlStack_t) 0x06060606uL;        /* t1                                                   */
    *(--p_stk) = (knlStack_t) 0x05050505uL;        /* t0                                                   */

    *(--p_stk) = (knlStack_t) 0x04040404uL;       /* ra: return address                                   */
    *(--p_stk) = (knlStack_t) task;                /* Entry Point                                          */

#if !(OS_SAVE_FPU_CONTEXT_DISABLE > 0)
    *(--p_stk) = (knlStack_t)0x02000000u;          /* FPSCR                                                */
                                               /* Initialize S0-S31 floating point registers           */
    *(--p_stk) = (knlStack_t)0x41F80000u;          /* F31                                                  */
    *(--p_stk) = (knlStack_t)0x41F00000u;          /* F30                                                  */
    *(--p_stk) = (knlStack_t)0x41E80000u;          /* F29                                                  */
    *(--p_stk) = (knlStack_t)0x41E00000u;          /* F28                                                  */
    *(--p_stk) = (knlStack_t)0x41D80000u;          /* F27                                                  */
    *(--p_stk) = (knlStack_t)0x41D00000u;          /* F26                                                  */
    *(--p_stk) = (knlStack_t)0x41C80000u;          /* F25                                                  */
    *(--p_stk) = (knlStack_t)0x41C00000u;          /* F24                                                  */
    *(--p_stk) = (knlStack_t)0x41B80000u;          /* F23                                                  */
    *(--p_stk) = (knlStack_t)0x41B00000u;          /* F22                                                  */
    *(--p_stk) = (knlStack_t)0x41A80000u;          /* F21                                                  */
    *(--p_stk) = (knlStack_t)0x41A00000u;          /* F20                                                  */
    *(--p_stk) = (knlStack_t)0x41980000u;          /* F19                                                  */
    *(--p_stk) = (knlStack_t)0x41900000u;          /* F18                                                  */
    *(--p_stk) = (knlStack_t)0x41880000u;          /* F17                                                  */
    *(--p_stk) = (knlStack_t)0x41800000u;          /* F16  												   */
    *(--p_stk) = (knlStack_t)0x41700000u;          /* F15                                                  */
    *(--p_stk) = (knlStack_t)0x41600000u;          /* F14                                                  */
    *(--p_stk) = (knlStack_t)0x41500000u;          /* F13                                                  */
    *(--p_stk) = (knlStack_t)0x41400000u;          /* F12                                                  */
    *(--p_stk) = (knlStack_t)0x41300000u;          /* F11                                                  */
    *(--p_stk) = (knlStack_t)0x41200000u;          /* F10                                                  */
    *(--p_stk) = (knlStack_t)0x41100000u;          /* F9                                                   */
    *(--p_stk) = (knlStack_t)0x41000000u;          /* F8                                                   */
    *(--p_stk) = (knlStack_t)0x40E00000u;          /* F7                                                   */
    *(--p_stk) = (knlStack_t)0x40C00000u;          /* F6                                                   */
    *(--p_stk) = (knlStack_t)0x40A00000u;          /* F5                                                   */
    *(--p_stk) = (knlStack_t)0x40800000u;          /* F4                                                   */
    *(--p_stk) = (knlStack_t)0x40400000u;          /* F3                                                   */
    *(--p_stk) = (knlStack_t)0x40000000u;          /* F2                                                   */
    *(--p_stk) = (knlStack_t)0x3F800000u;          /* F1                                                   */
    *(--p_stk) = (knlStack_t)0x00000000u;          /* F0                                                   */
#endif

    return (p_stk);
}

#if STACK_CHECK_EN == 1

#define cHex2Asc(A)  (A+'0')
int sDispStack(uint8_t *p)
{
	int wLen=0;
	int i;
    for(i = 0; i < KNL_TASK_NUM; i++)
    {
        p[wLen++]=cHex2Asc(i); p[wLen++]=':';
        p[wLen++]=cHex2Asc(knlStackTable[i].used/100);
        p[wLen++]=cHex2Asc((knlStackTable[i].used%100)/10);
        p[wLen++]=cHex2Asc((knlStackTable[i].used%100)%10);
        p[wLen++]=' ';p[wLen++]=' ';p[wLen++]=' ';
    }
    p[wLen++]='\r';p[wLen++]='\n';
   return wLen;
}

DSPstatic void knlCheckStack(void)
{
    static int32_t timer = 1000;
    uint16_t i;
    uint16_t* p;
//    uint8_t Test[100];
    if(kernelTick - timer >= 0)
    {
        timer = kernelTick + 1000;
        for(i = 0; i < KNL_TASK_NUM; i++)
        {
            
            #if STACK_DIRECTION == DIR_UP
            for(p = knlStackTable[i].start + knlStackTable[i].used; p <= knlStackTable[i].end; p++)
            #elif
            for(p = knlStackTable[i].start - knlStackTable[i].used; p >= knlStackTable[i].end; p--)
            #endif
            {
                if(*p != 0x55AA)
                {
                    knlStackTable[i].used++;
                }
                else
                {
                    break;
                }
            }
        }
    }
//    sDispStack(Test);
}


#endif

uint32_t tickCnts;

#define OS_TICK_TIMER_BASE  CPUTIMER0_BASE
#define OS_TICK_TIMER_INT   INT_TIMER0

__RAM_FUNC__ void  SysTick_Handler (void)
{
    CPUTimer_clearOverflowFlag(OS_TICK_TIMER_BASE);

    knlTimerTick();
    knlExitInt();
}

void  SysTickInitFreq (uint32_t  apbFreq)
{
    tickCnts = (apbFreq / KNL_TICKS_PER_SEC);               /* Determine nbr SysTick cnts between two OS tick intr. */

    /* Stop and clear the SysTimer. SysTimer as Non-Vector Interrupt */
    CPUTimer_init(OS_TICK_TIMER_BASE, tickCnts);
	Interrupt_register(OS_TICK_TIMER_INT, SysTick_Handler);
	Interrupt_SetPriority(OS_TICK_TIMER_INT, 0, 0);
	Interrupt_enable(OS_TICK_TIMER_INT);
	ECLIC_SetShvIRQ(OS_TICK_TIMER_INT, ECLIC_NON_VECTOR_INTERRUPT);

    /* Set SWI interrupt level to lowest level/priority, SysTimerSW as Vector Interrupt */
    ECLIC_DisableIRQ(SysTimerSW_IRQn);
    ECLIC_SetVector(SysTimerSW_IRQn, (uint32_t)Software_IRQHandler);
    ECLIC_SetShvIRQ(SysTimerSW_IRQn, ECLIC_VECTOR_INTERRUPT);
    ECLIC_SetLevelIRQ(SysTimerSW_IRQn, 0);
    ECLIC_EnableIRQ(SysTimerSW_IRQn);
}
