/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* Includes ------------------------------------------------------------------*/
#include <sys/stat.h>

#include "device.h"
#include "board_cfg.h"
#include "OS.h"
#include "sci_bsp.h"

volatile uint32_t os_task_cnt[8];

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Main program
  * @param  None
  * @retval None
  */

volatile uint32_t cputimer1_isr_cnt;

void Task0_Func(void)
{
	taskEvent_t event;

    while(1) {
    	os_task_cnt[0]++;
    	event = OSEventPend();

    	if(event & EVENT(OS_EVENT_TIMER)) {
    		DELAY_US(100);//log_info("this is Sci Comm task %d!\n", tickCnt);
    	}
    }
}

void Task1_Func(void)
{
	taskEvent_t event;

//    OSStartTimer(cPrioDataTask, 0, 1000);

    while(1) {
    	os_task_cnt[1]++;
    	event = OSEventPend();

    	if(event & EVENT(OS_EVENT_TIMER)) {
    		DELAY_US(100);//log_info("this is Data task %d!\n", tickCnt);
    	}
    }
}

void Task2_Func(void)
{
	taskEvent_t event;
	uint32_t task2_gpio_val = 0;

    while(1)
    {
    	os_task_cnt[2]++;
    	event = OSEventPend();

    	if(event & EVENT(OS_EVENT_TIMER))
    	{
        	GPIO_writePin(2, task2_gpio_val);
        	task2_gpio_val ^= 1;
        	DELAY_US(1000);
    	}
    }
}

void Task3_Func(void)
{
	taskEvent_t event;
	uint32_t task3_gpio_val = 0;

    while(1)
    {
    	event = OSEventPend();

    	if(event & EVENT(OS_EVENT_TIMER))
    	{
    		os_task_cnt[3]++;
			GPIO_writePin(3, task3_gpio_val);
			task3_gpio_val ^= 1;
			DELAY_US(100);
    	}

    	if(event & EVENT(1))
    	{
    		os_task_cnt[7]++;
    		GPIO_writePin(4, task3_gpio_val);
    		task3_gpio_val ^= 1;
    	}
    }
}

void Task4_Func(void)
{
	taskEvent_t event;
	uint32_t task4_gpio_val = 0;

    while(1)
    {
    	event = OSEventPend();

    	if(event & EVENT(OS_EVENT_TIMER))
    	{
    		os_task_cnt[4]++;
			GPIO_writePin(4, task4_gpio_val);
			task4_gpio_val ^= 1;
			DELAY_US(100);
    	}

    	if(event & EVENT(2))
    	{
    		os_task_cnt[7]++;
    		GPIO_writePin(4, task4_gpio_val);
    		task4_gpio_val ^= 1;
    	}
    }
}

void Task_Idle_Func(void)
{
    while(1)
    {
    	os_task_cnt[5]++;
    	DELAY_US(1);
    }
}

__INTERRUPT void Timer1_IRQHandler(void)
{
    CPUTimer_ClearInterruptStatus(CPUTIMER1_BASE);
    __DSB();
    cputimer1_isr_cnt ++;
}

int main(void)
{
    uint32_t oldTicks;

    __disable_irq();
    Device_init();

    /* Configure GPIO 28 and 29 as GPIO mode */
    GPIO_setPinConfig(GPIO_28_GPIO28);
    GPIO_setPinConfig(GPIO_29_GPIO29);
    GPIO_setPadConfig(29, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(29, GPIO_QUAL_3SAMPLE);

    /* Configure GPIO 12 and 13 as GPIO mode */
    GPIO_setPinConfig(GPIO_12_GPIO12);
    GPIO_setAnalogMode(13, GPIO_ANALOG_DISABLED);  //must set GPIO13 as digital mode
    GPIO_setPinConfig(GPIO_13_GPIO13);
    GPIO_setPadConfig(13, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(13, GPIO_QUAL_3SAMPLE);

    /* Initialize SCIA and SCIB */
    sci_init(SCIA_BASE);
    sci_init(SCIB_BASE);

    /* Configure GPIO 28 and 29 as SCIA TX RX */
    GPIO_setPinConfig(GPIO_28_SCIA_RX);
    GPIO_setPinConfig(GPIO_29_SCIA_TX);

    /* Configure GPIO 12 and 13 as SCIB TX RX */
    GPIO_setPinConfig(GPIO_12_SCIB_TX);
    GPIO_setPinConfig(GPIO_13_SCIB_RX);

    /* Register and enable SCIA and SCIB interrupt */
    Interrupt_register(INT_SCIA, SCIA_ISR);
    Interrupt_register(INT_SCIB, SCIB_ISR);
    Interrupt_enable(INT_SCIA);
    Interrupt_enable(INT_SCIB);

    /* Enable CPUTimer1 interrupt */
    CPUTimer_init(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/100000);   //100kHz for test
    Interrupt_register(INT_TIMER1, Timer1_IRQHandler);
    Interrupt_enable(INT_TIMER1);

    /* Initialize CPUTimer0 for RTOS timer tick */
    SysTickInitFreq(DEVICE_APBCLK_FREQ);

    /* OS initialization */
    OSInit();

    OSTaskCreate(Task0_Func, TaskStackSize, cPrioTask0);
    OSStartTimer(cPrioTask0, 0, 1);

    OSTaskCreate(Task1_Func, TaskStackSize, cPrioTask1);
    OSStartTimer(cPrioTask1, 0, 10);

    OSTaskCreate(Task2_Func, TaskStackSize, cPrioTask2);
    OSStartTimer(cPrioTask2, 0, 20);

    OSTaskCreate(Task3_Func, TaskStackSize, cPrioTask3);
    OSStartTimer(cPrioTask3, 0, 100);

    OSTaskCreate(Task4_Func, TaskStackSize, cPrioTask4);
    OSStartTimer(cPrioTask4, 0, 100);

    OSTaskCreate(Task_Idle_Func, TaskStackSize, cPrioIdle);
    OSStartTimer(cPrioIdle, 0, 100);

    /* start RTOS */
    OSStart();

    for(;;);

    return 0;
}

