/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    main.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdio.h>
#include "device.h"
#include "driverlib.h"
#include "eqep_ex2_calculation.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */


#define eqep_ex2_ans_size 16
//#define TB_CLK    DEVICE_SYSCLK_FREQ        // Time base clock is SYSCLK
//#define TB_CLK    ((20000000U * 48) / (2 * 4 * 1))
#define TB_CLK    25000000U
#define PWM_CLK   500                       // We want to output at 5 kHz
                                            // (300 rpm)
#define PRD_VAL   (TB_CLK / (PWM_CLK * 2))  // Calculate value period value
                                            // for up-down count mode

// .9999 / 4000 converted to IQ26 fixed point format
#define MECH_SCALER     16776
// 2 pole pairs in this example
#define POLE_PAIRS      2
// Angular offset between encoder and Phase A
#define CAL_ANGLE       0
// See Equation 5 in eqep_ex2_calculation.c
#define SPEED_SCALER    ((((uint64_t)32 * TB_CLK / 64) * 60) / (24000000))
// Base/max rpm is 6000rpm
#define BASE_RPM        6000


//
// EQEP1 -> myEQEP0 Pinmux
//
//
// EQEP1_A - GPIO Settings
//
#define GPIO_PIN_EQEP1_A 25
#define myEQEP0_EQEPA_GPIO 25
#define myEQEP0_EQEPA_PIN_CONFIG GPIO_25_EQEP1_A
//
// EQEP1_B - GPIO Settings
//
#define GPIO_PIN_EQEP1_B 21
#define myEQEP0_EQEPB_GPIO 21
#define myEQEP0_EQEPB_PIN_CONFIG GPIO_21_EQEP1_B
//
// EQEP1_STROBE - GPIO Settings
//
#define GPIO_PIN_EQEP1_STROBE 12
#define myEQEP0_EQEPSTROBE_GPIO 12
#define myEQEP0_EQEPSTROBE_PIN_CONFIG GPIO_12_EQEP1_STROBE
//
// EQEP1_INDEX - GPIO Settings
//
#define GPIO_PIN_EQEP1_INDEX 0
#define myEQEP0_EQEPINDEX_GPIO 0
#define myEQEP0_EQEPINDEX_PIN_CONFIG GPIO_0_EQEP1_INDEX


#define myEQEP0_BASE EQEP1_BASE
/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

PosSpeed_Object posSpeed =
{
    0, 0, 0, 0,     // Initialize outputs to zero
    MECH_SCALER,    // mechScaler
    POLE_PAIRS,     // polePairs
    CAL_ANGLE,      // calAngle
    SPEED_SCALER,   // speedScaler
    0,              // Initialize output to zero
    BASE_RPM,       // baseRPM
    0, 0, 0, 0 ,0      // Initialize outputs to zero
};

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

uint16_t interruptCount = 0;
uint32_t eqep_ex2_count =0;  // counter to check measurement gets saturated
uint32_t eqep_ex2_pass=0, eqep_ex2_fail =0; // Pass or fail indicator

//Temprory

int32_t eqep_ex2_ans[eqep_ex2_ans_size];
int32_t eqep_ex2_ans_freq[eqep_ex2_ans_size];
uint32_t eqep_ex2_ans_speedFR[eqep_ex2_ans_size];


/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
void eqep_ex2_myEQEP0_init();
void eqep_ex2_initEPWM(void);
void EPWM1XINT_IRQHandler(void);

/**
  * \brief  Main program
  * \param  None
  * \retval None
  */
int main(void)
{
	//
    // Initialize device clock and peripherals
    //
	Device_init();
    UartPrint_init(LOG_SCI_BASE, 115200);

    //
    // Board Initialization
    // Setup eQEP1, configuring the unit timer and quadrature capture units
    // Initialize GPIOs for use as EQEP1A, EQEP1B, and EQEP1I
    //
    eqep_ex2_myEQEP0_init();
    //
    // Setup ePWM1 to generate a 5 kHz signal to be an input to the eQEP
    //
    eqep_ex2_initEPWM();

    //epeq1a
    XBAR_enableEPWMMux(XBAR_TRIP4, XBAR_MUX01);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT1, 0);
    XBAR_setEPWMMuxConfig(XBAR_TRIP4, XBAR_EPWM_MUX01_INPUTXBAR1);

	//epeq1b
    XBAR_enableEPWMMux(XBAR_TRIP5, XBAR_MUX03);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT2, 1);
    XBAR_setEPWMMuxConfig(XBAR_TRIP5, XBAR_EPWM_MUX03_INPUTXBAR2);
	//epeq1I
    XBAR_enableEPWMMux(XBAR_TRIP7, XBAR_MUX05);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT3, 2);
    XBAR_setEPWMMuxConfig(XBAR_TRIP7, XBAR_EPWM_MUX05_INPUTXBAR3);

    Interrupt_register(INT_EPWM1,EPWM1XINT_IRQHandler);
	Interrupt_SetPriority(INT_EPWM1,2,2);
	Interrupt_enable(INT_EPWM1);

	__enable_irq();

    //
    // Loop indefinitely
    //

    while(1)
    {
        ;
    }

}

void eqep_ex2_myEQEP0_init()

	{

	//
	// Disable, clear all flags and interrupts
	//
	EQEP_disableInterrupt(myEQEP0_BASE,
		(EQEP_INT_GLOBAL     		|
		EQEP_INT_POS_CNT_ERROR		|
		EQEP_INT_PHASE_ERROR    	|
		EQEP_INT_DIR_CHANGE    		|
		EQEP_INT_WATCHDOG          	|
		EQEP_INT_UNDERFLOW         	|
		EQEP_INT_OVERFLOW        	|
		EQEP_INT_POS_COMP_READY    	|
		EQEP_INT_POS_COMP_MATCH   	|
		EQEP_INT_STROBE_EVNT_LATCH	|
		EQEP_INT_INDEX_EVNT_LATCH 	|
		EQEP_INT_UNIT_TIME_OUT   	|
		EQEP_INT_QMA_ERROR));
	EQEP_clearInterruptStatus(myEQEP0_BASE,
		(EQEP_INT_GLOBAL     		|
		EQEP_INT_POS_CNT_ERROR		|
		EQEP_INT_PHASE_ERROR    	|
		EQEP_INT_DIR_CHANGE    		|
		EQEP_INT_WATCHDOG          	|
		EQEP_INT_UNDERFLOW         	|
		EQEP_INT_OVERFLOW        	|
		EQEP_INT_POS_COMP_READY    	|
		EQEP_INT_POS_COMP_MATCH   	|
		EQEP_INT_STROBE_EVNT_LATCH	|
		EQEP_INT_INDEX_EVNT_LATCH 	|
		EQEP_INT_UNIT_TIME_OUT   	|
		EQEP_INT_QMA_ERROR));

	EQEP_SourceSelect source_myEQEP0 =
	{
		EQEP_SOURCE_PWMXBAR1, 		// eQEPA source
		EQEP_SOURCE_PWMXBAR2,		// eQEPB source
		EQEP_SOURCE_PWMXBAR3,  	// eQEP Index source
	};
//	EQEP_SourceSelect source_myEQEP0 =
//	{
//		EQEP_SOURCE_DEVICE_PIN, 		// eQEPA source
//		EQEP_SOURCE_DEVICE_PIN,		// eQEPB source
//		EQEP_SOURCE_DEVICE_PIN,  	// eQEP Index source
//	};

	//
	// Selects the source for eQEPA/B/I signals
	//
	EQEP_selectSource(myEQEP0_BASE, source_myEQEP0);
	//
	// Set the strobe input source of the eQEP module.
	//
	EQEP_setStrobeSource(myEQEP0_BASE,EQEP_STROBE_FROM_GPIO);
	//
	// Sets the polarity of the eQEP module's input signals.
	//
	EQEP_setInputPolarity(myEQEP0_BASE,false,false,false,false);
	//
	// Configures eQEP module's quadrature decoder unit.
	//
	EQEP_setDecoderConfig(myEQEP0_BASE, (EQEP_CONFIG_QUADRATURE | EQEP_CONFIG_1X_RESOLUTION | EQEP_CONFIG_NO_SWAP | EQEP_CONFIG_IGATE_DISABLE));
	//
	// Set the emulation mode of the eQEP module.
	//
	EQEP_setEmulationMode(myEQEP0_BASE,EQEP_EMULATIONMODE_RUNFREE);
	//
	// Configures eQEP module position counter unit.
	//
	EQEP_setPositionCounterConfig(myEQEP0_BASE,EQEP_POSITION_RESET_IDX,4294967295U);
	//
	// Sets the current encoder position.
	//
	EQEP_setPosition(myEQEP0_BASE,0U);
	//
	// Enables the eQEP module unit timer.
	//
	//EQEP_enableUnitTimer(myEQEP0_BASE,1200000U);
	EQEP_enableUnitTimer(myEQEP0_BASE,250000U);
	//
	// Disables the eQEP module watchdog timer.
	//
	EQEP_disableWatchdog(myEQEP0_BASE);
	//
	// Configures the quadrature modes in which the position count can be latched.
	//
	EQEP_setLatchMode(myEQEP0_BASE,(EQEP_LATCH_UNIT_TIME_OUT|EQEP_LATCH_RISING_STROBE|EQEP_LATCH_RISING_INDEX));
	//
	// Set the quadrature mode adapter (QMA) module mode.
	//
	EQEP_setQMAModuleMode(myEQEP0_BASE,EQEP_QMA_MODE_BYPASS);
	//
	// Disable Direction Change During Index
	//
	EQEP_disableDirectionChangeDuringIndex(myEQEP0_BASE);
	//
	// Configures the mode in which the position counter is initialized.
	//
	EQEP_setPositionInitMode(myEQEP0_BASE,(EQEP_INIT_DO_NOTHING));
	//
	// Sets the software initialization of the encoder position counter.
	//
	EQEP_setSWPositionInit(myEQEP0_BASE,true);
	//
	// Sets the init value for the encoder position counter.
	//
	EQEP_setInitialPosition(myEQEP0_BASE,0U);
	//
	// Enables the eQEP module.
	//
	EQEP_enableModule(myEQEP0_BASE);
	//
	// Configures eQEP module edge-capture unit.
	//
	EQEP_setCaptureConfig(myEQEP0_BASE,EQEP_CAPTURE_CLK_DIV_64,EQEP_UNIT_POS_EVNT_DIV_32);
	//
	// Enables the eQEP module edge-capture unit.
	//
	EQEP_enableCapture(myEQEP0_BASE);
}

void eqep_ex2_initEPWM(void)
{
    //
    // Disable the ePWM time base clock before configuring the module
    //
		#if 0
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
		#endif

    //
    // Set phase shift to 0 and clear the time base counter
    //
    EPWM_setPhaseShift(EPWM1_BASE, 0);
    EPWM_setTimeBaseCounter(EPWM1_BASE, 0);

    //
    // Disable the shadow load; the load will be immediate instead
    //
    EPWM_disableCounterCompareShadowLoadMode(EPWM1_BASE,
                                             EPWM_COUNTER_COMPARE_A);
    EPWM_disableCounterCompareShadowLoadMode(EPWM1_BASE,
                                             EPWM_COUNTER_COMPARE_B);

    //
    // Set the compare A value to half the period value, compare B to 0
    //
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, PRD_VAL/2);
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_B, 0);

    //
    // Set action qualifier behavior on compare A events
    // - EPWM1A --> 1 when CTR = CMPA and increasing
    // - EPWM1A --> 0 when CTR = CMPA and decreasing
    //
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

    //
    // Set action qualifier behavior on compare B events
    // - EPWM1B --> 1 when CTR = PRD and increasing
    // - EPWM1B --> 0 when CTR = 0 and decreasing
    //
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_B,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_B,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

    //
    // Enable interrupt when the counter is equal to PRD
    //
    EPWM_setInterruptSource(EPWM1_BASE, EPWM_INT_TBCTR_PERIOD);
    EPWM_enableInterrupt(EPWM1_BASE);

    //
    // Interrupt on first event
    //
    EPWM_setInterruptEventCount(EPWM1_BASE, 1);

    //
    // Set the time base clock prescaler to /1
    //
    EPWM_setClockPrescaler(EPWM1_BASE, EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);

    //
    // Set the period value; don't shadow the register
    //
    EPWM_setPeriodLoadMode(EPWM1_BASE, EPWM_PERIOD_DIRECT_LOAD);
    EPWM_setTimeBasePeriod(EPWM1_BASE, PRD_VAL);

    //
    // Put the time base counter into up-down count mode
    //
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP_DOWN);
#if (GS32F00xx==0x0012)
    //SysCtl_enablePeripheral_TBCLKSYNC();
    //HWREG(0x4003F000 + 0x120) = 0x10000; //0110bit

    HWREG(0x4003F000 + 0x44) = 0x40000; //0120bit
    HWREG(0x40008000 + 0x27C) = 0x3;
#else
    HWREG(0x4003F000 + 0x44) = 0x40000; //3.0bit
#endif


}



/**
 * \brief   ePWM1 ISR--interrupts once every 4 QCLK counts (one period)
 *
 * \param   none
 *
 * \retval  None
 */
void EPWM1XINT_IRQHandler(void)
{
    uint16_t i;

    //
    // Position speed and measurement
    //
    eqep_ex2_PosSpeed_calculate(&posSpeed, &eqep_ex2_count);

    //
    // Comparing the eQEP measured frequency with the ePWM frequency
    // After count becomes 3 , eQEP measurement gets saturated and if
    // measure speed is 5 more or less than input speed then eqep_ex2_pass = 1
    //
    if (eqep_ex2_count >= 2){
        if (((posSpeed.speedRPMFR - 30) < 5) &&
             ((posSpeed.speedRPMFR - 30) > -5))
        {
            eqep_ex2_pass = 1; eqep_ex2_fail = 0;
        }
        else {
            eqep_ex2_fail = 1; eqep_ex2_pass = 0;
        }
       }

		uint32_t N = 0;
		if (eqep_ex2_count >= N && eqep_ex2_count <N + eqep_ex2_ans_size){
				eqep_ex2_ans[eqep_ex2_count-N] = posSpeed.oldPos;
				eqep_ex2_ans_freq[eqep_ex2_count-N] = posSpeed.speedRPMFR;
				eqep_ex2_ans_speedFR[eqep_ex2_count-N] = posSpeed.speedFR;
		}else if(eqep_ex2_count >= N + eqep_ex2_ans_size){
			//for(; eqep_ex2_count <200; eqep_ex2_count++);
		}

    //
    // Control loop for position control and speed control
    //
    interruptCount++;
    if(interruptCount == 1000)
    {
        //
        // Pulse index signal (1 pulse/rev)
        //

				#if 1
        GPIO_writePin(3, 1);
				#endif
        for(i = 0; i < 700; i++)
        {
            ;
        }
        #if 1
        GPIO_writePin(3, 0);
				#endif

				//GPIO_disableWrite(0x0F0F);
    }


    //
    // Clear interrupt flag and issue ACK
    //
    EPWM_clearEventTriggerInterruptFlag(EPWM1_BASE);
    //Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP3);
}


#ifdef __cplusplus
}
#endif

/*****END OF FILE****/
