/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "gpio_ex4_aio_external_interrupt.h"
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

volatile uint32_t xint1Count = 0; /* interrupt counter */
uint32_t          loopCount  = 0; /* loop count */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
void PinMux_init()
{
    /* Set pin configurations for input and output pins */
    GPIO_setPinConfig(myAIOInputInterrupt0_CONFIG);
    GPIO_setPinConfig(myGPIOOutput0_CONFIG);
    GPIO_setPinConfig(myGPIOOutput1_CONFIG);

    /* Check pin type and disable analog mode if not a GPIO */
    if ((check_pin_type(myAIOInputInterrupt0) != HW_PIN_TYPE_GPIO)) {
        GPIO_setAnalogMode(myAIOInputInterrupt0, GPIO_ANALOG_DISABLED);
    }
    if (check_pin_type(myGPIOOutput0) != HW_PIN_TYPE_GPIO) {
        GPIO_setAnalogMode(myGPIOOutput0, GPIO_ANALOG_DISABLED);
    }
    if (check_pin_type(myGPIOOutput1) != HW_PIN_TYPE_GPIO) {
        GPIO_setAnalogMode(myGPIOOutput1, GPIO_ANALOG_DISABLED);
    }
}

void myAIOInputInterrupt0_init()
{
    /* Set qualification period for input pin (debounce or sampling period) */
    GPIO_setQualificationPeriod(myAIOInputInterrupt0, 510);

    /* Configure pin settings: standard and pulldown resistor */
    GPIO_setPadConfig(myAIOInputInterrupt0, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLDOWN);

    /* Set input qualification mode to synchronous */
    GPIO_setQualificationMode(myAIOInputInterrupt0, GPIO_QUAL_SYNC);

    /* Set pin direction mode to input (for interrupt) */
    GPIO_setDirectionMode(myAIOInputInterrupt0, AIO_IE_MODE_IN);
}

void myGPIOOutput0_init()
{
    /* Configure output pin settings */
    GPIO_setPadConfig(myGPIOOutput0, GPIO_PIN_TYPE_STD);

    /* Set qualification mode to synchronous */
    GPIO_setQualificationMode(myGPIOOutput0, GPIO_QUAL_SYNC);

    /* Set pin direction mode to output */
    GPIO_setDirectionMode(myGPIOOutput0, GPIO_DIR_MODE_OUT);
}

void myGPIOOutput1_init()
{
    /* Configure output pin settings */
    GPIO_setPadConfig(myGPIOOutput1, GPIO_PIN_TYPE_STD);

    /* Set qualification mode to synchronous */
    GPIO_setQualificationMode(myGPIOOutput1, GPIO_QUAL_SYNC);

    /* Set pin direction mode to output */
    GPIO_setDirectionMode(myGPIOOutput1, GPIO_DIR_MODE_OUT);
}

void GPIO_init()
{
    /* Initialize the specific GPIOs for input and output */
    myAIOInputInterrupt0_init();
    myGPIOOutput0_init();
    myGPIOOutput1_init();
}

/*
 * @brief
 * Initialize the board by setting up pin multiplexing and GPIO initialization
 */
void Board_init(void)
{
    /* Initialize the pin multiplexer (set pin configurations) */
    PinMux_init();

    /* Initialize the GPIOs for the application */
    GPIO_init();
}

/*
 * @brief  XINT1 Interrupt Service Routine (ISR) for handling the interrupt
 */
#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void XINT1_IRQHandler(void)
{
    /* Save current interrupt context for proper context switching */
    SAVE_IRQ_CSR_CONTEXT();

    /* Toggle output pin to signal interrupt */
    GPIO_togglePin(myGPIOOutput1);

    /* Ensure that all memory operations are completed */
    __DSB();

    /* Increment the interrupt counter */
    xint1Count++;

    /* Restore interrupt context */
    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void XINT1_IRQHandler(void)
{
    /* Toggle output pin to signal interrupt */
    GPIO_togglePin(myGPIOOutput1);

    /* Ensure that all memory operations are completed */
    __DSB();

    /* Increment the interrupt counter */
    xint1Count++;
}
#endif

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function for the application
 */
int main(void)
{
    uint32_t xint1CountTemp;

    /* Initialize device */
    Device_init();

    /* Initialize the board (pin mux, GPIOs, etc.) */
    Board_init();

    /* Configure xint1 */
    GPIO_setInterruptPin(myAIOInputInterrupt0, GPIO_INT_XINT1);       /* Select myAIOInputInterrupt0 as XINT1 */
    GPIO_setInterruptType(GPIO_INT_XINT1, GPIO_INT_TYPE_RISING_EDGE); /* Rising edge interruption */
    GPIO_enableInterrupt(GPIO_INT_XINT1);                             /* Enable XINT1 */

    /* Register and enable the interrupt handler */
    Interrupt_register(INT_XINT1, &XINT1_IRQHandler);
    Interrupt_enable(INT_XINT1);

    /* Initialize output pins with specific initial states */
    GPIO_writePin(myGPIOOutput0, 0);
    GPIO_writePin(myGPIOOutput1, 1);

    /* Enable global interrupts */
    __enable_irq();

    /* Loop */
    while (1) {

        /* Store the current interrupt count to detect changes */
        xint1CountTemp = xint1Count;

        /* Set output pin high */
        GPIO_writePin(myGPIOOutput0, 1);

        /* Wait until the interrupt count changes (i.e., interrupt occurs) */
        while (xint1Count == xint1CountTemp) {
            ;
        }

        /* Check if the interrupt count has increased by 1 (indicating an interrupt occurred) */
        if (xint1Count == (xint1CountTemp + 1)) {
            loopCount++;                     /* Increment the loop count */
            GPIO_writePin(myGPIOOutput0, 0); /* Set output pin low */
        }

        /* Delay for 1 second (1000 ms) */
        DELAY_US(1000 * 1000);
    }

    return 0;
}
