#include "OS.h"
#include "riscv_encoding.h"

//#pragma DATA_SECTION(OSTaskStack,"kernelStack");
OSStack_t OSTaskStack[OS_TOTAL_STK_SIZE]; /* Stack of all tasks*/

//******************************************************************************
//*                 the constant definition(only use in local file)
//******************************************************************************
const kernel_t OSUnMapTbl[16] =
{
    OS_TASK_LOWEST_PRIO, 0, 1, 0,
    2, 0, 1, 0,
    3, 0, 1, 0,
    2, 0, 1, 0
};

//******************************************************************************
//*                 the global variable define 
//******************************************************************************
OS_tcb_t OSTCBTable[OS_TASK_NUM]; /* Table of TCB*/
OS_stack_t OSStackTable[OS_TASK_NUM]; /* Table of Stack*/

//******************************************************************************
//*                 the local variable define 
//******************************************************************************
static uint16_t OSRdyMap;

kernel_t OSNextTaskPrio;     /* Priority of highest priority in ready table  */
kernel_t OSCurrentTaskPrio;  /* Priority of current task              */
OS_tcb_t *OSNextTaskTCB;    /* Pointer to highest priority TCB in ready table*/
OS_tcb_t *OSCurrentTaskTCB; /* Pointer to TCB currently running */
OSStack_t *OSStackPtr;

DSPstatic int32_t kernelTick = 0;
#if PRE_EMPTIVE_EN == 0
DSPstatic bool_t arm = false;
#endif
//******************************************************************************
//*                 the define for the os 
//******************************************************************************
//the Macro define for some operation
#define OSTaskReady(prio)            {OSRdyMap |= BIT(prio);}   //set task ready
#define OSTaskSuspend(prio)        {OSRdyMap &= ~BIT(prio);} //suspend a task
#ifndef timerTickHook
#define timerTickHook()
#else
extern void timerTickHook(void);
#endif
//******************************************************************************
//* Local function Prototypes
//******************************************************************************
DSPstatic kernel_t OSFindNextTaskPrio(void);
void OSSched(void);
void OSInitSys(void);
DSPstatic void OSInitRdyList(void);
DSPstatic void OSInitTCBList(void);

DSPstatic OSStack_t *OSTaskStkInit(void(*task)(void), OSStack_t *ptos);

#if STACK_CHECK_EN == 1
DSPstatic void OSCheckStack(void);
#endif

/***********************global function prototypes*****************************/
#ifndef __RAM_FUNC__
#define __RAM_FUNC__   __attribute__((section(".RamFunc")))
#endif

/*******************************************************************************
* Function Name  : OSInit
* Description    : initialize the kernel
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void OSInit(void)
{
    OSInitSys();
    OSInitRdyList();
    OSInitTCBList();
}


/*******************************************************************************
* Function Name  : OSStartKernel
* Description    : start the kernel os 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void OSStart(void)
{
    OSNextTaskPrio = OSFindNextTaskPrio();
    OSCurrentTaskPrio = OSNextTaskPrio;
    OSNextTaskTCB = &OSTCBTable[OSNextTaskPrio];
    OSCurrentTaskTCB = OSNextTaskTCB;
    OSStartTask();
}


/*******************************************************************************
* Function Name  : OSTaskCreate
* Description    : os create a task 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void OSTaskCreate(void(*task)(void), kernel_t stksize, kernel_t prio)
{
    OS_tcb_t *ptcb;
    OSStack_t *psp;
    OSStack_t *ptos;

    ptos = OSStackPtr;
    stksize &= 0xFFFE; //must be even
    OSStackTable[prio].start = OSStackPtr;
    #if STACK_DIRECTION == DIR_UP
    OSStackPtr += stksize;
    #else
    OSStackPtr -= stksize;
    #endif
    OSStackTable[prio].end = OSStackPtr - 1;
    OSStackTable[prio].used = 0;
    psp = (OSStack_t*)OSTaskStkInit(task, ptos);
    ptcb = &OSTCBTable[prio];
    ptcb->tcbStackPtr = psp;
    ptcb->timerPeriod = 0;
    ptcb->targetTime = 0;
    ptcb->eventRx = 0;
    OSTaskReady(prio);
}


/*******************************************************************************
* Function Name  : OSMaskEventPend
* Description    : wait specified events for current task.if no specified events
                   found, suspend current task and run next ready task with highest
                   priority
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
taskEvent_t OSMaskEventPend(taskEvent_t eventWait)
{
    taskEvent_t event;

#if PRE_EMPTIVE_EN  == 0
    OSTimerTick();
#endif
#if STACK_CHECK_EN == 1
    OSCheckStack();
#endif

    
    PROTECT_EN();

    OSCurrentTaskTCB->eventWait = eventWait;

    event = OSCurrentTaskTCB->eventRx;

    if (checkEvent(event, eventWait) == 0)
    {
        OSTaskSuspend(OSCurrentTaskPrio);
        PROTECT_DIS();

        OSSched();

        PROTECT_EN();
        event = OSCurrentTaskTCB->eventRx;
    }
  
    OSCurrentTaskTCB->eventRx &= (~eventWait);
    PROTECT_DIS();
    return(event &eventWait);
}


/*******************************************************************************
* Function Name  : OSEventGet
* Description    : This function get the current task event
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
taskEvent_t OSEventGet(void)
{
    taskEvent_t event;
#if PRE_EMPTIVE_EN == 0
    OSTimerTick();
#endif
    PROTECT_EN();
    event = OSCurrentTaskTCB->eventRx;
    OSCurrentTaskTCB->eventRx;
    PROTECT_DIS();
    return (event);
}


/*******************************************************************************
* Function Name  : OSISREventSend
* Description    : send an event to specified task in ISR level
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ void OSISREventSend(kernel_t prio, kernel_t eventId)
{
    OS_tcb_t *ptcb;
    
    ptcb = &OSTCBTable[prio];
    ptcb->eventRx |= EVENT(eventId);
    if (checkEvent(ptcb->eventRx, ptcb->eventWait))
    {
        OSTaskReady(prio);
    }  
}
/*******************************************************************************
* Function Name  : OSEventSend
* Description    : send an event to specified task in task level
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void OSEventSend(kernel_t prio, kernel_t eventId)
{
    OS_tcb_t *ptcb;
    PROTECT_EN();
    ptcb = &OSTCBTable[prio];
    ptcb->eventRx |= EVENT(eventId);
    if (checkEvent(ptcb->eventRx, ptcb->eventWait))
    {
        OSTaskReady(prio);
    }
    PROTECT_DIS();
}


/*******************************************************************************
* Function Name  : OSGetTargetTime
* Description    : This function get the target time
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
int32_t OSGetTargetTime(uint32_t tick)
{
    int32_t tmp;
    tmp = kernelTick + tick;
    if(tmp == 0)tmp = 1;
    return tmp;
}


/*******************************************************************************
* Function Name  : OSIsTimeOut
* Description    : This function check the time out or not
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
bool_t OSChkTimeOut(int32_t* targetTime)
{
    bool_t res;
    if(*targetTime == 0)return true;
    res = (kernelTick - *targetTime) >= 0 ? true : false;
    if(res)*targetTime = 0;
    return res;
}


/*******************************************************************************
* Function Name  : OSDelayTask
* Description    : This function delays the task
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void OSDelayTask(kernel_t prio, uint32_t tick)
{
    OSTCBTable[prio].targetTime = kernelTick + tick;
}

/*******************the local function definition ******************************/
/*******************************************************************************
* Function Name  : OSFindNextTaskPrio
* Description    : This function is used to find the highest task next to execute.
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ DSPstatic kernel_t OSFindNextTaskPrio(void)
{
    if(OSRdyMap == 0)
    {
        return OS_TASK_LOWEST_PRIO;
    }

    else if(OSRdyMap & 0x000F)
    {
        return(OSUnMapTbl[OSRdyMap & 0x000F]);
    }
    
    else if(OSRdyMap & 0x00F0)
    {
        return (OSUnMapTbl[(OSRdyMap >> 4) & 0x000F] + 4);
    }
    else if(OSRdyMap & 0x0F00)
    {
        return(OSUnMapTbl[(OSRdyMap >> 8) & 0x000F] + 8);
    }
    else
    {
        return(OSUnMapTbl[(OSRdyMap >> 12) & 0x000F] + 12);
    }
}


/*******************************************************************************
* Function Name  : OSSched
* Description    : This function is called by other OS services to determine whether a
                   new, high level code priority task has been made ready to run.This 
                   function is invoked by TASK and is not used to reschedule task from 
                   ISRs (see OSIntExit() for ISR rescheduling).
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void OSSched(void)
{
    PROTECT_EN();
    OSNextTaskPrio = OSFindNextTaskPrio(); 
    OSCurrentTaskPrio = OSNextTaskPrio;
    OSNextTaskTCB = &OSTCBTable[OSNextTaskPrio];
    TASK_SW();
    PROTECT_DIS();
}


/*******************************************************************************
* Function Name  : OSInitSys
* Description    : This function do the initialize os system 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void OSInitSys(void)
{
    int i;
    for(i = 0; i < OS_TOTAL_STK_SIZE; i++)
    {
        OSTaskStack[i] = 0x55AA55AA;
    }
  #if STACK_DIRECTION == DIR_UP
    OSStackPtr = OSTaskStack;
  #else
    OSStackPtr = OSTaskStack + OS_TOTAL_STK_SIZE - 1;
  #endif
}


/*******************************************************************************
* Function Name  : OSInitRdyList
* Description    : This function do the initialize os system ready list 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
DSPstatic void OSInitRdyList(void)
{
    OSRdyMap = 0; //no task is ready at the beginning
    OSCurrentTaskPrio = 0;
    OSCurrentTaskTCB = (OS_tcb_t*)0;
    OSNextTaskPrio = 0;
    OSNextTaskTCB = (OS_tcb_t*)0;
}


/*******************************************************************************
* Function Name  : OSInitTCBList
* Description    : This function do the initialize os system task control block list 
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
DSPstatic void OSInitTCBList(void)
{
    kernel_t i;
    OS_tcb_t *ptcb;

    ptcb = &OSTCBTable[0];
    
    for (i = 0; i < OS_TASK_NUM; i++)
    {
        ptcb->tcbStackPtr = (void*)0;
        ptcb->timerPeriod = 0;
        ptcb->targetTime = 0;
        ptcb->eventRx = 0;
        ptcb->eventWait = 0;
        ptcb++;
    }
}


/*******************************************************************************
* Function Name  : OSTimerTick
* Description    : This function do the the time tick  
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ void OSTimerTick(void)
{
    kernel_t prio = 0;
    OS_tcb_t* pTcb;
#if PRE_EMPTIVE_EN == 0
    if(arm == false)return;
    arm = false;
#else
    kernelTick++;
#endif

//    timerTickHook();
  
    while (prio < OS_TASK_NUM)
    {
        pTcb = &OSTCBTable[prio];
        if (pTcb->timerPeriod != 0)
        {
            if((kernelTick - pTcb->targetTime) >= 0)
            {
                pTcb->targetTime = kernelTick + pTcb->timerPeriod;
#if PRE_EMPTIVE_EN == 0
                OSEventSend(prio, OS_EVENT_TIMER);
#else
                OSISREventSend(prio, OS_EVENT_TIMER);
#endif
            }
        }
        
        prio++;
    }
}


/*******************************************************************************
* Function Name  : OSIntExit
* Description    : function used to check if a task switch is needed when exiting 
                   from an ISR
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
__RAM_FUNC__ void OSIntExit(void)
{
    OSNextTaskPrio = OSFindNextTaskPrio();
    if (OSNextTaskPrio != OSCurrentTaskPrio)
    {
        OSCurrentTaskPrio = OSNextTaskPrio;
        OSNextTaskTCB = &OSTCBTable[OSNextTaskPrio];
        OSIntCtxSw();
    }
}


/**************************************NOTE*************************************
* for the different chip, must be refer to the interrupt instruction or data 
* sheet modify the initialize the register
*** ****************************************************************************/
/*******************************************************************************
* Function Name  : OSTaskStkInit
* Description    : initialization of task stack                  
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/

#define DEFAULT_MSTATUS		( MSTATUS_XS | MSTATUS_MPP | MSTATUS_MPIE | MSTATUS_FS_INITIAL | MSTATUS_VS_INITIAL)

DSPstatic OSStack_t *OSTaskStkInit(void(*task)(void), OSStack_t *ptos)
{
    OSStack_t *p_stk;

    p_stk = ptos + 1u;                        /* Load stack pointer and align it to 8bytes            */
    p_stk = (OSStack_t *)((OSStack_t)(p_stk) & 0xFFFFFFF8u);

    *(--p_stk) = (OSStack_t) DEFAULT_MSTATUS;     /* mstatus											   */

    *(--p_stk) = (OSStack_t) 0x31313131uL;        /* t6                                                   */
    *(--p_stk) = (OSStack_t) 0x30303030uL;        /* t5                                                   */
    *(--p_stk) = (OSStack_t) 0x29292929uL;        /* t4                                                   */
    *(--p_stk) = (OSStack_t) 0x28282828uL;        /* t3                                                   */
                                               /* Saved Registers                                      */
    *(--p_stk) = (OSStack_t) 0x27272727uL;        /* s11                                                  */
    *(--p_stk) = (OSStack_t) 0x26262626uL;        /* s10                                                  */
    *(--p_stk) = (OSStack_t) 0x25252525uL;        /* s9                                                   */
    *(--p_stk) = (OSStack_t) 0x24242424uL;        /* s8                                                   */
    *(--p_stk) = (OSStack_t) 0x23232323uL;        /* s7                                                   */
    *(--p_stk) = (OSStack_t) 0x22222222uL;        /* s6                                                   */
    *(--p_stk) = (OSStack_t) 0x21212121uL;        /* s5                                                   */
    *(--p_stk) = (OSStack_t) 0x20202020uL;        /* s4                                                   */
    *(--p_stk) = (OSStack_t) 0x19191919uL;        /* s3                                                   */
    *(--p_stk) = (OSStack_t) 0x18181818uL;        /* s2                                                   */
                                               /* Function Arguments                                   */
    *(--p_stk) = (OSStack_t) 0x17171717uL;        /* a7                                                   */
    *(--p_stk) = (OSStack_t) 0x16161616uL;        /* a6                                                   */
    *(--p_stk) = (OSStack_t) 0x15151515uL;        /* a5                                                   */
    *(--p_stk) = (OSStack_t) 0x14141414uL;        /* a4                                                   */
    *(--p_stk) = (OSStack_t) 0x13131313uL;        /* a3                                                   */
    *(--p_stk) = (OSStack_t) 0x12121212uL;        /* a2                                                   */
                                               /* Function Arguments/return values                     */
    *(--p_stk) = (OSStack_t) 0x11111111uL;        /* a1                                                   */
    *(--p_stk) = (OSStack_t) 0x10101010uL;               /* a0                                                   */
    *(--p_stk) = (OSStack_t) 0x09090909uL;        /* s1   : Saved register                                */
    *(--p_stk) = (OSStack_t) 0x08080808uL;        /* s0/fp: Saved register/Frame pointer                  */
                                               /* Temporary registers                                  */
    *(--p_stk) = (OSStack_t) 0x07070707uL;        /* t2                                                   */
    *(--p_stk) = (OSStack_t) 0x06060606uL;        /* t1                                                   */
    *(--p_stk) = (OSStack_t) 0x05050505uL;        /* t0                                                   */

    *(--p_stk) = (OSStack_t) 0x04040404uL;       /* ra: return address                                   */
    *(--p_stk) = (OSStack_t) task;                /* Entry Point                                          */

#if !(OS_SAVE_FPU_CONTEXT_DISABLE > 0)
    *(--p_stk) = (OSStack_t)0x02000000u;          /* FPSCR                                                */
                                               /* Initialize S0-S31 floating point registers           */
    *(--p_stk) = (OSStack_t)0x41F80000u;          /* F31                                                  */
    *(--p_stk) = (OSStack_t)0x41F00000u;          /* F30                                                  */
    *(--p_stk) = (OSStack_t)0x41E80000u;          /* F29                                                  */
    *(--p_stk) = (OSStack_t)0x41E00000u;          /* F28                                                  */
    *(--p_stk) = (OSStack_t)0x41D80000u;          /* F27                                                  */
    *(--p_stk) = (OSStack_t)0x41D00000u;          /* F26                                                  */
    *(--p_stk) = (OSStack_t)0x41C80000u;          /* F25                                                  */
    *(--p_stk) = (OSStack_t)0x41C00000u;          /* F24                                                  */
    *(--p_stk) = (OSStack_t)0x41B80000u;          /* F23                                                  */
    *(--p_stk) = (OSStack_t)0x41B00000u;          /* F22                                                  */
    *(--p_stk) = (OSStack_t)0x41A80000u;          /* F21                                                  */
    *(--p_stk) = (OSStack_t)0x41A00000u;          /* F20                                                  */
    *(--p_stk) = (OSStack_t)0x41980000u;          /* F19                                                  */
    *(--p_stk) = (OSStack_t)0x41900000u;          /* F18                                                  */
    *(--p_stk) = (OSStack_t)0x41880000u;          /* F17                                                  */
    *(--p_stk) = (OSStack_t)0x41800000u;          /* F16  												   */
    *(--p_stk) = (OSStack_t)0x41700000u;          /* F15                                                  */
    *(--p_stk) = (OSStack_t)0x41600000u;          /* F14                                                  */
    *(--p_stk) = (OSStack_t)0x41500000u;          /* F13                                                  */
    *(--p_stk) = (OSStack_t)0x41400000u;          /* F12                                                  */
    *(--p_stk) = (OSStack_t)0x41300000u;          /* F11                                                  */
    *(--p_stk) = (OSStack_t)0x41200000u;          /* F10                                                  */
    *(--p_stk) = (OSStack_t)0x41100000u;          /* F9                                                   */
    *(--p_stk) = (OSStack_t)0x41000000u;          /* F8                                                   */
    *(--p_stk) = (OSStack_t)0x40E00000u;          /* F7                                                   */
    *(--p_stk) = (OSStack_t)0x40C00000u;          /* F6                                                   */
    *(--p_stk) = (OSStack_t)0x40A00000u;          /* F5                                                   */
    *(--p_stk) = (OSStack_t)0x40800000u;          /* F4                                                   */
    *(--p_stk) = (OSStack_t)0x40400000u;          /* F3                                                   */
    *(--p_stk) = (OSStack_t)0x40000000u;          /* F2                                                   */
    *(--p_stk) = (OSStack_t)0x3F800000u;          /* F1                                                   */
    *(--p_stk) = (OSStack_t)0x00000000u;          /* F0                                                   */
#endif

    return (p_stk);
}

#if STACK_CHECK_EN == 1

#define cHex2Asc(A)  (A+'0')
int sDispStack(uint8_t *p)
{
	int wLen=0;
	int i;
    for(i = 0; i < OS_TASK_NUM; i++)
    {
        p[wLen++]=cHex2Asc(i); p[wLen++]=':';
        p[wLen++]=cHex2Asc(OSStackTable[i].used/100);
        p[wLen++]=cHex2Asc((OSStackTable[i].used%100)/10);
        p[wLen++]=cHex2Asc((OSStackTable[i].used%100)%10);
        p[wLen++]=' ';p[wLen++]=' ';p[wLen++]=' ';
    }
    p[wLen++]='\r';p[wLen++]='\n';
   return wLen;
}

DSPstatic void OSCheckStack(void)
{
    static int32_t timer = 1000;
    uint16_t i;
    uint16_t* p;
//    uint8_t Test[100];
    if(kernelTick - timer >= 0)
    {
        timer = kernelTick + 1000;
        for(i = 0; i < OS_TASK_NUM; i++)
        {
            
            #if STACK_DIRECTION == DIR_UP
            for(p = OSStackTable[i].start + OSStackTable[i].used; p <= OSStackTable[i].end; p++)
            #elif
            for(p = OSStackTable[i].start - OSStackTable[i].used; p >= OSStackTable[i].end; p--)
            #endif
            {
                if(*p != 0x55AA)
                {
                    OSStackTable[i].used++;
                }
                else
                {
                    break;
                }
            }
        }
    }
//    sDispStack(Test);
}


#endif

uint32_t tickCnts;

#define OS_TICK_TIMER_BASE  CPUTIMER0_BASE
#define OS_TICK_TIMER_INT   INT_TIMER0

__INTERRUPT void  SysTick_Handler (void)
{
    CPUTimer_clearOverflowFlag(OS_TICK_TIMER_BASE);

    OSTimerTick();
    OSIntExit();
}

void  SysTickInitFreq (uint32_t  apbFreq)
{
    tickCnts = (apbFreq / OS_TICKS_PER_SEC);               /* Determine nbr SysTick cnts between two OS tick intr. */

    /* Stop and clear the SysTimer. SysTimer as Non-Vector Interrupt */
    CPUTimer_init(OS_TICK_TIMER_BASE, tickCnts);
	Interrupt_register(OS_TICK_TIMER_INT, SysTick_Handler);
	Interrupt_SetPriority(OS_TICK_TIMER_INT, 0, 0);
	Interrupt_enable(OS_TICK_TIMER_INT);

    /* Set SWI interrupt level to lowest level/priority, SysTimerSW as Vector Interrupt */
    ECLIC_DisableIRQ(SysTimerSW_IRQn);
    ECLIC_SetVector(SysTimerSW_IRQn, (uint32_t)Software_IRQHandler);
    ECLIC_SetShvIRQ(SysTimerSW_IRQn, ECLIC_VECTOR_INTERRUPT);
    ECLIC_SetLevelIRQ(SysTimerSW_IRQn, 0);
    ECLIC_EnableIRQ(SysTimerSW_IRQn);
}
