/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * @examplespi tri wire
 * @brief  
 *   SPI Three-line model.
 *
 *   This test routine uses two SPI modules. SPIA is configured as the slave device,
 *   while SPIB is configured as the master device.
 *   Before the host sends data, it sets the TALK bit; before the slave receives data,
 *   it clears the TALK bit.
 *   Before sending data from the slave device, the TALK bit should be set.
 *   Before receiving data from the host, the TALK bit should be reset, and before receiving the data,
 *   a virtual data should be written to the SPI data register.
 * 
 *   Watch Variables:
 *                   - TxData_SPIA[] - SPIA data to send
 *                   - RxData_SPIA[] - SPIA received data
 *                   - TxData_SPIB[] - SPIB data to send
 *                   - RxData_SPIB[] - SPIB received data
 *
 *   Pin connectionL:
 *                   SPIA_CLK  - SPIB_CLK
 *                   SPIA_MOSI - SPIB_MISO
 *                   SPIA_CS   - SPIB_CS
 *
 * @note:
 *   1.This routine requires the chip to support two SPI channels.
 *
 */

/**
 * @examplespi tri wire
 * @brief  
 *   SPI ģʽ
 * 
 *   ˲Գʹ SPI ģ顣SPIA Ϊ豸SPIB Ϊ豸
 *   ֮ǰȽTALKλӻ֮ǰȽTALK㣻
 *   ӻ֮ǰȽTALKλ֮ǰȽTALK㣬ڽ֮ǰӦSPIݼĴдݡ
 *
 *   :
 *                   - TxData_SPIA[] - SPIA ͵
 *                   - RxData_SPIA[] - SPIA յ
 *                   - TxData_SPIB[] - SPIB ͵
 *                   - RxData_SPIB[] - SPIB յ
 *
 *   ӣ
 *                   SPIA_CLK  - SPIB_CLK
 *                   SPIA_MOSI - SPIB_MISO
 *                   SPIA_CS   - SPIB_CS
 *
 * @note:
 *   1.ҪоƬ֧·SPI
 *
 */


#include "spi_tri_wire.h"
#include "device.h"
#include "log.h"

int main(void)
{
    uint16_t i;

    volatile uint16_t TxData_SPIA[] = {0xFF, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F};
    volatile uint16_t RxData_SPIA[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    volatile uint16_t TxData_SPIB[] = {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F};
    volatile uint16_t RxData_SPIB[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    /** @brief Initialize device clock and peripherals */
    Device_init();

    __enable_irq();

    spi_master_init();
    spi_slave_init();

    DEVICE_DELAY_US(23);

    SPI_enableTalk(CONFIG_SPI_MASTER_BASE);
    SPI_disableTalk(CONFIG_SPI_SLAVE_BASE);
    for(i = 0; i < 16; i++)
    {
    	SPI_writeDataBlockingNonFIFO(CONFIG_SPI_MASTER_BASE, TxData_SPIA[i]);

    	RxData_SPIB[i] = SPI_readDataBlockingNonFIFO(CONFIG_SPI_SLAVE_BASE);

		if(RxData_SPIB[i] != TxData_SPIA[i])
		{
			ESTOP0;
		}
    }

    SPI_enableTalk(CONFIG_SPI_SLAVE_BASE);
    SPI_disableTalk(CONFIG_SPI_MASTER_BASE);
    for(i = 0; i < 16; i++)
    {
    	SPI_writeDataBlockingNonFIFO(CONFIG_SPI_SLAVE_BASE, TxData_SPIB[i]);

    	SPI_writeDataBlockingNonFIFO(CONFIG_SPI_MASTER_BASE, 0xFFFF);

    	RxData_SPIA[i] = SPI_readDataBlockingNonFIFO(CONFIG_SPI_MASTER_BASE);

    	if(RxData_SPIA[i] != TxData_SPIB[i])
		{
			ESTOP0;
		}
    }

    while(1);

    return 0;
}
