/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define NMI_TRIGGER_TYPE 	SYSCTL_NMI_CLOCKFAIL /* Define the NMI trigger type*/

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

volatile uint32_t nmi_count                = 0; // NMI count variable

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* NMI interrupt handler function */
void NMI_Handler(void)
{
	if(nmi_count == 0xFFFFFFFF){
		nmi_count =0x0;
	}else{
		nmi_count += 0xFF;
	}

	if (SysCtl_isNMIFlagSet(NMI_TRIGGER_TYPE))
	{
/* Restore external clock input */
#if (NMI_TRIGGER_TYPE == SYSCTL_NMI_CLOCKFAIL)
		/* Clear external clock counter */
		SysCtl_setX1CntClr();
		SysCtl_delay(10);

		/* Get the value of the external clock counter */
		uint32_t count = SysCtl_getX1Cnt();
		if(count < 0x7FF && count > 0)
		{
			/* Clear CMD reporting flag */
			SysCtl_setCmdStClr(1);
			SysCtl_setCmdStClr(0);
			SysCtl_delay(0x2000);
		}
#endif
	}

    SysCtl_delay(nmi_count);

    /* Flip GPIO */
    GPIO_togglePin(GPIO_PIN_LED2);

    /* Clear all NMI flags */
    SysCtl_clearAllNMIFlags();
}

/**
 * @brief  Initialize the NMI watchdog
 */
void NMI_Wdg_init(void)
{
	/* Block all NMI flags */
	SysCtl_maskNMIStatus(SYSCTL_NMI_FLAG_MASK);

	/* Clear all NMI flags */
	SysCtl_clearAllNMIFlags();

	/* Do not block NMI_TRIGGER-TYPE NMI flag */
	SysCtl_unmaskNMIStatus(NMI_TRIGGER_TYPE);

    /* Disable NMI reset request */
    SysCtl_disableNMIResetRequest();

    /* Register the NMI interrupt handler */
    Exception_Register_EXC(NMI_EXCn, (unsigned long)NMI_Handler);

    /* Enable global NMI interrupts */
    SysCtl_enableNMIGlobalInterrupt();
}

void GPIO_init(void)
{
    /* Set the pin to gpio function */
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setPinConfig(GPIO_CFG_LED2);

    /* Output level is low */
    GPIO_writePin(GPIO_PIN_LED1, 0);
    GPIO_writePin(GPIO_PIN_LED2, 0);

    /* Disable simulation function */
    GPIO_setAnalogMode(GPIO_PIN_LED1, GPIO_ANALOG_DISABLED);
    GPIO_setAnalogMode(GPIO_PIN_LED2, GPIO_ANALOG_DISABLED);

    /* Set pins to output mode */
    GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);
    GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/**
 * @brief  Main function
 * @param  None
 * @return None
 */
int main(void)
{
    __disable_irq(); /* Disable global interrupts */

#if (NMI_TRIGGER_TYPE == SYSCTL_NMI_CLOCKFAIL)
    /* Set observation pins */
    GPIO_setPinConfig(GPIO_16_XCLKOUT);
    /* Set observation clock */
    SysCtl_setObsSigSel(OBS_SIG_PLL_REF);
    /* 100 frequency division */
    SysCtl_setObsSigDiv(50);
#endif
    Device_init(); /* Initialize the device */

    GPIO_init();/* GPIO init */

    /* Initialize NMI watchdog */
    NMI_Wdg_init();

    while (1) {
        /* Delay for 100ms */
        DELAY_US(100*1000);
        GPIO_togglePin(GPIO_PIN_LED1);
        nmi_count=0;
    }

    /* This point will never be reached */
    for (;;) {
    }

    return 0;
}
