/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "driverlib.h"
#include "load_img.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"
#include "ipc.h"

#define CORE_LED_PIN            LED1_PIN
#define IPC_CMD_READ_MEM0   	0x1001
#define IPC_CMD_READ_MEM1   	0x1002
#define TEST_PASS0              0x5555
#define TEST_FAIL0              0xAAAA
#define TEST_PASS1              0x6666
#define TEST_FAIL1              0xBBBB

__SHARED_BSS__ volatile uint32_t cpu1Ticks;
__SHARED_BSS__ volatile uint32_t cpu2Ticks;
__SHARED_BSS__ volatile uint32_t readData0[10];

volatile uint32_t IPC_IRQFlag0 = FALSE;

#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER0_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    cpu1Ticks++;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void TIMER0_IRQHandler(void)
{
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    cpu1Ticks++;
}
#endif

void IPC_IRQHandler0(void)
{
    int i;
    uint32_t command, addr, data;
    bool status = false;

    //
    // Read the command
    //
    IPC_readCommand(IPC_CPU1_L_CPU2_R, IPC_FLAG1, IPC_ADDR_CORRECTION_ENABLE,
                    &command, &addr, &data);

    if(command == IPC_CMD_READ_MEM1)
    {
        status = true;

        //
        // Read and compare data
        //
        for(i=data; i>0; i--)
        {
            if(*((uint32_t *)addr + i) != i)
                status = false;
        }
    }

    //
    // Send response to C28x core
    //
    if(status)
    {
        IPC_sendResponse(IPC_CPU1_L_CPU2_R, TEST_PASS1);
    }
    else
    {
        IPC_sendResponse(IPC_CPU1_L_CPU2_R, TEST_FAIL1);
    }

    //
    // Acknowledge the flag
    //
    IPC_ackFlagRtoL(IPC_CPU1_L_CPU2_R, IPC_FLAG1);

    IPC_IRQFlag0 = TRUE;

}

void Timer_init(void)
{
    CPUTimer_init(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);      //register interrupt handler and setup interrupt mode
    Interrupt_enable(INT_TIMER0);
}

/* Private functions ---------------------------------------------------------*/
/**
 * @brief
 * @param  None
 * @retval None
 */
int main(void)
{
    uint32_t oldTicks;

    __disable_irq();

    Device_init();

    GPIO_enableWritePin(CORE_LED_PIN);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP300 IPC Project!\r\n");
    log_info("Core%d running @ %d MHz\r\n", __get_hart_id(), DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);

    bringup_cpu2();

    //
    // Enable IPC interrupts
    //
    IPC_registerInterrupt(IPC_CPU1_L_CPU2_R, IPC_INT1, IPC_IRQHandler0);

    __enable_irq();

    IPC_sync(IPC_CPU1_L_CPU2_R, IPC_FLAG31);

    //
    // Fill in the data to be sent
    //
    for (int i = 0; i < 10; i++) {
        readData0[i] = i;
    }

    //
    // Send a message without message queue
    // Length of the data to be read is passed as data.
    //
    IPC_sendCommand(IPC_CPU1_L_CPU2_R, IPC_FLAG0, IPC_ADDR_CORRECTION_ENABLE,
    				IPC_CMD_READ_MEM0, (uint32_t)readData0, 10);

    //
    // Wait for acknowledgment
    //
    IPC_waitForAck(IPC_CPU1_L_CPU2_R, IPC_FLAG0);

    //
    // Read response
    //
    if(IPC_getResponse(IPC_CPU1_L_CPU2_R) == TEST_PASS0)
    {
        log_info("CPU1 TEST_PASS\r\n");
    }
    else
    {
        log_info("CPU1 TEST_FAIL\r\n");
    }

    while (!IPC_IRQFlag0)
        ;

    Timer_init();

    oldTicks = cpu1Ticks;

    while (1) {
        if (cpu1Ticks != oldTicks) {
            oldTicks = cpu1Ticks;

            if ((oldTicks % 100) == 0) {
                bool ret = IPC_sendCommand(IPC_CPU1_L_CPU2_R, IPC_FLAG10, IPC_ADDR_CORRECTION_ENABLE, 1, 2, cpu1Ticks);

                log_debug("cpu1Ticks: %d cpu2Ticks: %d, send ipc cmd to cpu2 %s.\r\n", cpu1Ticks, cpu2Ticks, ret == true ? "ok" : "fail");

                GPIO_togglePin(CORE_LED_PIN);      //not thread safe
            }
        }

        if (IPC_isFlagBusyRtoL(IPC_CPU1_L_CPU2_R, IPC_FLAG11)) {
            uint32_t command, addr, data;

            IPC_readCommand(IPC_CPU1_L_CPU2_R, IPC_FLAG11, IPC_ADDR_CORRECTION_ENABLE, &command, &addr, &data);

            IPC_ackFlagRtoL(IPC_CPU1_L_CPU2_R, IPC_FLAG11);

            log_debug("get cpu2 ipc, cmd %d, addr %d, data %d.\r\n", command, addr, data);
        }
    }

    for(;;);

    return 0;
}


