/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* Includes ------------------------------------------------------------------*/
#include <sys/stat.h>

#include "device.h"
#include "board_cfg.h"
#include "load_img.h"
#include "log.h"
#include "OS.h"

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Main program
  * @param  None
  * @retval None
  */

volatile uint32_t tickCnt;

void Task_SciComm(void)
{
	taskEvent_t event;

    OSStartTimer(cPrioSciComm, 500, 1000);

    while(1) {
    	event = OSEventPend();

    	if(event & EVENT(OS_EVENT_TIMER)) {
    		log_info("this is Sci Comm task %d!\n", tickCnt);
    	}
    }
}

void Task_Data(void)
{
	taskEvent_t event;

    OSStartTimer(cPrioDataTask, 0, 1000);

    while(1) {
    	event = OSEventPend();

    	if(event & EVENT(OS_EVENT_TIMER)) {
    		log_info("this is Data task %d!\n", tickCnt);
    	}
    }
}

void Task_Idle(void)
{
    while(1)
    {
    }
}

__interrupt void Timer1_IRQHandler(void)
{
    CPUTimer_ClearInterruptStatus(CPUTIMER1_BASE);
    tickCnt += 1;
}

int main(void)
{
    uint32_t oldTicks;

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_info("Hello RV32 uCOS mod!\r\n");
    log_info("Core running @ %dMHz\r\n", DEVICE_SYSCLK_FREQ/1000000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);

    CPUTimer_init(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/100000);   //100kHz for test
    Interrupt_register(INT_TIMER1, Timer1_IRQHandler);
    Interrupt_enable(INT_TIMER1);

    SysTickInitFreq(DEVICE_APBCLK_FREQ);

    OSInit();

    OSTaskCreate(Task_SciComm, TaskStackSize, cPrioSciComm);
    OSTaskCreate(Task_Data, TaskStackSize, cPrioDataTask);
    OSTaskCreate(Task_Idle, 128, cPrioIdle);

    OSStart();

    for(;;);

    return 0;
}

