/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file     main.c
*   @brief    template main.c file
*   @details  toggle GPIO to drive LED, print output via UART.
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "driverlib.h"
#include "load_img.h"
#include "log.h"
#include "board_cfg.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

__SHARED_BSS__ volatile uint32_t cpu1Ticks;
__SHARED_BSS__ volatile uint32_t cpu2Ticks;

/* ========================================================================== */
/*                          External Function Prototypes                      */
/* ========================================================================== */
extern void Shared_show(void);

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief    CPUTimer0 IRQ handler
 */
#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER0_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    cpu1Ticks++;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void TIMER0_IRQHandler(void)
{
    SAVE_FPU_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    cpu1Ticks++;

    RESTORE_FPU_CONTEXT();
}
#endif

/*
 * @brief    CPUTimer0 initialization
 */
void Timer_init(void)
{
    CPUTimer_init(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);      //register interrupt handler and setup interrupt mode
    Interrupt_enable(INT_TIMER0);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
  * @brief  main function
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t oldTicks;

    __disable_irq();

    Device_init();

    GPIO_enableWritePin(GPIO_PIN_LED1);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP300 Dualcore Template Project!\r\n");
    log_info("Core%d running @ %d MHz\r\n", __get_hart_id(), DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("%d Cores, %d IRQn\r\n", CIDU_GetCoreNum(), CIDU_GetIntNum());

    bringup_cpu2();

    IPC_init(IPC_CPU1_L_CPU2_R);
    IPC_sync(IPC_CPU1_L_CPU2_R, IPC_FLAG0);

    Timer_init();

    __enable_irq();

    oldTicks = cpu1Ticks;

    while (1) {
        if (cpu1Ticks != oldTicks) {
            oldTicks = cpu1Ticks;

            if ((oldTicks % 100) == 0) {
                log_debug("cpu1Ticks: %d cpu2Ticks: %d\r\n", cpu1Ticks, cpu2Ticks);
                Shared_show();

                GPIO_togglePin(GPIO_PIN_LED1);      //not thread safe
            }
        }
    }

    for(;;);

    return 0;
}

#ifdef __cplusplus
}
#endif
