/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "sdfm.h"
#include "sdfm_pwm_sync_cpuread.h"

#define MAX_SAMPLES            1024
#define SDFM_INT_MASK          0x8000F000U
#define EPWM_TIMER_TBPRD       65534

int16_t  filter1Result[MAX_SAMPLES];
int16_t  filter2Result[MAX_SAMPLES];
int16_t  filter3Result[MAX_SAMPLES];
int16_t  filter4Result[MAX_SAMPLES];

volatile uint32_t MF1cnt = 0;
volatile uint32_t MF2cnt = 0;
volatile uint32_t MF3cnt = 0;
volatile uint32_t MF4cnt = 0;

void pinmux_init(void)
{
#if SDFM_BASE == SDFM1_BASE
    /* SDFM1 Pin Mux */
    /* SDFM1 SD1_C1 _GP17 */
    GPIO_setPinConfig(SD1_C1_PIN_MUX);
    GPIO_setPadConfig(SD1_C1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D1_GP16 */
    GPIO_setPinConfig(SD1_D1_PIN_MUX);
    GPIO_setPadConfig(SD1_D1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C2 _GP19 */
    GPIO_setPinConfig(SD1_C2_PIN_MUX);
    GPIO_setPadConfig(SD1_C2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D2 _GP32 */
    GPIO_setPinConfig(SD1_D2_PIN_MUX);
    GPIO_setPadConfig(SD1_D2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C3 _GP21 */
    GPIO_setPinConfig(SD1_C3_PIN_MUX);
    GPIO_setPadConfig(SD1_C3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D3 _GP20 */
    GPIO_setPinConfig(SD1_D3_PIN_MUX);
    GPIO_setPadConfig(SD1_D3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C4 _GP23 */
    GPIO_setPinConfig(SD1_C4_PIN_MUX);
    GPIO_setPadConfig(SD1_C4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C4_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D4 _GP22 */
    GPIO_setPinConfig(SD1_D4_PIN_MUX);
    GPIO_setPadConfig(SD1_D4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D4_PIN, GPIO_QUAL_ASYNC);

#elif SDFM_BASE == SDFM2_BASE
    /* SDFM2 Pin Mux */
    /* SDFM2 SD2_C1 _GP25 */
    GPIO_setPinConfig(SD2_C1_PIN_MUX);
    GPIO_setPadConfig(SD2_C1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D1_GP24 */
    GPIO_setPinConfig(SD2_D1_PIN_MUX);
    GPIO_setPadConfig(SD2_D1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C2 _GP27 */
    GPIO_setPinConfig(SD2_C2_PIN_MUX);
    GPIO_setPadConfig(SD2_C2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D2 _GP26 */
    GPIO_setPinConfig(SD2_D2_PIN_MUX);
    GPIO_setPadConfig(SD2_D2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C3 _GP29 */
    GPIO_setPinConfig(SD2_C3_PIN_MUX);
    GPIO_setPadConfig(SD2_C3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D3 _GP28 */
    GPIO_setPinConfig(SD2_D3_PIN_MUX);
    GPIO_setPadConfig(SD2_D3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C4 _GP31 */
    GPIO_setPinConfig(SD2_C4_PIN_MUX);
    GPIO_setPadConfig(SD2_C4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C4_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D4 _GP30 */
    GPIO_setPinConfig(SD2_D4_PIN_MUX);
    GPIO_setPadConfig(SD2_D4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D4_PIN, GPIO_QUAL_ASYNC);
#endif
}

void sdfm_init(uint32_t base)
{
    uint16_t  hlt, llt;

    /* Configure SDFM type to 0 and see if data ack generated SDINT. */
    SysCtl_configureType(0);

    /* Input Control Unit */

    /* Set modulator clock mode: Modulator Clock rate = Modulator data rate */
    SDFM_setupModulatorClock(base, SDFM_FILTER_1,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_2,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_3,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_4,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);

    /* Selects clock source for SDFM channels. */
    SDFM_selectClockSource(base, SDFM_FILTER_1, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_2, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_3, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_4, SDFM_CLK_SOURCE_SD1_CLK);

    /* Comparator Unit */

    /* over and under value threshold settings */
    hlt = 0x7FFF;
    llt = 0x0000;

    /* Configure Comparator Unit's comparator filter type and comparator's
     * OSR value, higher threshold, lower threshold
     */
    SDFM_configComparator(base,
                         (SDFM_FILTER_1 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_3 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_4 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);

    /* Data Filter Unit */

    /* filter type, OSR value, data shift bit values, FIFO waterline and enable
     * or disable data filter
     */
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_1 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(0x0007)));

    SDFM_configDataFilter(base,
                         (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(0x0007)));

    SDFM_configDataFilter(base,
                         (SDFM_FILTER_3 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(0x0007)));

    SDFM_configDataFilter(base,
                         (SDFM_FILTER_4 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(0x0007)));

    /* Enable Master filter bit:
     * Unless this bit is set none of the filter modules can be enabled.
     * All the filter modules are synchronized when master filter
     * bit is enabled after individual filter modules are enabled.
     */
    SDFM_enableMainFilter(base);

    SDFM_enableExternalReset(base, SDFM_FILTER_1);
    SDFM_enableExternalReset(base, SDFM_FILTER_2);
    SDFM_enableExternalReset(base, SDFM_FILTER_3);
    SDFM_enableExternalReset(base, SDFM_FILTER_4);

    /* Set the PWM sync mode. */
    SDFM_setPWMSyncSource(base, SDFM_FILTER_1, SDFM_SYNC_PWM1_SOCA);
    SDFM_setPWMSyncSource(base, SDFM_FILTER_2, SDFM_SYNC_PWM1_SOCA);
    SDFM_setPWMSyncSource(base, SDFM_FILTER_3, SDFM_SYNC_PWM1_SOCA);
    SDFM_setPWMSyncSource(base, SDFM_FILTER_4, SDFM_SYNC_PWM1_SOCA);

    initEPWM(EPWM1_BASE);

    /* Enable interrupts */

    /* Following SDFM interrupts can be enabled / disabled using this function.
     * Enable / disable comparator high threshold
     * Enable / disable comparator low threshold
     * Enable / disable modulator clock failure
     * Enable / disable data filter acknowledge
     */
    SDFM_enableInterrupt(base, SDFM_FILTER_1,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);
    SDFM_enableInterrupt(base, SDFM_FILTER_2,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);
    SDFM_enableInterrupt(base, SDFM_FILTER_3,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);
    SDFM_enableInterrupt(base, SDFM_FILTER_4,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);

    SDFM_disableInterrupt(base, SDFM_FILTER_1,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_2,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_3,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_4,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));

    SDFM_disableInterrupt(base, SDFM_FILTER_1, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_2, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_3, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_4, SDFM_MODULATOR_FAILURE_INTERRUPT);

    SDFM_disableInterrupt(base, SDFM_FILTER_1, SDFM_FIFO_OVERFLOW_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_2, SDFM_FIFO_OVERFLOW_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_3, SDFM_FIFO_OVERFLOW_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_4, SDFM_FIFO_OVERFLOW_INTERRUPT);

    while ((HWREGH(EPWM1_BASE + EPWM_O_TBCTR)) < 550);

    /* Enable master interrupt so that any of the filter interrupts can trigger
     * by SDFM interrupt to CPU.
     */
    SDFM_enableMainInterrupt(base);
}

/* sdfm1ISR - SDFM 1 ISR */

void sdfm1ISR(void)
{
    static uint16_t loopCounter1 = 0;

    /* Wait for result from all the filters (SDIFLG) */
    while ((HWREG(SDFM1_BASE + SDFM_O_SDIFLG) & SDFM_INT_MASK) != SDFM_INT_MASK);

    /* Reset the loop counter */
    if (loopCounter1 >= MAX_SAMPLES)
    {
        loopCounter1 = 0;
    }

    /* Read each SDFM filter output and store it in respective filter
     * result array */
    filter1Result[loopCounter1] = (int16_t)(SDFM_getFilterData(
        SDFM1_BASE, SDFM_FILTER_1) >> 16U);
    filter2Result[loopCounter1] = (int16_t)(SDFM_getFilterData(
        SDFM1_BASE, SDFM_FILTER_2) >> 16U);
    filter3Result[loopCounter1] = (int16_t)(SDFM_getFilterData(
        SDFM1_BASE, SDFM_FILTER_3) >> 16U);
    filter4Result[loopCounter1++] = (int16_t)(SDFM_getFilterData(
        SDFM1_BASE, SDFM_FILTER_2) >> 16U);

    /* Clear SDFM flag register */
    SDFM_clearInterruptFlag(SDFM1_BASE, SDFM_INT_MASK);
}

/* sdfm2ISR - SDFM 2 ISR */

void sdfm2ISR(void)
{
    static uint16_t loopCounter2 = 0;

    /* Wait for result from all the filters (SDIFLG) */
    while ((HWREG(SDFM2_BASE + SDFM_O_SDIFLG) & SDFM_INT_MASK) != SDFM_INT_MASK);

    /* Reset the loop counter */
    if (loopCounter2 >= MAX_SAMPLES)
    {
        loopCounter2 = 0;
    }

    /* Read each SDFM filter output and store it in respective filter
     * result array */
    filter1Result[loopCounter2] = (int16_t)(SDFM_getFilterData(
        SDFM2_BASE, SDFM_FILTER_1) >> 16U);
    filter2Result[loopCounter2] = (int16_t)(SDFM_getFilterData(
        SDFM2_BASE, SDFM_FILTER_2) >> 16U);
    filter3Result[loopCounter2] = (int16_t)(SDFM_getFilterData(
        SDFM2_BASE, SDFM_FILTER_3) >> 16U);
    filter4Result[loopCounter2++] = (int16_t)(SDFM_getFilterData(
        SDFM2_BASE, SDFM_FILTER_4) >> 16U);

    /* Clear SDFM flag register */
    SDFM_clearInterruptFlag(SDFM2_BASE, SDFM_INT_MASK);
}

/* initEPWM - Initialize specified EPWM settings */

void initEPWM(uint32_t epwmInstance)
{
    uint16_t compCVal, compDVal;

    compCVal = 200;
    compDVal = 200;

    DINT;

    /* It can be observed through an oscilloscope. */
    GPIO_setPinConfig(GPIO_0_EPWM1_A);
    GPIO_setPadConfig(0, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(0, GPIO_QUAL_SYNC);

    GPIO_setPinConfig(GPIO_1_EPWM1_B);
    GPIO_setPadConfig(1, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(1, GPIO_QUAL_SYNC);

    /* Disable sync(Freeze clock to PWM as well) */
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /**
     * Setup TBCLK: Configure timer period = 801 TBCLKs, phase = 0 &
     * clear counter
     */
    EPWM_setTimeBasePeriod(epwmInstance, EPWM_TIMER_TBPRD);
    EPWM_setPhaseShift(epwmInstance, 0U);
    EPWM_setTimeBaseCounter(epwmInstance, 0U);

    /* Set CMPA, CMPB, CMPC & CMPD values */
    EPWM_setCounterCompareValue(epwmInstance, EPWM_COUNTER_COMPARE_C, compCVal);
    EPWM_setCounterCompareValue(epwmInstance, EPWM_COUNTER_COMPARE_D, compDVal);
    EPWM_setCounterCompareValue(epwmInstance, EPWM_COUNTER_COMPARE_A, compCVal);
    EPWM_setCounterCompareValue(epwmInstance, EPWM_COUNTER_COMPARE_B, compDVal);

    /* Setup counter mode */
    EPWM_setTimeBaseCounterMode(epwmInstance, EPWM_COUNTER_MODE_UP);
    EPWM_setClockPrescaler(epwmInstance,
                          EPWM_CLOCK_DIVIDER_1,
                          EPWM_HSCLOCK_DIVIDER_1);

    /**
     * Set actions:
     * Toggle PWMxA on event A, up-count
     * Toggle PWMxB on event A, up-count
     */
    EPWM_setActionQualifierAction(EPWM1_BASE,
                                 EPWM_AQ_OUTPUT_A,
                                 EPWM_AQ_OUTPUT_TOGGLE,
                                 EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

    EPWM_setActionQualifierAction(EPWM1_BASE,
                                 EPWM_AQ_OUTPUT_B,
                                 EPWM_AQ_OUTPUT_TOGGLE,
                                 EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);

    /* Enable sync and clock to PWM */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}
