/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file     clark.h
 *   @brief
 *
 */

#ifndef _CLARKE_H
#define _CLARKE_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#if __ENABLE_NICE
#include "gs32_math.h"
#else
#include "math.h"
#endif
#include "IQmathLib.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define INVSQRT3 _IQ(0.5773503f)

/*
 * @brief		Default initalizer for the CLARKE object.
 */
#define CLARKE_DEFAULTS \
    { 0,                \
      0,                \
      0,                \
      0,                \
      0 }

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/*
 * @brief		 Clarke Transform Parameters for 3 phase stator current
 */
typedef struct
{
    _iq As;    // Input: phase-a stator variable
    _iq Bs;    // Input: phase-b stator variable
    _iq Cs;    // Input: phase-c stator variable
    _iq Alpha; // Output: stationary d-axis stator variable
    _iq Beta;  // Output: stationary q-axis stator variable
} CLARKE;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/*
 * @brief		Clarke transformation calculation
 */
static inline void Clarke_calc(register CLARKE *v) {
	#if __ENABLE_NICE //DSP instruction
	__gs_dsp_mv3_f2mpr_r0(v->As, v->Bs, v->Cs); //1 cycle
	v->Alpha = v->As;
	v->Beta = __vmulsum_v3_const(1);
	#else
    v->Alpha = v->As;
    v->Beta = _IQmpy((v->As + _IQmpy2(v->Bs)), _IQ(INVSQRT3));
	#endif
}

#ifdef __cplusplus
}
#endif

#endif /* _CLARKE_H */
