/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file     ipark.h
 *   @brief
 *
 */

#ifndef _IPARK_H
#define _IPARK_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#if __ENABLE_NICE
#include "gs32_math.h"
#else
#include "math.h"
#endif
#include "IQmathLib.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/*
 * @brief		  Default initalizer for the IPARK object.
 */
#define IPARK_DEFAULTS \
    {                  \
        0,             \
            0,         \
            0,         \
            0,         \
            0,         \
    }

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/*
 * @brief		  Inverse Park Transformation Parameters
 */
typedef struct
{
    _iq ds;  // Output: stationary d-axis stator variable
    _iq qs;  // Output: stationary q-axis stator variable
    _iq ang; // Input: rotating angle (pu)
    _iq de;  // Input: rotating d-axis stator variable
    _iq qe;  // Input: rotating q-axis stator variable
} IPARK;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/*
 * @brief		  Inverse Park transform calculation
 */
static inline void IPARK_calc(IPARK *v) {
    register _iq cos_ang, sin_ang;

	#if __ENABLE_NICE //DSP instruction
    sin_ang = __sinpuf32(v->ang);
    cos_ang = __cospuf32(v->ang);

    __mv2_f2mpr(0, 0, 0, 1, v->de, v->qe);
    v->ds = __vmulsum2_f2_r1_neg2(cos_ang, sin_ang);
    v->qs = __vmulsum2_f2_r1(sin_ang, cos_ang);
	#else
    sin_ang = sinf(v->ang * TWO_PI);
    cos_ang = cosf(v->ang * TWO_PI);

    v->ds = _IQmpy(v->de, cos_ang) - _IQmpy(v->qe, sin_ang);
    v->qs = _IQmpy(v->qe, cos_ang) + _IQmpy(v->de, sin_ang);
	#endif

}

#ifdef __cplusplus
}
#endif

#endif /* _IPARK_H */
