/*
 * Copyright (c) 2019 Nuclei Limited. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/******************************************************************************
 * @file startup_gs32_dsp.S
 * @brief Boot ROM start up.
 * @version V1.00
 * @date 1. Jul 2024
 *
 * commit history:
 * 2024/07/1, Jihong, add DSP BootROM code.
 ******************************************************************************/
OUTPUT_ARCH( "riscv" )

ENTRY( _start )

MEMORY
{
	sysmem1	(rxa!w) : ORIGIN = 0x1FFE0000, LENGTH = 16K
	dlm		(wxa!r) : ORIGIN = 0x20140000, LENGTH = 64K
	ilm		(wxa!r) : ORIGIN = 0x20100000, LENGTH = 256K
}

REGION_ALIAS("SYSMEM1", sysmem1)
REGION_ALIAS("DLM", dlm)
REGION_ALIAS("ILM", ilm)

SECTIONS
{
	/* To provide symbol __STACK_SIZE, __HEAP_SIZE and __SMP_CPU_CNT */
	PROVIDE(__STACK_SIZE = 2K);
	PROVIDE(__HEAP_SIZE = 1K);
	PROVIDE(__SMP_CPU_CNT = 1);
	__TOT_STACK_SIZE = __STACK_SIZE * __SMP_CPU_CNT;

	.init :
	{
		KEEP (*(SORT_NONE(.init)));
		. = ALIGN(16);
	} >SYSMEM1 AT>SYSMEM1

	.text :
	{
		. = ALIGN(4);
		__start_text__ = .;
		*(.text.unlikely .text.unlikely.*)
		*(.text.startup .text.startup.*)
		*(.text .text.*)
		*(.boot_text_section)
		*(.gnu.linkonce.t.*)
		__end_text__ = .;
		. = ALIGN(16);
	} >SYSMEM1 AT>SYSMEM1

	.ilm :
	{
		. = ALIGN(4);
		*(.RamFunc .RamFunc.*)
		*(.IlmFunc .Ilmfunc.*)
		*(.boot_rom_to_ram_section)
		. = ALIGN(16);
	} >ILM AT>SYSMEM1

	.data :
	{
		. = ALIGN(8);
		KEEP(*(.data.ctest*))
		KEEP(*(.data.ctest*))
		*(.data .data.*)
		*(.gnu.linkonce.d.*)
		. = ALIGN(8);
		*(.sdata .sdata.* .sdata*)
		*(.gnu.linkonce.s.*)
		/* readonly data placed in RAM for access speed */
		. = ALIGN(8);
		*(.srodata.cst16)
		*(.srodata.cst8)
		*(.srodata.cst4)
		*(.srodata.cst2)
		*(.srodata .srodata.*)
		*(.rdata)
		*(.rodata .rodata.*)
		*(.gnu.linkonce.r.*)
		*(.boot_data_section)
		/* below sections are used for rt-thread */
		. = ALIGN(4);
		__rt_init_start = .;
		KEEP(*(SORT(.rti_fn*)))
		__rt_init_end = .;
		. = ALIGN(4);
		__fsymtab_start = .;
		KEEP(*(FSymTab))
		__fsymtab_end = .;
		. = ALIGN(4);
		__vsymtab_start = .;
		KEEP(*(VSymTab))
		__vsymtab_end = .;
		. = ALIGN(16);
	} >DLM AT>SYSMEM1

	.tdata :
	{
		. = ALIGN(8);
		PROVIDE( __tls_base = . );
		*(.tdata .tdata.* .gnu.linkonce.td.*)
		. = ALIGN(16);
	} >DLM AT>SYSMEM1

	.bss (NOLOAD):
	{
		. = ALIGN(8);
		__start_bss__ = .;
		*(.sbss*)
		*(.gnu.linkonce.sb.*)
		*(.bss .bss.*)
		PROVIDE( __global_pointer$ = . + 0x800 );
		*(.gnu.linkonce.b.*)
		*(COMMON)
		*(.boot_bss_section)
		. = ALIGN(4);
		__end_bss__ = .;
	} >DLM AT>DLM

	.copy.table :
	{
		. = ALIGN(4);
		__copy_table_start__ = .;

		LONG (LOADADDR(.data))
		LONG (ADDR(.data))
		LONG (SIZEOF(.data) / 4)

		LONG (LOADADDR(.tdata))
		LONG (ADDR(.tdata))
		LONG (SIZEOF(.tdata) / 4)

		LONG (LOADADDR(.ilm))
		LONG (ADDR(.ilm))
		LONG (SIZEOF(.ilm) / 4)

		__copy_table_end__ = .;
		. = ALIGN(16);
	} > SYSMEM1

	PROVIDE( __copy_table_start = __copy_table_start__ );
	PROVIDE( __copy_table_end = __copy_table_end__ );

	.zero.table :
	{
		. = ALIGN(4);
		__zero_table_start__ = .;

		LONG (ADDR(.bss))
		LONG (SIZEOF(.bss) / 4)

		__zero_table_end__ = .;
		. = ALIGN(16);
	} > SYSMEM1

	PROVIDE( __zero_table_start = __zero_table_start__ );
	PROVIDE( __zero_table_end = __zero_table_end__ );

	/* Nuclei C Runtime Library requirements:
	* 1. heap need to be align at 16 bytes
	* 2. __heap_start and __heap_end symbol need to be defined
	* 3. reserved at least __HEAP_SIZE space for heap
	*/
	.heap (NOLOAD) : ALIGN(16)
	{
		. = ALIGN(16);
		PROVIDE( __heap_start = . );
		. += __HEAP_SIZE;
		. = ALIGN(16);
		PROVIDE( __heap_limit = . );
	} >DLM AT>DLM

	.stack ORIGIN(DLM) + LENGTH(DLM) - __TOT_STACK_SIZE (NOLOAD) :
	{
		. = ALIGN(16);
		PROVIDE( _heap_end = . );
		PROVIDE( __heap_end = . );
		PROVIDE( __StackLimit = . );
		PROVIDE( __StackBottom = . );
		. += __TOT_STACK_SIZE;
		. = ALIGN(16);
		PROVIDE( __StackTop = . );
		PROVIDE( _sp = . );
	} >DLM AT>DLM
}
