/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __FDP_H__
#define __FDP_H__

#include <stdbool.h>

#include "fdp_port.h"
#include "fdp_watchdog.h"
#include "fdp_defconfig.h"
#include "fdp_jump.h"
#include "fdp_app_info.h"
#include "fdp_err.h"

/**
 * @brief FDP frame types.
 * 
 */
/* Bytestream(such as uart). */
#define FDP_FRAME_CHAR			0
/* Packet(such as Ethernet, CAN, etc.). */
#define FDP_FRAME_PACKET		1

/**
 * @brief FDP command tyeps.
 * 
 */
/* Select download device. */
#define FDP_CMD_DOWNLOADER		0xCF
/* The host sends image update request frame to device. */
#define FDP_CMD_START			0xDA
/* The host sends vender ID verification frame to device. */
#define FDP_CMD_VENDER_CHECK	0xDB
/* The host sends a request to obtain
the maximum frame length of the device to the device. */
#define FDP_CMD_FRAME_SIZE		0xDC
/* The host sends Image size to device. */
#define FDP_CMD_FILE_SIZE		0xDD
/* The host sends the address of the image to be written to the device. */
#define FDP_CMD_GET_ADDRESS		0xD5
/* The host sends the initialization memory command to the device. */
#define FDP_CMD_INIT_MEM		0xD7
/**
 * The host sends transmission data request
 * frame to device.
 */
#define FDP_CMD_TRANSFER_REQ	0xDE
/**
 * The host sends an image transmission completion
 * request frame to the device.
 */
#define FDP_CMD_TRANSFER_END	0xDF
/* The host sends an image verification frame to device. */
#define FDP_CMD_TRANSFER_CHECK	0xBE
/* Check if the image is correct. */
#define FDP_CMD_CHECK_IMAGE		0xBF
/* Check if it is necessary to release core. */
#define FDP_CMD_RELEASE_CORE	0xC0

/**
 * If the host and device communication is successfully,
 * the host or device will sends a successful ACK.
 */
#define FDP_CMD_ACK_SUCCESS		0xEA
/**
 * If communication between the host and device fails,
 * the host or device will send a failed ACK.
 */
#define FDP_CMD_ACK_ERROR		0xEB

/**
 * @brief FDP frame direction.
 *
 */
/* Message initiated by the host. */
#define FDP_HOST				0x5A
/* Message initiated by the device. */
#define FDP_DEVICE				0xB5

typedef enum {
	/* Program running in command phase. */
	FDP_PRG_CMD_PHASE			= 0,
	/* Program running in data dispose phase. */
	FDP_PRG_DATA_PHASE			= 1,
	/* Program running in check image phase. */
	FDP_PRG_CHECK_PHASE			= 2,
} fdp_program_stage_t;

/**
 * @brief The length of commands frame.
 * 
 */
#define FDP_CMD_SELECT_DOWNLOAD_LENGTH	4U
#define FDP_CMD_START_LENGTH			4U
#define FDP_CMD_VENDER_CHECK_LENGTH		8U
#define FDP_CMD_FRAME_SIZE_LENGTH		8U
#define FDP_CMD_FILE_SIZE_LENGTH		8U
#define FDP_CMD_STORAGE_ADDR_LENGTH		8U
#define FDP_CMD_INIT_MEMORY_LENGTH		8U
#define FDP_CMD_TRANSFER_REQ_LENGTH		8U
#define FDP_CMD_TRANSFER_END_LENGTH		4U
#define FDP_CMD_TRANSFER_CHECK_LENGTH	8U
#define FDP_CMD_CHECK_IMAGE_LENGTH		8U
#define FDP_CMD_MAX_OF_LENGTH			8U
#define FDP_CMD_RELEASE_CORE_LENGTH		8U
#define FDP_CMD_ACK_LENGTH				4U
#define FDP_CMD_MAX_LENGTH				8U

typedef enum fdp_update_program_flow {
	/* Wait for update image. */
	FDP_BOOT_FLOW_WAIT_FOR_UPTATE		= 0,
	/* Start update image. */
	FDP_BOOT_FLOW_UPDATE				= 1,
	/* Check if the vender is correct. */
	FDP_BOOT_FLOW_CHECK_VENDER			= 2,
	/* The host requests the maximum frame length to be sent from the device. */
	FDP_BOOT_FLOW_SEND_MAX_FRAME_SIZE	= 3,
	/* The host sends the size of the image to be written to the device. */
	FDP_BOOT_FLOW_GET_IMAGE_SIZE		= 4,
	/* The host sends the address of the image to be written to the device. */
	FDP_BOOT_FLOW_GET_STORAGE_ADDR		= 5,
	/* The host sends the initialization memory command to the device. */
	FDP_BOOT_FLOW_INIT_MEM				= 6,
	/* The host sends transmission data request frame to device. */
	FDP_BOOT_FLOW_GET_IMAGE_DATA		= 7,
	/* The host sends an image transmission completion request frame to the device. */
	FDP_BOOT_FLOW_CHECK_TRANSFER		= 8,
	/* The image received. */
	FDP_BOOT_FLOW_DOWNLOAD_END			= 9,
	/* The host sends an end of image transfer frame to device. */
	FDP_BOOT_FLOW_UPDATE_END			= 10,
	/* Check if the image is correct. */
	FDP_BOOT_FLOW_CHECK_IMAGE			= 11,
	/* Check if release core. */
	FDP_BOOTL_FLOW_RELEASE_CORE			= 12,
} fdp_update_program_flow_t;

/**
 * @brief Record FDP status.
 * 
 * @param download_ready Check if the Image download is ready.
 * @param mem_init_ready Check if the initialization memory is ready.
 * @param transfer_done Check if the transmission is complete.
 * @param image_ready Check if the image is correct.
 * @param program_flow Save the current program running flow(@fdp_update_program_flow_t).
 * @param program_stage Save the current program running status(@fdp_program_stage_t).
 * @param dev_type The type of device.
 * @param crc_val Save data phase verification value.
 * @param received_index The number of correct data received.
 * @param max_frame_size Allow the upper computer to transmit the maximum frame length.
 * @param frame_size The length of data in data phase.
 * @param file_size The size of image.
 * @param address Record to the address of the image to be written.
 * @param received_data_size Record to the number of data received.
 * @param rx_read Record the read index of the receiving cache area.
 * @param rx_write Record the write index of the receiving cache area.
 * @param watchdog_timeout Save the watchdog timeout value.
 */
typedef struct fdp_record_stat {
	bool download_ready;
	bool mem_init_ready;
	bool transfer_done;
	bool image_ready;
	bool fdp_done;
	fdp_update_program_flow_t program_flow;
	fdp_program_stage_t program_stage;
	fdp_device_type_t dev_type;
	uint32_t crc_val;
	uint32_t received_index;
	uint32_t max_frame_size;
	uint32_t frame_size;
	uint32_t file_size;
	uint32_t address;
	uint32_t received_data_size;
	uint32_t rx_read;
	uint32_t rx_write;
	uint32_t watchdog_timeout;
	uint32_t release_core;
} fdp_record_stat_t;

/**
 * @brief Initialize FDP.
 *
 * @param type Type of communication equipment.
 * @return int If return value is less than 0, it is an error, else success.
 */
fdp_record_stat_t *fdp_init(fdp_device_type_t type);

/**
 * @brief Initialize all communication equipment.
 *
 * @return int If return value is less than 0, it is an error, else success.
 */
int fdp_init_all_device(void);

int fdp_reset_all_device(void);

/**
 * @brief Wait for the first device to send ack.
 *
 * @param type Type of communication equipment.
 * @return int If return value is less than 0, it is an error, else success.
 */
int fdp_wait_first_device_ack(fdp_device_type_t *type);

/**
 * @brief Reset fdp status record value.
 *
 */
void fdp_reset_fdp(void);

/**
 * @brief Device replies to the upper computer.
 *
 * @param type Type of communication equipment.
 * @param ack True indicates success, false indicates failure.
 * @return int If return value is less than 0, it is an error, else success.
 */
int fdp_ack_to_host(fdp_device_type_t type, bool ack);

/**
 * @brief Recive data to buffer.
 * The function will callback user defined @fdp_receive function.
 * 
 * @return int return non-zero value means receive data fail, else success.
 */
int fdp_receive_data_to_buffer(void);

/**
 * @brief FDP download image function.
 * The function will excute the download process.
 * 
 * @param type Communication peripherals type.
 * @return int return non-zero value means download fail, else success.
 */
int fdp_download(fdp_device_type_t type);

#endif
