/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <string.h>

#include "fdp_err.h"
#include "fdp_storage_port.h"
#include "fdp_cache.h"
#include "fdp_defconfig.h"
#include "boot_flash.h"

/**
 * @brief User define include header file.
 * 
 */


/**
 * @brief User define macro.
 *
 */


int fdp_storage_dev_init(uint32_t start_addr, uint32_t size)
{
	boot_flash_disable_register_write_protect(BOOT_FLASH_EFC0_ADDR);
	boot_flash_disable_register_write_protect(BOOT_FLASH_EFC1_ADDR);

	boot_flash_disable_fast_program_mode(BOOT_FLASH_EFC0_ADDR);
	boot_flash_disable_fast_program_mode(BOOT_FLASH_EFC1_ADDR);

	boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);
	boot_flash_clear_efc_status(BOOT_FLASH_EFC1_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

	return FDP_SUCCESS;
}

int fdp_storage_dev_erase(uint32_t start_addr, uint32_t size)
{
	boot_flash_user_opt_info_t info;
	boot_flash_efc_status_t bank0_status;
	boot_flash_efc_status_t bank1_status;
	boot_flash_addr_info_t addr_info;
	uint32_t sector_size;
	uint32_t current;

	boot_flash_get_user_opt_infomation(BOOT_FLASH_EFC0_ADDR, &info);

	/* check if the bank mode is dual bank mode. */
	if (info.boot_dual_bank_mode)
		sector_size = BOOT_FLASH_DUAL_BANK_SECTOR_SIZE;
	else
		sector_size = BOOT_FLASH_SINGLE_BANK_SECTOR_SIZE;

	for (current = start_addr; current <= start_addr + size; current += sector_size) {

		/* clear flash efc status */
		boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);
		boot_flash_clear_efc_status(BOOT_FLASH_EFC1_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

		/* erase flash sector */
		boot_flash_erase_main_flash_sector(current);

		/* get flash information by address */
		boot_flash_get_addr_info(current, &addr_info);

		if (addr_info.bank_mode) {
			do {
				boot_flash_get_efc_status(addr_info.flash_efc_addr, &bank0_status);
			} while (bank0_status.efc_busy);
		} else {
			do {
				boot_flash_get_efc_status(BOOT_FLASH_EFC0_ADDR, &bank0_status);
				boot_flash_get_efc_status(BOOT_FLASH_EFC1_ADDR, &bank1_status);
			} while (bank0_status.efc_busy || bank1_status.efc_busy);
		}
	}

	return FDP_SUCCESS;
}

int fdp_storage_dev_write(uint8_t *buffer, uint32_t start_addr, uint32_t len)
{
	boot_flash_efc_status_t bank_status;
	boot_flash_addr_info_t addr_info;
	uint32_t current_addr;

	for (current_addr = start_addr; current_addr < start_addr + len;
		current_addr += BOOT_FLASH_MIN_WRITE_SIZE, buffer += BOOT_FLASH_MIN_WRITE_SIZE) {

		/* clear flash efc status */
		boot_flash_clear_efc_status(addr_info.flash_efc_addr, BOOT_FLASH_ALL_STATUS_MASK);

		/* write data to flash */
		boot_flash_write_main_flash(current_addr, buffer);

		/* get flash information by address */
		boot_flash_get_addr_info(current_addr, &addr_info);

		do {
			boot_flash_get_efc_status(addr_info.flash_efc_addr, &bank_status);
		} while (bank_status.efc_busy);
	}

	return FDP_SUCCESS;
}

int fdp_storage_dev_read(uint8_t *buffer, uint32_t start_addr, uint32_t len)
{
	if (buffer == NULL)
		return -FDP_ERRINPARAM;

	memcpy((void *)buffer, (void *)(uintptr_t)start_addr, len);

	return FDP_SUCCESS;
}

int fdp_security_mem_check(uintptr_t addr, uint32_t size)
{
	if (addr >= BANK_MIN_ADDR && addr < BANK_MAX_ADDR) {

		if (addr + size > BANK_MAX_ADDR)
			return -FDP_ERRMEMO;

		return FDP_SUCCESS;
	} else
		return -FDP_ERRMEMO;
}
