/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

OUTPUT_ARCH( "riscv" )

ENTRY( _start )

PROVIDE(__STACK_SIZE = 2K);

MEMORY
{
	sysmem		(rxa!w) : ORIGIN = 0x1FFE0000, LENGTH = 16K
	ilm       	(wxa!r) : ORIGIN = 0x20100000, LENGTH = 64K
	dlm       	(wxa!r) : ORIGIN = 0x20120000, LENGTH = 16K-__STACK_SIZE
	dlm_stk   	(wxa!r) : ORIGIN = 0x20120000 + 16K -__STACK_SIZE, LENGTH = __STACK_SIZE
}

REGION_ALIAS("SYSMEM", sysmem)
REGION_ALIAS("ILM", ilm)
REGION_ALIAS("DLM", dlm)
REGION_ALIAS("STACK", dlm_stk)

SECTIONS
{
	/* To provide symbol __STACK_SIZE, __HEAP_SIZE and __SMP_CPU_CNT */
	PROVIDE(__HEAP_SIZE = 1K);
	PROVIDE(__SMP_CPU_CNT = 1);
	__TOT_STACK_SIZE = __STACK_SIZE * __SMP_CPU_CNT;

	.rela.init : { *(.rela.init) }

	.init :
	{
		KEEP (*(SORT_NONE(.init)))
		. = ALIGN(16);
	} >SYSMEM AT>SYSMEM

	.rela.dyn : { *(.rela.dyn) }    /* add to avoid orphan sections warnings */
	.rela.text : { *(.rela.text*) } /* add to avoid orphan sections warnings */

	.text :
	{
		*(.text.unlikely .text.unlikely.*)
		*(.text.startup .text.startup.*)
		*(.text .text.*)
		*(.gnu.linkonce.t.*)
		*(.rom_data)
		*(.boot_text_section)
		. = ALIGN(16);
	} >SYSMEM AT>SYSMEM

	.copy.table :
	{
		. = ALIGN(4);
		__copy_table_start__ = .;

		LONG (LOADADDR(.data))
		LONG (ADDR(.data))
		LONG (SIZEOF(.data) / 4)

		LONG (LOADADDR(.ilm))
		LONG (ADDR(.ilm))
		LONG (SIZEOF(.ilm) / 4)

		__copy_table_end__ = .;

		. = ALIGN(16);
	} > SYSMEM

	PROVIDE( __copy_table_start = __copy_table_start__ );
	PROVIDE( __copy_table_end = __copy_table_end__ );

	.zero.table :
	{
		. = ALIGN(4);
		__zero_table_start__ = .;

		LONG (ADDR(.bss))
		LONG (SIZEOF(.bss) / 4)

		LONG (ADDR(.ilm_bss))
		LONG (SIZEOF(.ilm_bss) / 4)

		__zero_table_end__ = .;

		. = ALIGN(16);
	} > SYSMEM

	PROVIDE( __zero_table_start = __zero_table_start__ );
	PROVIDE( __zero_table_end = __zero_table_end__ );

	PROVIDE( _text_lma = LOADADDR(.text) );
	PROVIDE( _text = ADDR(.text) );
	PROVIDE (_etext = .);
	PROVIDE (__etext = .);
	PROVIDE (etext = .);

	.ilm :
	{
		. = ALIGN(4);
		*(.RamFunc)
		*(.RamFunc.*)
		*(.IlmFunc)
		*(.IlmFunc.*)
		*(.IlmData)
		*(.IlmData.*)
		*(.boot_rom_to_ram_section)

		. = ALIGN(16);
	} >ILM AT>SYSMEM

	.ilm_bss (NOLOAD) :
	{
		. = ALIGN(4);
		*(.ilmbss)
		*(.ilmbss.*)
		. = ALIGN(4);
	} >ILM

	PROVIDE( _ilm_lma = LOADADDR(.ilm) );
	PROVIDE( _ilm = ADDR(.ilm) );
	PROVIDE( _eilm = . );
	PROVIDE( eilm = . );

	/* add to avoid orphan sections warnings */
	.rela.sdata : { *(.rela.sdata*) }
	/* add to avoid orphan sections warnings */
	.rela.data : { *(.rela.data*) }

	.data : ALIGN(8)
	{
		KEEP(*(.data.ctest*))
		*(.data .data.*)
		*(.gnu.linkonce.d.*)
		. = ALIGN(8);
		PROVIDE( __gp_rel16$ = . + 0x8000 );
		PROVIDE( __global_pointer$ = . + 0x800 );
		*(.sdata .sdata.* .sdata*)
		*(.gnu.linkonce.s.*)
		/* readonly data placed in RAM for access speed */
		. = ALIGN(8);
		*(.srodata.cst16)
		*(.srodata.cst8)
		*(.srodata.cst4)
		*(.srodata.cst2)
		*(.srodata .srodata.*)
		*(.rdata)
		*(.rodata .rodata.*)
		*(.boot_data_section)
		*(.gnu.linkonce.r.*)
		/* below sections are used for rt-thread */
		. = ALIGN(4);
		__rt_init_start = .;
		KEEP(*(SORT(.rti_fn*)))
		__rt_init_end = .;
		. = ALIGN(4);
		__fsymtab_start = .;
		KEEP(*(FSymTab))
		__fsymtab_end = .;
		. = ALIGN(4);
		__vsymtab_start = .;
		KEEP(*(VSymTab))
		__vsymtab_end = .;
		. = ALIGN(16);
	} >DLM AT>SYSMEM

	.tdata : ALIGN(8)
	{
		PROVIDE( __tls_base = . );
		*(.tdata .tdata.* .gnu.linkonce.td.*)
		. = ALIGN(16);
	} >DLM AT>SYSMEM

	PROVIDE( _data_lma = LOADADDR(.data) );
	PROVIDE( _data = ADDR(.data) );
	PROVIDE( _edata = . );
	PROVIDE( edata = . );

	PROVIDE( _fbss = . );
	PROVIDE( __bss_start = . );

	.tbss (NOLOAD) : ALIGN(8)
	{
		*(.tbss .tbss.* .gnu.linkonce.tb.*)
		*(.tcommon)
		PROVIDE( __tls_end = . );
	} >DLM AT>DLM

	.tbss_space (NOLOAD) : ALIGN(8)
	{
		. = . + SIZEOF(.tbss);
	} >DLM AT>DLM

	.bss (NOLOAD) : ALIGN(8)
	{
		__bss_start__ = .;
		*(.sbss*)
		*(.gnu.linkonce.sb.*)
		*(.bss .bss.*)
		*(.gnu.linkonce.b.*)
		*(COMMON)
		*(.boot_bss_section)
		. = ALIGN(4);
		__bss_end__ = .;
	} >DLM AT>DLM

	PROVIDE( _end = . );
	PROVIDE( end = . );

	/* Nuclei C Runtime Library requirements:
	 * 1. heap need to be align at 16 bytes
	 * 2. __heap_start and __heap_end symbol need to be defined
	 * 3. reserved at least __HEAP_SIZE space for heap
	 */
	.heap (NOLOAD)   : ALIGN(16)
	{
		. = ALIGN(16);
		PROVIDE( __heap_start = . );
		. += __HEAP_SIZE;
		. = ALIGN(16);
		PROVIDE( __heap_limit = . );
	} >DLM AT>DLM

	.stack ORIGIN(STACK) (NOLOAD) :
	{
		. = ALIGN(16);
		PROVIDE( _heap_end = . );
		PROVIDE( __heap_end = . );
		PROVIDE( __StackLimit = . );
		PROVIDE( __StackBottom = . );
		. += __TOT_STACK_SIZE;
		. = ALIGN(16);
		PROVIDE( __StackTop = . );
		PROVIDE( _sp = . );
	} >STACK AT>STACK

	.comment 0 : { *(.comment) }
	.debug_abbrev 0 : { *(.debug_abbrev) }
	.debug_aranges 0 : { *(.debug_aranges) }  
	.debug_frame 0 : { *(.debug_frame) }
	.debug_info 0 : { *(.debug_info) }
	.debug_line 0 : { *(.debug_line) }
	.debug_line_str 0 : { *(.debug_line_str) }
	.debug_loc 0 : { *(.debug_loc) }
	.debug_loclists 0 : { *(.debug_loclists) }
	.debug_macro 0 : { *(.debug_macro) }
	.debug_pubnames 0 : { *(.debug_pubnames) }
	.debug_pubtypes 0 : { *(.debug_pubtypes) }
	.debug_ranges 0 : { *(.debug_ranges) }
	.debug_rnglists 0 : { *(.debug_rnglists) }
	.debug_str 0 : { *(.debug_str) }
	.debug_str_offsets 0 : { *(.debug_str_offsets) }
	.riscv.attributes 0 : { *(.riscv.attributes) }
}
