/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if __riscv

#include "riscv_encoding.h"

/**
 * @brief  Global interrupt disabled
 * @details
 * This function disable global interrupt.
 * @remarks
 * - All the interrupt requests will be ignored by CPU.
 */
.macro BOOT_DISABLE_MIE
	csrc CSR_MSTATUS, MSTATUS_MIE
.endm

/**
 * @brief  Macro for context save
 * @details
 * This macro save ABI defined caller saved registers in the stack.
 * @remarks
 * - This Macro could use to save context when you enter to interrupt
 * or exception
*/
/* Save caller registers */
.macro BOOT_SAVE_CONTEXT
	/* Allocate stack space for context saving */
#ifndef __riscv_32e
	addi sp, sp, -20*REGBYTES
#else
	addi sp, sp, -14*REGBYTES
#endif /* __riscv_32e */

	STORE x1, 0*REGBYTES(sp)
	STORE x4, 1*REGBYTES(sp)
	STORE x5, 2*REGBYTES(sp)
	STORE x6, 3*REGBYTES(sp)
	STORE x7, 4*REGBYTES(sp)
	STORE x10, 5*REGBYTES(sp)
	STORE x11, 6*REGBYTES(sp)
	STORE x12, 7*REGBYTES(sp)
	STORE x13, 8*REGBYTES(sp)
	STORE x14, 9*REGBYTES(sp)
	STORE x15, 10*REGBYTES(sp)
#ifndef __riscv_32e
	STORE x16, 14*REGBYTES(sp)
	STORE x17, 15*REGBYTES(sp)
	STORE x28, 16*REGBYTES(sp)
	STORE x29, 17*REGBYTES(sp)
	STORE x30, 18*REGBYTES(sp)
	STORE x31, 19*REGBYTES(sp)
#endif /* __riscv_32e */
.endm

/**
 * @brief  Macro for restore caller registers
 * @details
 * This macro restore ABI defined caller saved registers from stack.
 * @remarks
 * - You could use this macro to restore context before you want return
 * from interrupt or exeception
 */
/* Restore caller registers */
.macro BOOT_RESTORE_CONTEXT
	LOAD x1, 0*REGBYTES(sp)
	LOAD x4, 1*REGBYTES(sp)
	LOAD x5, 2*REGBYTES(sp)
	LOAD x6, 3*REGBYTES(sp)
	LOAD x7, 4*REGBYTES(sp)
	LOAD x10, 5*REGBYTES(sp)
	LOAD x11, 6*REGBYTES(sp)
	LOAD x12, 7*REGBYTES(sp)
	LOAD x13, 8*REGBYTES(sp)
	LOAD x14, 9*REGBYTES(sp)
	LOAD x15, 10*REGBYTES(sp)
#ifndef __riscv_32e
	LOAD x16, 14*REGBYTES(sp)
	LOAD x17, 15*REGBYTES(sp)
	LOAD x28, 16*REGBYTES(sp)
	LOAD x29, 17*REGBYTES(sp)
	LOAD x30, 18*REGBYTES(sp)
	LOAD x31, 19*REGBYTES(sp)

	/* De-allocate the stack space */
	addi sp, sp, 20*REGBYTES
#else
	/* De-allocate the stack space */
	addi sp, sp, 14*REGBYTES
#endif /* __riscv_32e */

.endm

/**
 * @brief  Macro for save necessary CSRs to stack
 * @details
 * This macro store MCAUSE, MEPC, MSUBM to stack.
 */
.macro BOOT_SAVE_CSR_CONTEXT
	/* Store CSR mcause to stack using pushmcause */
	csrrwi  x0, CSR_PUSHMCAUSE, 11
	/* Store CSR mepc to stack using pushmepc */
	csrrwi  x0, CSR_PUSHMEPC, 12
	/* Store CSR msub to stack using pushmsub */
	csrrwi  x0, CSR_PUSHMSUBM, 13
.endm

/**
 * @brief  Macro for restore necessary CSRs from stack
 * @details
 * This macro restore MSUBM, MEPC, MCAUSE from stack.
 */
.macro BOOT_RESTORE_CSR_CONTEXT
	LOAD x5,  13*REGBYTES(sp)
	csrw CSR_MSUBM, x5
	LOAD x5,  12*REGBYTES(sp)
	csrw CSR_MEPC, x5
	LOAD x5,  11*REGBYTES(sp)
	csrw CSR_MCAUSE, x5
.endm

/**
 * @brief  Exception/NMI Entry
 * @details
 * This function provide common entry functions for exception/nmi.
 * @remarks
 * This function provide a default exception/nmi entry.
 * ABI defined caller save register and some CSR registers
 * to be saved before enter interrupt handler and be restored before return.
 */
	.section .boot_text_section
	/* In CLIC mode, the exeception entry must be 64bytes aligned */
	.align 6
	.global boot_exc_entry
	.weak boot_exc_entry
boot_exc_entry:
	/* Save the caller saving registers (context) */
	BOOT_SAVE_CONTEXT
	/* Save the necessary CSR registers */
	BOOT_SAVE_CSR_CONTEXT

	/*
	 * Set the exception handler function arguments
	 * argument 1: mcause value
	 * argument 2: current stack point(SP) value
	 */
	csrr a0, mcause
	mv a1, sp
	/*
	 * TODO: Call the exception handler function
	 * By default, the function template is provided in
	 * system_Device.c, you can adjust it as you want
	 */
	call boot_default_intexc_handler

	/* Restore the necessary CSR registers */
	BOOT_RESTORE_CSR_CONTEXT
	/* Restore the caller saving registers (context) */
	BOOT_RESTORE_CONTEXT

	/* Return to regular code */
	mret

/**
 * @brief  Non-Vector Interrupt Entry
 * @details
 * This function provide common entry functions for handling
 * non-vector interrupts
 * @remarks
 * This function provide a default non-vector interrupt entry.
 * ABI defined caller save register and some CSR registers need
 * to be saved before enter interrupt handler and be restored before return.
 */
	/* .section .text.irq */
	.section .boot_rom_to_ram_section, "ax"		/* Place irq_entry into ILM to reduce response time */
	/* In CLIC mode, the interrupt entry must be 4bytes aligned */
	.align 2
	.global boot_irq_entry
	.weak boot_irq_entry
/* This label will be set to MTVT2 register */
boot_irq_entry:
	/* Save the caller saving registers (context) */
	BOOT_SAVE_CONTEXT
	/* Save the necessary CSR registers */
	BOOT_SAVE_CSR_CONTEXT

	/* This special CSR read/write operation, which is actually
	 * claim the CLIC to find its pending highest ID, if the ID
	 * is not 0, then automatically enable the mstatus.MIE, and
	 * jump to its vector-entry-label, and update the link register
	 */
	csrrw ra, CSR_JALMNXTI, ra

	/* Critical section with interrupts disabled */
	BOOT_DISABLE_MIE

	/* Restore the necessary CSR registers */
	BOOT_RESTORE_CSR_CONTEXT
	/* Restore the caller saving registers (context) */
	BOOT_RESTORE_CONTEXT

	/* Return to regular code */
	mret

/* Default Handler for Exceptions / Interrupts */
	.section .boot_text_section
	.global boot_default_intexc_handler
	.weak boot_default_intexc_handler
boot_default_intexc_handler:
1:
	j 1b

#endif	/* __riscv */
