/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __BOOT_FLASH_V1_2_H__
#define __BOOT_FLASH_V1_2_H__

#ifdef __cplusplus
extern "C"{
#endif

 #if (GS32F00xx == 0x1200)

#include "boot_hw_type.h"
#include "inc/hw_flash.h"

#ifndef BOOT_MIN
#define BOOT_MIN(a, b) (((a) < (b)) ? (a) : (b))
#endif

#ifndef BOOT_MAX
#define BOOT_MAX(a, b) (((a) > (b)) ? (a) : (b))
#endif

#define BOOT_FLASH_UNLOCK_CODE1							0x7123
#define BOOT_FLASH_UNLOCK_CODE2							0x812a
#define BOOT_FLASH_UNLOCK_CODE3							0xbee1

#define BOOT_FLASH_BASE									(0x08000000U)		/* Flash */
#define BOOT_FLASH_INTERFACE_EFC_BASE					(0x50002000U)
#define BOOT_FLASH_INTERFACE_EFC2_BASE					(0x50003000U)

#define BOOT_FLASH_FUNC_SECTION							BOOT_RAM

/**
 * @brief The number of flash bank.
 *
 */
#define BOOT_FlashBankNumber							2U

/**
 * @brief FLASH register offset.
 *
 */
#define BOOT_FLASH_O_CONTROL							0x0
#define BOOT_FLASH_O_TIMING0							0x4
#define BOOT_FLASH_O_TIMING1							0x8
#define BOOT_FLASH_O_TIMING2							0xC
#define BOOT_FLASH_O_UNLOCK1							0x18
#define BOOT_FLASH_O_UNLOCK2							0x1C
#define BOOT_FLASH_O_ADDRESS							0x20
#define BOOT_FLASH_O_DATA0								0x24
#define BOOT_FLASH_O_DATA1								0x28
#define BOOT_FLASH_O_DATA2								0x2C
#define BOOT_FLASH_O_DATA3								0x30
#define BOOT_FLASH_O_DATA4								0x34
#define BOOT_FLASH_O_STATUS								0x44
#define BOOT_FLASH_O_REDUN0								0x48
#define BOOT_FLASH_O_REDUN1								0x4C
#define BOOT_FLASH_O_SEC_COUNT							0x50
#define BOOT_FLASH_O_DED_COUNT							0x54
#define BOOT_FLASH_O_SMW_SET0							0x5C
#define BOOT_FLASH_O_SMW_SET10							0x60
#define BOOT_FLASH_O_SMW_SET11							0x64
#define BOOT_FLASH_O_SMW_SET2							0x68
#define BOOT_FLASH_O_SMW_SET3							0x6C
#define BOOT_FLASH_O_SMW_SMP0							0x70
#define BOOT_FLASH_O_SMW_SMP1							0x74
#define BOOT_FLASH_O_SMW_SME0							0x78
#define BOOT_FLASH_O_SMW_SME1							0x7C
#define BOOT_FLASH_O_SMW_CFG							0x80
#define BOOT_FLASH_O_WP0								0x84
#define BOOT_FLASH_O_WP1								0x88
#define BOOT_FLASH_O_WP2								0x8C

#define BOOT_FLASH_CACHE_O_CCR							0x00
#define BOOT_FLASH_CACHE_O_SR							0x04
#define BOOT_FLASH_CACHE_O_IRQMASK						0x08
#define BOOT_FLASH_CACHE_O_IRQSTAT						0x0C
#define BOOT_FLASH_CACHE_O_HWPARAMS						0x10
#define BOOT_FLASH_CACHE_O_CSHR							0x14
#define BOOT_FLASH_CACHE_O_CSMR							0x18

#define BOOT_FLASH_CONTROL_ECC_EN						0x02U
#define BOOT_FLASH_CONTROL_FAST_PRG_MODE				0x04U
#define BOOT_FLASH_CONTROL_WRITE_PROTECT				0x01U

#define BOOT_FLASH_CONTROL_SLP_MODE						0x01U
#define BOOT_FLASH_CONTROL_ERASE						0x08U
#define BOOT_FLASH_CONTROL_ERASE_PAGE					0x10U
#define BOOT_FLASH_CONTROL_PROGRAM_START				0x20U
#define BOOT_FLASH_CONTROL_ERASE_REFERENCE_CELL			0x40U
#define BOOT_FLASH_CONTROL_RECALL_TRIM_CODE				0x80U
#define BOOT_FLASH_CONTROL_PROGRAM_END					0x100U
#define BOOT_FLASH_CONTROL_ERROR_INDICATE				0x200U
#define BOOT_FLASH_CONTROL_WRITE_COMP_INTR_ENABLE		0x400U
#define BOOT_FLASH_CONTROL_ECC_INTR_ENABLE				0x800U
#define BOOT_FLASH_CONTROL_DED_INTR_ENABLE				0x1000U

#define BOOT_FLASH_TIMING0_TACC_CYCLES					0x3FU
#define BOOT_FLASH_TIMING0_PROG_CYCLES					0x3FFC0U
#define BOOT_FLASH_TIMING0_EIGHT_US_CYCLES				0xFFFC0000U

#define BOOT_FLASH_TIMING1_ERC_ERASE_CYCLES				0x3FFFFFU
#define BOOT_FLASH_TIMING1_TWO_US_CYCLES				0xFFC00000U

#define BOOT_FLASH_TIMING2_PROG_ERC_SHARED_CYCLES		0xFFFFU
#define BOOT_FLASH_TIMING2_HUNDRED_NS_CYCLES			0xFF0000U
#define BOOT_FLASH_TIMING2_TEN_NS_CYCLES				0xF000000U

#define BOOT_FLASH_ADDRESS_GROUP_ADDR					0xFU
#define BOOT_FLASH_ADDRESS_YADDR						0x1F0U
#define BOOT_FLASH_ADDRESS_XADDR						0x7FE00U
#define BOOT_FLASH_ADDRESS_IFREN_ADDR					0x80000U

#define BOOT_FLASH_STATUS_EFC_BUSY						0x01U
#define BOOT_FLASH_STATUS_WRITE_ERROR					0x02U
#define BOOT_FLASH_STATUS_DISABLE_REDUND				0x04U
#define BOOT_FLASH_STATUS_DISCHARGED_STATUS				0x08U
#define BOOT_FLASH_STATUS_BROWNOUT_STATUS				0x10U
#define BOOT_FLASH_STATUS_WRITE_COMP_STATUS				0x20U
#define BOOT_FLASH_STATUS_SEC_DETECTED					0x40U
#define BOOT_FLASH_STATUS_DED_DETECTED					0x80U
#define BOOT_FLASH_STATUS_SMW_ERR_STATUS				0x100U
#define BOOT_FLASH_STATUS_SMW_LATEST_STATUS				0x3FE00U
#define BOOT_FLASH_STATUS_SMW_LOOP_STATUS				0xFFC0000U
#define BOOT_FLASH_STATUS_ALL							0xFFFFFFFFU

#define BOOT_FLASH_REDUN0_REDUN_INDICATE				0x01U
#define BOOT_FLASH_REDUN0_REDUN_PAGE_ADDR				0x7F000U

#define BOOT_FLASH_REDUN1_REDUN_INDICATE				0x01U
#define BOOT_FLASH_REDUN1_REDUN_PAGE_ADDR				0x7F000U

#define BOOT_FLASH_WP0_INFO_WRITE_PROTECT				0x01U
#define BOOT_FLASH_WP0_WRITE_PROTECT_ALL				0x10000U

#define BOOT_FLASH_CACHE_CCR_EN							0x01U
#define BOOT_FLASH_CACHE_CCR_INV_REQ					0x02U
#define BOOT_FLASH_CACHE_CCR_POW_REQ					0x04U
#define BOOT_FLASH_CACHE_CCR_SET_MAN_POW				0x08U
#define BOOT_FLASH_CACHE_CCR_SET_MAN_INV				0x10U
#define BOOT_FLASH_CACHE_CCR_SET_PREFETCH				0x20U
#define BOOT_FLASH_CACHE_CCR_STATISTIC_EN				0x40U

#define BOOT_FLASH_CACHE_SR_CS							0x03U
#define BOOT_FLASH_CACHE_SR_INV_STAT					0x04U
#define BOOT_FLASH_CACHE_SR_POW_STAT					0x10U

#define BOOT_FLASH_CACHE_SR_DISABLED					0
#define BOOT_FLASH_CACHE_SR_ENABLING					1
#define BOOT_FLASH_CACHE_SR_ENABLED						2
#define BOOT_FLASH_CACHE_SR_DISABLING					3

#define BOOT_FLASH_CACHE_IRQMASK_POW_ERR				0x01U
#define BOOT_FLASH_CACHE_IRQMASK_MAN_INV_ERR			0x02U

#define BOOT_FLASH_CACHE_IRQSTAT_POW_ERR				0x01U
#define BOOT_FLASH_CACHE_IRQSTAT_MAN_INV_ERR			0x02U

#define BOOT_FLASH_O_STATIC_CFG							0x10
#define BOOT_FLASH_O_WRITE_PROTECT						0x14
#define BOOT_OPT_UNLOCK									0x100
#define BOOT_OTP_LOCK									0x104
#define BOOT_OTP_MOD_CTRL								0x108
#define BOOT_OTP_MOD_STATUS								0x10c
#define BOOT_SET_USER_OPT0								0x200
#define BOOT_SET_USER_OPT1								0x204
#define BOOT_SET_RDP									0x208
#define BOOT_SET_PCROP_L								0x20C
#define BOOT_SET_PCROP_H								0x210
#define BOOT_SET_WRP_L									0x214
#define BOOT_SET_WRP_H									0x218
#define BOOT_SET_SEC_SIZE								0x21C
#define BOOT_SET_BOOT_LOCK								0x220
#define BOOT_SET_PCROP_RDP								0x224
#define BOOT_SET_SEC_DEBUG_EN							0x228

#define BOOT_OPT_UNLOCK_KEY1_VALUE						0x08192A3B
#define BOOT_OPT_UNLOCK_KEY2_VALUE						0x4C5D6E7F

/**
 * @brief flash page size 8KB
 *
 */
#define BOOT_FlashMainPageSize							0x2000U
#define BOOT_FlashInfoPageSize							0x2000U

/**
 * @brief Get the flash lock status.
 *
 * @param flashCtlBase Flash control base address.
 * @return uint32_t 0 is the flash unlocked, else locked.
 */
BOOT_FLASH_FUNC_SECTION uint32_t boot_flash_lock_status(uint32_t flashCtlBase);

/**
 * @brief Lock the flash control register.
 *
 * @param flashCtlBase Flash control base address.
 */
BOOT_FLASH_FUNC_SECTION void boot_flash_lock_register(uint32_t flashCtlBase);

BOOT_FLASH_FUNC_SECTION void boot_flash_unlock_register(uint32_t flashCtlBase);

/**
 * @brief wake up from sleep mode
 *
 * @param base is the base address of the flash registers.
 */
BOOT_FLASH_FUNC_SECTION void boot_flash_wakeup_from_lpm(uint32_t ctrlBase);

BOOT_FLASH_FUNC_SECTION void boot_flash_set_read_wait_cycle(uint32_t flashCtlBase,
															uint32_t cycle_num);

#endif	/* (GS32F00xx == 0x3000) */

#ifdef __cplusplus
}
#endif

#endif
