/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __BOOT_RISCV_CACHE_H__
#define __BOOT_RISCV_CACHE_H__

#ifdef __cplusplus
extern "C"{
#endif

#include "boot_hw_type.h"
#include "riscv_encoding.h"
#include "core_feature_base.h"

/**
 * @brief Cache CCM Command Types
 *
 */
typedef enum CCM_CMD_TYPE {
	BOOT_CCM_DC_INVAL		= 0x0,				/* Unlock and invalidate D-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_DC_WB			= 0x1,				/* Flush the specific D-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_DC_WBINVAL		= 0x2,				/* Unlock, flush and invalidate the specific D-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_DC_LOCK		= 0x3,				/* Lock the specific D-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_DC_UNLOCK		= 0x4,				/* Unlock the specific D-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_DC_WBINVAL_ALL	= 0x6,				/* Unlock and flush and invalidate all the valid and dirty D-Cache lines */
	BOOT_CCM_DC_WB_ALL		= 0x7,				/* Flush all the valid and dirty D-Cache lines */
	BOOT_CCM_DC_INVAL_ALL	= 0x17,				/* Unlock and invalidate all the D-Cache lines */
	BOOT_CCM_IC_INVAL		= 0x8,				/* Unlock and invalidate I-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_IC_LOCK		= 0xb,				/* Lock the specific I-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_IC_UNLOCK		= 0xc,				/* Unlock the specific I-Cache line specified by CSR CCM_XBEGINADDR */
	BOOT_CCM_IC_INVAL_ALL	= 0xd				/* Unlock and invalidate all the I-Cache lines */
} CCM_CMD_t;

/**
 * @brief Check ICache Unit Present or Not
 *
 * This function check icache unit present or not via mcfg_info csr
 *
 * - This function might not work for some old nuclei processors
 * - Please make sure the version of your nuclei processor contain ICACHE bit in mcfg_info
 * @return 1 if present otherwise 0
*/
BOOT_TEXT int32_t boot_ICache_present(void);

/**
 * @brief Check DCache Unit Present or Not
 *
 * This function check dcache unit present or not via mcfg_info csr
 *
 * - This function might not work for some old nuclei processors
 * - Please make sure the version of your nuclei processor contain DCACHE bit in mcfg_info
 * @return 1 if present otherwise 0
 */
BOOT_TEXT int32_t boot_DCache_present(void);

/**
 * @brief Enable ICache
 *
 * This function enable I-Cache
 *
 * - This function can be called in M-Mode only.
 * - This @ref CSR_MCACHE_CTL register control I Cache enable.
 * - @ref DisableICache
*/
BOOT_TEXT void boot_enable_ICache(void);

/**
 * @brief Disable ICache
 *
 * This function Disable I-Cache
 *
 * - This function can be called in M-Mode only.
 * - This @ref CSR_MCACHE_CTL register control I Cache enable.
 */
BOOT_TEXT void boot_disable_ICache(void);

/**
 * @brief Enable DCache
 *
 * This function enable D-Cache
 *
 * - This function can be called in M-Mode only.
 * - This @ref CSR_MCACHE_CTL register control D Cache enable.
 *
 * - @ref DisableDCache
*/
BOOT_TEXT void boot_enable_DCache(void);

/**
 * @brief Disable DCache
 *
 * This function Disable D-Cache
 *
 * - This function can be called in M-Mode only.
 * - This @ref CSR_MCACHE_CTL register control D Cache enable.
 *
 * - @ref EnableDCache
 */
BOOT_TEXT void boot_disable_DCache(void);

/**
 * @brief  Flush one D-Cache line specified by address in M-Mode
 *
 * This function flush one D-Cache line specified by the address.
 * Command @ref CCM_DC_WB is written to CSR @ref CSR_CCM_MCOMMAND.
 *
 * This function must be executed in M-Mode only.
 * @param [in] addr start address to be flushed
 */
BOOT_TEXT void boot_machine_mode_flush_DCache_line(unsigned long addr);

/**
 * @brief Flush several D-Cache lines specified by address in M-Mode
 *
 * This function flush several D-Cache lines specified
 * by the address and line count.
 * Command @ref CCM_DC_WB is written to CSR @ref CSR_CCM_MCOMMAND.
 *
 * This function must be executed in M-Mode only.
 * @param [in] addr start address to be flushed
 * @param [in] cnt count of cache lines to be flushed
 */
BOOT_TEXT void boot_machine_mode_flush_DCache_lines(unsigned long addr, unsigned long cnt);

/**
 * @brief  Invalidate one D-Cache line specified by address in M-Mode
 *
 * This function unlock and invalidate one D-Cache line specified
 * by the address.
 *
 * Command @ref CCM_DC_INVAL is written to CSR @ref CSR_CCM_MCOMMAND.
 *
 * This function must be executed in M-Mode only.
 * @param [in] addr start address to be invalidated
 */
BOOT_TEXT void boot_machine_mode_invalid_DCache_line(unsigned long addr);

/**
 * @brief Invalidate several D-Cache lines specified by address in M-Mode
 *
 * This function unlock and invalidate several D-Cache lines specified
 * by the address and line count.
 * Command @ref CCM_DC_INVAL is written to CSR @ref CSR_CCM_MCOMMAND.
 *
 * This function must be executed in M-Mode only.
 * @param [in] addr start address to be invalidated
 * @param [in] cnt count of cache lines to be invalidated
 */
BOOT_TEXT void boot_machine_mode_invalid_DCache_lines(unsigned long addr, unsigned long cnt);

/**
 * @brief Invalidate one I-Cache line specified by address in M-Mode
 *
 * This function unlock and invalidate one I-Cache line specified
 * by the address.
 * Command @ref CCM_IC_INVAL is written to CSR @ref CSR_CCM_MCOMMAND.
 *
 * This function must be executed in M-Mode only.
 * @param [in] addr start address to be invalidated
 */
BOOT_TEXT void boot_machine_mode_invalid_ICache_line(unsigned long addr);

/**
 * @brief Invalidate several I-Cache lines specified by address in M-Mode
 *
 * This function unlock and invalidate several I-Cache lines specified
 * by the address and line count.
 * Command @ref CCM_IC_INVAL is written to CSR @ref CSR_CCM_MCOMMAND.
 *
 * This function must be executed in M-Mode only.
 * @param [in] addr start address to be invalidated
 * @param [in] cnt count of cache lines to be invalidated
 */
BOOT_TEXT void boot_machine_mode_invalid_ICache_lines(unsigned long addr, unsigned long cnt);

#ifdef __cplusplus
}
#endif

#endif
