/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __BOOT_RISCV_ECLIC_H__
#define __BOOT_RISCV_ECLIC_H__

#include "boot_hw_type.h"
#include "boot_irq_number.h"

#ifdef __cplusplus
extern "C"{
#endif

/**
 * @brief End of Doxygen Group NMSIS_Core_ExceptionAndNMI
 *
 */
#define BOOT_FALLBACK_DEFAULT_ECLIC_BASE				0x0C000000UL
#define BOOT_FALLBACK_DEFAULT_SYSTIMER_BASE				0x02000000UL

#define BOOT_SOC_EXTERNAL_MAP_TO_ECLIC_IRQn_OFFSET		19

#define BOOT_CLIC_CLICCFG_NLBIT_Pos				1U											/* CLIC CLICCFG: NLBIT Position */
#define BOOT_CLIC_CLICCFG_NLBIT_Msk				(0xFUL << BOOT_CLIC_CLICCFG_NLBIT_Pos)		/* CLIC CLICCFG: NLBIT Mask */

#define BOOT_CLIC_CLICINFO_CTLBIT_Pos			21U											/* CLIC INTINFO: __ECLIC_GetInfoCtlbits() Position */
#define BOOT_CLIC_CLICINFO_CTLBIT_Msk			(0xFUL << BOOT_CLIC_CLICINFO_CTLBIT_Pos)	/* CLIC INTINFO: __ECLIC_GetInfoCtlbits() Mask */

#define BOOT_CLIC_CLICINFO_VER_Pos				13U											/* CLIC CLICINFO: VERSION Position */
#define BOOT_CLIC_CLICINFO_VER_Msk				(0xFFUL << BOOT_CLIC_CLICCFG_NLBIT_Pos)		/* CLIC CLICINFO: VERSION Mask */

#define BOOT_CLIC_CLICINFO_NUM_Pos				0U											/* CLIC CLICINFO: NUM Position */
#define BOOT_CLIC_CLICINFO_NUM_Msk				(0xFFFUL << BOOT_CLIC_CLICINFO_NUM_Pos)		/* CLIC CLICINFO: NUM Mask */

#define BOOT_CLIC_INTIP_IP_Pos					0U											/* CLIC INTIP: IP Position */
#define BOOT_CLIC_INTIP_IP_Msk					(0x1UL << BOOT_CLIC_INTIP_IP_Pos)			/* CLIC INTIP: IP Mask */

#define BOOT_CLIC_INTIE_IE_Pos					0U											/* CLIC INTIE: IE Position */
#define BOOT_CLIC_INTIE_IE_Msk					(0x1UL << BOOT_CLIC_INTIE_IE_Pos)			/* CLIC INTIE: IE Mask */

#if defined(__TEE_PRESENT) && (__TEE_PRESENT == 1)
#define BOOT_CLIC_INTATTR_MODE_Pos				6U											/* CLIC INTATTA: Mode Position */
#define BOOT_CLIC_INTATTR_MODE_Msk				(0x3U << BOOT_CLIC_INTATTR_MODE_Pos)		/* CLIC INTATTA: Mode Mask */
#endif

#define BOOT_CLIC_INTATTR_TRIG_Pos				1U											/* CLIC INTATTR: TRIG Position */
#define BOOT_CLIC_INTATTR_TRIG_Msk				(0x3UL << BOOT_CLIC_INTATTR_TRIG_Pos)		/* CLIC INTATTR: TRIG Mask */

#define BOOT_CLIC_INTATTR_SHV_Pos				0U											/* CLIC INTATTR: SHV Position */
#define BOOT_CLIC_INTATTR_SHV_Msk				(0x1UL << BOOT_CLIC_INTATTR_SHV_Pos)		/* CLIC INTATTR: SHV Mask */

#define BOOT_ECLIC_MAX_NLBITS					8U											/* Max nlbit of the CLICINTCTLBITS */
#define BOOT_ECLIC_MODE_MTVEC_Msk				3U											/* ECLIC Mode mask for MTVT CSR Register */

#define BOOT_ECLIC_NON_VECTOR_INTERRUPT			0x0											/* Non-Vector Interrupt Mode of ECLIC */
#define BOOT_ECLIC_VECTOR_INTERRUPT				0x1

/**
 * @brief ECLIC Trigger Enum for different Trigger Type
 *
 */
typedef enum ECLIC_TRIGGER {
	BOOT_ECLIC_LEVEL_TRIGGER			= 0x0,			/* Level Triggerred, trig[0] = 0 */
	BOOT_ECLIC_POSTIVE_EDGE_TRIGGER		= 0x1,			/* Postive/Rising Edge Triggered, trig[0] = 1, trig[1] = 0 */
	BOOT_ECLIC_NEGTIVE_EDGE_TRIGGER		= 0x3,			/* Negtive/Falling Edge Triggered, trig[0] = 1, trig[1] = 1 */
	BOOT_ECLIC_MAX_TRIGGER				= 0x3			/* MAX Supported Trigger Mode */
} BOOT_ECLIC_TRIGGER_t;

/**
 * @brief Access to the machine mode register structure of INTIP, INTIE, INTATTR, INTCTL.
 *
 */
typedef struct {
	uint8_t INTIP;						/* Offset: 0x000 (R/W)  Interrupt set pending register */
	uint8_t INTIE;						/* Offset: 0x001 (R/W)  Interrupt set enable register */
	uint8_t INTATTR;					/* Offset: 0x002 (R/W)  Interrupt set attributes register */
	uint8_t INTCTRL;					/* Offset: 0x003 (R/W)  Interrupt configure register */
} BOOT_CLIC_CTRL_REG_t;

/**
 * @brief Access to the structure of ECLIC Memory Map, which is compatible with TEE.
 *
 */
typedef struct {
	uint8_t CFG;						/* Offset: 0x000 (R/W)  CLIC configuration register */
	uint8_t RESERVED0[3];
	uint32_t INFO;						/* Offset: 0x004 (R/ )  CLIC information register */
	uint8_t RESERVED1;
#if defined(__TEE_PRESENT) && (__TEE_PRESENT == 1)
	uint8_t STH;						/* Offset: 0x009 (R/W )  CLIC supervisor mode interrupt-level threshold */
#else
	uint8_t RESERVED2;
#endif
	uint8_t RESERVED3;
	uint8_t MTH;						/* Offset: 0x00B(R/W)  CLIC machine mode interrupt-level threshold */
	uint32_t RESERVED4[1021];
#if defined(__TEE_PRESENT) && (__TEE_PRESENT == 1)
	BOOT_CLIC_CTRL_REG_t CTRL[1024];	/* Offset: 0x1000 (R/W) CLIC machine mode register structure for INTIP, INTIE, INTATTR, INTCTL */
	uint32_t RESERVED5[2];
	uint8_t RESERVED6;
	uint8_t SSTH;						/* Offset: 0x2009 (R)  CLIC supervisor mode threshold register, which is a mirror to mintthresh.sth */
	uint8_t RESERVED7;
	uint8_t RESERVED8;
	uint32_t RESERVED9[1021];
	BOOT_CLIC_CTRL_REG_t SCTRL[1024];	/* Offset: 0x3000 (R/W) CLIC supervisor mode register structure for INTIP, INTIE, INTATTR, INTCTL */
#else
	BOOT_CLIC_CTRL_REG_t CTRL[4096];		/* Offset: 0x1000 (R/W) CLIC machine mode register structure for INTIP, INTIE, INTATTR, INTCTL */
#endif
} Eclic_Reg_t;

/**
 * @brief Core Internal Region Information
 *
*/
typedef struct Core_IRegion_Info {
	unsigned long iregion_base;			/* Internal region base address */
	unsigned long eclic_base;			/* eclic base address */
	unsigned long systimer_base;		/* system timer base address */
	unsigned long smp_base;				/* smp base address */
	unsigned long idu_base;				/* idu base address */
} Core_IRegion_Info_t;

/**
 * @brief Get IRegion Info
 *
 * @return The core iregion infomation
 */
BOOT_TEXT Core_IRegion_Info_t boot_get_core_iregion_info(void);

/**
 * @brief Set Machine Mode Interrupt Level Threshold
 *
 * This function sets machine mode interrupt level threshold.
 *
 * @param eclic Pointer to the base address of the eclic controller register
 * @param mth Interrupt Level Threshold.
 */
BOOT_TEXT void boot_eclic_set_machine_mode(Eclic_Reg_t *eclic, uint8_t mth);

/**
 * @brief Get CLICINTCTLBITS
 * This function gets CLICINTCTLBITS from CLICINFO register.
 *
 * @param eclic Pointer to the base address of the eclic controller register
 * @return uint32_t CLICINTCTLBITS from CLICINFO register.
 */
BOOT_TEXT uint32_t boot_eclic_get_info_ctl_bits(Eclic_Reg_t *eclic);

/**
 * @brief  Get nlbits value
 *
 * This function get the nlbits value of CLICCFG register.
 *
 * @param  eclic Pointer to the base address of the eclic controller register
 * @return nlbits value of CLICCFG register
 *
 * - nlbits is used to set the width of level in the CLICINTCTL[i].
 *
 * - @ref ECLIC_SetCfgNlbits
 */
BOOT_TEXT uint32_t boot_eclic_get_cfg_nl_bits(Eclic_Reg_t *eclic);

/**
 * @brief Set nlbits value
 * This function set the nlbits value of CLICCFG register.
 *
 * @param eclic Pointer to the base address of the eclic controller register
 * @param nlbits is used to set the width of level in the CLICINTCTL[i].
 */
BOOT_TEXT void boot_eclic_set_cfg_nl_bits(Eclic_Reg_t *eclic, uint32_t nlbits);

/**
 * @brief Set trigger mode and polarity for a specific interrupt
 *
 * @param eclic Pointer to the base address of the eclic controller register
 * @param IRQn Interrupt number
 * @param trig	- 00  level trigger, @ref ECLIC_LEVEL_TRIGGER
 *				- 01  positive edge trigger, @ref ECLIC_POSTIVE_EDGE_TRIGGER
 *				- 02  level trigger, @ref ECLIC_LEVEL_TRIGGER
 *				- 03  negative edge trigger, @ref ECLIC_NEGTIVE_EDGE_TRIGGER
 */
BOOT_TEXT void boot_eclic_set_trigger_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, BOOT_ECLIC_TRIGGER_t trig);

/**
 * @brief Enable a specific interrupt
 *
 * This function enables the specific interrupt @em IRQn.
 *
 * @param [in]  IRQn  Interrupt number
 *
 * - IRQn must not be negative.
 *
 * - @ref ECLIC_DisableIRQ
 */
BOOT_TEXT void boot_eclic_enable_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn);

/**
 * @brief Disable a specific interrupt
 *
 * This function disables the specific interrupt @em IRQn.
 *
 * @param eclic Pointer to the base address of the eclic controller register
 * @param IRQn Number of the external interrupt to disable
 */
BOOT_TEXT void boot_eclic_disable_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn);

/**
 * @brief Set interrupt working mode for a specific interrupt
 *
 * This function set selective hardware vector or non-vector working mode of the specific interrupt @em IRQn.
 *
 * @param eclicc Pointer to the base address of the eclic controller register
 * @param [in] IRQn Interrupt number
 * @param [in] shv
 * - 0 non-vector mode, @ref ECLIC_NON_VECTOR_INTERRUPT
 * - 1 vector mode, @ref ECLIC_VECTOR_INTERRUPT
 *
 * - IRQn must not be negative.
 *
 * - @ref ECLIC_GetShvIRQ
 */
BOOT_TEXT void boot_eclic_set_ShvIRQ(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint32_t shv);

/**
 * @brief Get ECLIC Interrupt Input Control Register value for a specific interrupt
 *
 * This function modify ECLIC Interrupt Input Control register of the specific interrupt \em IRQn.
 *
 * @param eclicc Pointer to the base address of the eclic controller register
 * @param [in] IRQn  Interrupt number
 * @return value of ECLIC Interrupt Input Control register
 *
 * - IRQn must not be negative.
 *
 * - @ref ECLIC_SetCtrlIRQ
 */
BOOT_TEXT uint8_t boot_eclic_get_ctrl_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn);

/**
 * @brief Modify ECLIC Interrupt Input Control Register for a specific interrupt
 *
 * This function modify ECLIC Interrupt Input Control(CLICINTCTL[i]) register of the specific interrupt @em IRQn.
 *
 * @param eclicc Pointer to the base address of the eclic controller register
 * @param [in] IRQn Interrupt number
 * @param [in] intctrl Set value for CLICINTCTL[i] register
 *
 * - IRQn must not be negative.
 *
 * - @ref ECLIC_GetCtrlIRQ
 */
BOOT_TEXT void boot_eclic_set_ctrl_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint8_t intctrl);

/**
 * @brief Set ECLIC Interrupt level of a specific interrupt
 *
 * This function set interrupt level of the specific interrupt @em IRQn.
 *
 * @param eclicc Pointer to the base address of the eclic controller register
 * @param [in] IRQn Interrupt number
 * @param [in] lvl_abs Interrupt level
 *
 * - IRQn must not be negative.
 * - If lvl_abs to be set is larger than the max level allowed, it will be force to be max level.
 * - When you set level value you need use clciinfo.nlbits to get the width of level.
 * Then we could know the maximum of level. CLICINTCTLBITS is how many total bits are
 * present in the CLICINTCTL register.
 *
 * - @ref ECLIC_GetLevelIRQ
 */
BOOT_TEXT void boot_eclic_set_level_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint8_t lvl_abs);

/**
 * @brief  Get ECLIC Interrupt priority of a specific interrupt
 *
 * This function get interrupt priority of the specific interrupt @em IRQn.
 *
 * @param eclicc Pointer to the base address of the eclic controller register 
 * @param [in] IRQn  Interrupt number
 * @param [in] pri   Interrupt priority
 *
 * - IRQn must not be negative.
 * - If pri to be set is larger than the max priority allowed, it will be force to be max priority.
 * - Priority width is CLICINTCTLBITS minus clciinfo.nlbits if clciinfo.nlbits
 * is less than CLICINTCTLBITS. Otherwise priority width is 0.
 *
 * - @ref ECLIC_GetPriorityIRQ
 */
BOOT_TEXT void boot_eclic_set_priority_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint8_t priority);

#ifdef __cplusplus
}
#endif

#endif
