/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if (GS32F00xx == 0x3000)

#include "boot_flash_v3_0.h"
#include "boot_delay.h"

BOOT_FLASH_SECTION static inline bool boot_flash_is_base_valid(uintptr_t base)
{
#if BOOT_FLASH_CHECK_BASE_SWITCH

	if (base != BOOT_FLASH_EFC0_ADDR && base != BOOT_FLASH_EFC1_ADDR )
		return false;

	return true;

#else

	return true;

#endif
}

int boot_flash_set_read_wait_timing(uintptr_t base, uint32_t read_cycle)
{
	uint32_t reg_val;

	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	reg_val = HWREG(base + BOOT_FLASH_TIMING_0);

	reg_val &= ~BOOT_FLASH_TIMING_0_READ_TCYC_M;

	reg_val |= (read_cycle & BOOT_FLASH_TIMING_0_READ_TCYC_M);

	HWREG(base + BOOT_FLASH_TIMING_0) = reg_val;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_set_prog_cycles(uintptr_t base, uint32_t prog_cycle)
{
	uint32_t reg_val;

	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	reg_val = HWREG(base + BOOT_FLASH_TIMING_0);

	reg_val &= ~BOOT_FLASH_TIMING_0_PROG_CYCLES_M;

	reg_val |= (prog_cycle & ((BOOT_FLASH_TIMING_0_PROG_CYCLES_M >>
								BOOT_FLASH_TIMING_0_PROG_CYCLES_S) <<
								BOOT_FLASH_TIMING_0_PROG_CYCLES_S));

	HWREG(base + BOOT_FLASH_TIMING_0) = reg_val;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_set_eight_us_cycles(uintptr_t base, uint32_t eight_us_cycles)
{
	uint32_t reg_val;

	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	reg_val = HWREG(base + BOOT_FLASH_TIMING_0);

	reg_val &= ~(BOOT_FLASH_TIMING_0_EIGHT_US_CYCLES_M);

	reg_val |= (eight_us_cycles &
				((BOOT_FLASH_TIMING_0_EIGHT_US_CYCLES_M >>
				BOOT_FLASH_TIMING_0_EIGHT_US_CYCLES_S) <<
				BOOT_FLASH_TIMING_0_EIGHT_US_CYCLES_S));

	HWREG(base + BOOT_FLASH_TIMING_0) = reg_val;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_set_erc_erase_cycles(uintptr_t base, uint32_t erase_cycles)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_TIMING_1) |= (erase_cycles & BOOT_FLASH_TIMING_1_ERC_ERASE_CYCLES_M);

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_set_two_us_cycles(uintptr_t base, uint32_t two_us_cycles)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_TIMING_1) |= (two_us_cycles & ((BOOT_FLASH_TIMING_1_TWO_US_CYCLES_M >>
															BOOT_FLASH_TIMING_1_TWO_US_CYCLES_S) <<
															BOOT_FLASH_TIMING_1_TWO_US_CYCLES_S));

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_set_test_hold_cycles(uintptr_t base, uint32_t test_hold_cycles)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_TIMING_2) |= (test_hold_cycles &
										BOOT_FLASH_TIMING_2_TEST_HOLD_CYCLES_M);

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_set_hundred_ns_cycles(uintptr_t base, uint32_t hundred_ns_cycles)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_TIMING_2) |= (hundred_ns_cycles &
										(BOOT_FLASH_TIMING_2_HUNDRED_NS_CYCLES_M >>
										BOOT_FLASH_TIMING_2_HUNDRED_NS_CYCLES_S) <<
										BOOT_FLASH_TIMING_2_HUNDRED_NS_CYCLES_S);

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_set_ten_ns_cycles(uintptr_t base, uint32_t ten_ns_cycles)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_TIMING_2) |= (ten_ns_cycles &
										(BOOT_FLASH_TIMING_2_TEN_NS_CYCLES_M >>
										BOOT_FLASH_TIMING_2_TEN_NS_CYCLES_S) <<
										BOOT_FLASH_TIMING_2_TEN_NS_CYCLES_S);

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_enable_slp_mode(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) |= BOOT_FLASH_STATIC_CFG_SLP_MODE_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_disable_slp_mode(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) &= ~BOOT_FLASH_STATIC_CFG_SLP_MODE_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_enable_ecc(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) |= BOOT_FLASH_STATIC_CFG_ECC_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;

}

int boot_flash_disable_ecc(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) &= ~BOOT_FLASH_STATIC_CFG_ECC_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_enable_fast_program_mode(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) |= BOOT_FLASH_STATIC_CFG_FAST_PRG_MODE_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_disable_fast_program_mode(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) &= ~BOOT_FLASH_STATIC_CFG_FAST_PRG_MODE_M;

	return BOOT_FLASH_STAT_ERR;
}

int boot_flash_enable_err_indicate(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) |= BOOT_FLASH_STATIC_CFG_ERR_INDICATE_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_disable_err_indicate(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) &= ~BOOT_FLASH_STATIC_CFG_ERR_INDICATE_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_enable_program_erase_complete_interrupt(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) |= BOOT_FLASH_STATIC_CFG_WRITE_CMP_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_disable_program_erase_complete_interrupt(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) &= ~BOOT_FLASH_STATIC_CFG_WRITE_CMP_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_enable_ecc_correction_interrupt(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) |= BOOT_FLASH_STATIC_CFG_ECC_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_disable_ecc_correction_interrupt(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) &= ~BOOT_FLASH_STATIC_CFG_ECC_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_enable_ecc_detected_interrupt(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) |= BOOT_FLASH_STATIC_CFG_DED_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_disable_ecc_detected_interrupt(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATIC_CFG) &= ~BOOT_FLASH_STATIC_CFG_DED_INTR_EN_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_enable_register_write_protect(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_UNLOCK_1) = BOOT_FLASH_UNLOCK_1_CODE0;
	HWREG(base + BOOT_FLASH_UNLOCK_2) = BOOT_FLASH_UNLOCK_2_CODE1;
	HWREG(base + BOOT_FLASH_UNLOCK_1) = BOOT_FLASH_UNLOCK_1_CODE2;

	HWREG(base + BOOT_FLASH_WRITE_PROTECT) = BOOT_FLASH_WRITE_PROTECT_WRITE_PROTECT_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_disable_register_write_protect(uintptr_t base)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_UNLOCK_1) = BOOT_FLASH_UNLOCK_1_CODE0;
	HWREG(base + BOOT_FLASH_UNLOCK_2) = BOOT_FLASH_UNLOCK_2_CODE1;
	HWREG(base + BOOT_FLASH_UNLOCK_1) = BOOT_FLASH_UNLOCK_1_CODE2;

	HWREG(base + BOOT_FLASH_WRITE_PROTECT) &= ~BOOT_FLASH_WRITE_PROTECT_WRITE_PROTECT_M;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_get_user_opt_infomation(uintptr_t base, boot_flash_user_opt_info_t *info)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	info->boot_pin_sel = HWREG(base + BOOT_FLASH_SET_USER_OPT_0);

	info->user_opt_1 = HWREG(base + BOOT_FLASH_SET_USER_OPT_1);

	return BOOT_FLASH_STAT_SUCCESS;
}

void boot_flash_get_addr_info(uint32_t address, boot_flash_addr_info_t *info)

{
	boot_flash_user_opt_info_t opt_info;
	uint32_t actual_addr;

	boot_flash_get_user_opt_infomation(BOOT_FLASH_EFC0_ADDR, &opt_info);

	if (opt_info.boot_dual_bank_mode) {
		if (address < BOOT_FLASH_BANK1_MIN_ADDR)
			info->flash_efc_addr = BOOT_FLASH_EFC0_ADDR;
		else
			info->flash_efc_addr = BOOT_FLASH_EFC1_ADDR;

		info->address = address;
		info->bank_mode = 1;
		info->flash_page = (address & BOOT_FLASH_MAIN_ADDR_MASK) / BOOT_FLASH_DUAL_BANK_SECTOR_SIZE;
	} else {
		if (address % 32)
			info->flash_efc_addr = BOOT_FLASH_EFC1_ADDR;
		else
			info->flash_efc_addr = BOOT_FLASH_EFC0_ADDR;

		info->bank_mode = 0;

		actual_addr = address / 32U;
		actual_addr *= 16U;

		info->address = actual_addr;
		info->flash_page = (info->address & BOOT_FLASH_MAIN_ADDR_MASK) /
											BOOT_FLASH_DUAL_BANK_SECTOR_SIZE;
	}
}

int boot_flash_config_efc_cmd(uintptr_t base, boot_flash_efc_cmd_t cmd)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_CONTROL) = cmd;

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_erase_main_flash_sector(uint32_t address)
{
	boot_flash_addr_info_t info;
	uint32_t current_addr;

	if(address < BOOT_FLASH_BANK_MIN_ADDR || address > BOOT_FLASH_BANK_MAX_ADDR)
		return BOOT_FLASH_STAT_ERR;

	boot_flash_get_addr_info(address, &info);

	current_addr = info.flash_page * BOOT_FLASH_DUAL_BANK_SECTOR_SIZE;

	HWREG(info.flash_efc_addr + BOOT_FLASH_ADDRESS) = current_addr;
	boot_flash_config_efc_cmd(info.flash_efc_addr, BOOT_FLASH_EFC_CMD_ERASE_PAGE);

	if (info.bank_mode == 0) {
		if (info.flash_efc_addr == BOOT_FLASH_EFC0_ADDR) {
			HWREG(BOOT_FLASH_EFC1_ADDR + BOOT_FLASH_ADDRESS) = current_addr;
			boot_flash_config_efc_cmd(BOOT_FLASH_EFC1_ADDR, BOOT_FLASH_EFC_CMD_ERASE_PAGE);
		} else {
			HWREG(BOOT_FLASH_EFC0_ADDR + BOOT_FLASH_ADDRESS) = current_addr;
			boot_flash_config_efc_cmd(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_EFC_CMD_ERASE_PAGE);
		}
	}

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_write_main_flash(uint32_t address, uint8_t *buf)
{
	boot_flash_addr_info_t info;

	if(address < BOOT_FLASH_BANK_MIN_ADDR || address > BOOT_FLASH_BANK_MAX_ADDR)
		return BOOT_FLASH_STAT_ERR;

	if (address % BOOT_FLASH_MIN_WRITE_SIZE != 0)
		return BOOT_FLASH_STAT_ERR;

	boot_flash_get_addr_info(address, &info);

	HWREG(info.flash_efc_addr + BOOT_FLASH_ADDRESS) = info.address & BOOT_FLASH_MAIN_ADDR_MASK;

	HWREG(info.flash_efc_addr + BOOT_FLASH_DATA_0) = *(uint32_t *)buf;
	HWREG(info.flash_efc_addr + BOOT_FLASH_DATA_1) = *(uint32_t *)(buf + 0x4);
	HWREG(info.flash_efc_addr + BOOT_FLASH_DATA_2) = *(uint32_t *)(buf + 0x8);
	HWREG(info.flash_efc_addr + BOOT_FLASH_DATA_3) = *(uint32_t *)(buf + 0xc);

	boot_flash_config_efc_cmd(info.flash_efc_addr, BOOT_FLASH_EFC_CMD_PROGRAM_START);

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_get_efc_status(uintptr_t base, boot_flash_efc_status_t *status)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	status->efc_status = HWREG(base + BOOT_FLASH_STATUS);

	return BOOT_FLASH_STAT_SUCCESS;
}

int boot_flash_clear_efc_status(uintptr_t base, uint32_t status)
{
	if (!boot_flash_is_base_valid(base))
		return BOOT_FLASH_STAT_ERR;

	HWREG(base + BOOT_FLASH_STATUS) = status;

	return BOOT_FLASH_STAT_SUCCESS;
}

#endif
