/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if (GS32F00xx == 0x3000)

#include "boot_gpio_v3_0.h"

#if (GS32_PART_NUM == 0x035 || GS32_PART_NUM == 0x027)
void boot_gpio_set_pin_config(uint32_t pinConfig)
{
    uint32_t muxRegAddr;
    uint32_t pinMask, shiftAmt;

    muxRegAddr = (uint32_t)IO_CFG_BASE + (pinConfig >> 16);
    shiftAmt   = ((pinConfig >> 8) & (uint32_t)0xFFU);
    pinMask    = (uint32_t)0x3U << shiftAmt;

    //
    // Clear fields in MUX register first to avoid glitches
    //
    HWREG(muxRegAddr) &= ~pinMask;

    //
    // Write value into MUX register
    //
    HWREG(muxRegAddr) |= ((pinConfig & (uint32_t)0x3U) << shiftAmt);

}
#else
void boot_gpio_set_pin_config(uint32_t pinConfig)
{
    uint32_t muxRegAddr;
    uint32_t pinMask, shiftAmt;

    muxRegAddr = (uint32_t)GPIOCTRL_BASE + (pinConfig >> 16);
    shiftAmt   = ((pinConfig >> 8) & (uint32_t)0xFFU);
    pinMask    = (uint32_t)0x3U << shiftAmt;

    //
    // Clear fields in MUX register first to avoid glitches
    //
    HWREG(muxRegAddr) &= ~pinMask;

    if (muxRegAddr != (GPIOCTRL_BASE + GPIO_O_GPGMUX1)) // Non GPOG configuration
    {
        //
        // Write value into GMUX register
        //
        HWREG(muxRegAddr + GPIO_MUX_TO_GMUX) =
            (HWREG(muxRegAddr + GPIO_MUX_TO_GMUX) & ~pinMask) |
            (((pinConfig >> 2) & (uint32_t)0x3U) << shiftAmt);

        //
        // Write value into MUX register
        //
        HWREG(muxRegAddr) |= ((pinConfig & (uint32_t)0x3U) << shiftAmt);
    } else // GPOG configuration
    {
        //
        // Write value into GMUX register
        //
        HWREG(GPIOCTRL_BASE + GPIO_O_GPGGMUX1) =
            (HWREG(GPIOCTRL_BASE + GPIO_O_GPGGMUX1) & ~pinMask) |
            (((pinConfig >> 2) & (uint32_t)0x3U));

        //
        // Write value into MUX register
        //
        HWREG(muxRegAddr) |= ((pinConfig & (uint32_t)0x3U));
    }
}
#endif

#if !(GS32_PART_NUM == 0x035 || GS32_PART_NUM == 0x027)

static HwPinType boot_check_pin_type(int pin_num)
{
    // AIO pin judgment
    switch (pin_num) {
    case 231:
    case 232:
    case 234:
    case 235:
    case 238:
    case 239:
    case 241:
    case 244:
    case 248:
        return HW_PIN_TYPE_AIO;
    }

    // AGPIO pin determination
    switch (pin_num) {
    case 11:
    case 12:
    case 13:
    case 16:
    case 17:
    case 20:
    case 21:
    case 24:
    case 28:
    case 33:
    case 200:
    case 224:
    case 225:
    case 226:
    case 227:
    case 228:
    case 230:
    case 233:
    case 236:
    case 237:
    case 242:
    case 245:
        return HW_PIN_TYPE_AGPIO;
    }

    // If it is not AIO or AGPIO, it is considered a regular GPIO
    if (pin_num >= 0) {
        return HW_PIN_TYPE_GPIO;
    }

    // Negative or other invalid valuess
    return HW_PIN_TYPE_UNKNOWN;
}
#endif

void boot_gpio_set_analog_mode(uint32_t pin, GPIO_AnalogMode mode)
{
#if !(GS32_PART_NUM == 0x035 || GS32_PART_NUM == 0x027)
    uint32_t pinMask;
    HwPinType pinType;

    pinType = boot_check_pin_type(pin);

    //
    // Check the arguments.
    //
    ASSERT((pinType == HW_PIN_TYPE_AIO) ||
        (pinType == HW_PIN_TYPE_AGPIO));

    if (pin != 200) // Non GPOG configuration
    {
        volatile uint32_t *gpioBaseAddr;
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);
        pinMask      = (uint32_t)1U << (pin % 32U);

        //
        // Set the analog mode selection.
        //
        if (mode == GPIO_ANALOG_ENABLED)
        {
            //
            // Enable analog mode
            //
            gpioBaseAddr[GPIO_GPxAMSEL_INDEX] |= pinMask;
            if(pinType == HW_PIN_TYPE_AGPIO)
            {
                //
                // Set AGPIOCTL
                //
            	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLA +
                       ((pin / 32U) * 0x4U)) |= (pinMask);
            }
        }
        else
        {
            //
            // Disable analog mode
            //
            gpioBaseAddr[GPIO_GPxAMSEL_INDEX] &= ~pinMask;
            if(pinType == HW_PIN_TYPE_AGPIO)
            {
                //
                // Clear AGPIOCTL
                //
            	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLA +
                       ((pin / 32U) * 0x4U)) &= ~(pinMask);
            }
        }
    }
    else
    {
        pinMask = (uint32_t)1U;
        //
        // Set the analog mode selection.
        //
        if (mode == GPIO_ANALOG_ENABLED)
        {
            //
            // Enable analog mode
            //
            HWREG(GPIOCTRL_BASE + GPIO_O_GPGAMSEL) |= pinMask;
        }
        else
        {
            //
            // Disable analog mode
            //
            HWREG(GPIOCTRL_BASE + GPIO_O_GPGAMSEL) &= ~pinMask;
        }
    }
#endif
}
#if (GS32_PART_NUM == 0x035 || GS32_PART_NUM == 0x027)
void boot_gpio_set_pad_config(uint32_t pin, uint32_t pinType)
{
    volatile uint32_t *gpioBaseAddr1;
    volatile uint32_t *gpioBaseAddr2;
    uint32_t           PudIndex;
    uint32_t           PddIndex;
    uint32_t           OutInvIndex;
    uint32_t           pinMask;

    gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE +
                            (pin / 32U) * (GPIO_O_GPBCTRL_3527 - GPIO_O_GPACTRL_3527));
    gpioBaseAddr2 = (uint32_t *)(IO_CFG_BASE +
                            GPIO_O_GPAPDD + (pin / 32U) * GPIO_EXT_REGS_STEP);

    PudIndex      = GPIO_GPxPUD_INDEX;
    PddIndex      = GPIO_GPxPDD_INDEX;
    OutInvIndex   = GPIO_GPxOUTINV_INDEX;
    pinMask       = (uint32_t)1U << (pin % 32U);

    //
    // Enable pull-up if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_PULLUP) != 0U) {
        gpioBaseAddr1[PudIndex] &= ~pinMask;
    } else {
        gpioBaseAddr1[PudIndex] |= pinMask;
    }

    //
    // Pull-Down if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_PULLDOWN) != 0U) {
        gpioBaseAddr2[PddIndex] &= ~pinMask;
    } else {
        gpioBaseAddr2[PddIndex] |= pinMask;
    }

    //
    // Inverted Push-Pull if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_OUTPUT_INVERT) != 0U) {
        gpioBaseAddr2[OutInvIndex] |= pinMask;
    } else {
        gpioBaseAddr2[OutInvIndex] &= ~pinMask;
    }
}
#else
void boot_gpio_set_pad_config(uint32_t pin, uint32_t pinType)
{
	volatile uint32_t *gpioBaseAddr1;
	volatile uint32_t *gpioBaseAddr2;
	uint32_t PudIndex;
	uint32_t InvIndex;
	uint32_t PddIndex;
	uint32_t OutInvIndex;
	uint32_t pinMask;

	if (pin != 200) {			/* Non GPOG configuration */
		gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE +
						(pin / 32U) * GPIO_CTRL_REGS_STEP);
		PudIndex = GPIO_GPxPUD_INDEX;
		InvIndex = GPIO_GPxINV_INDEX;
		PddIndex = GPIO_GPxPDD_INDEX;
		OutInvIndex	= GPIO_GPxOUTINV_INDEX;
		pinMask	= (uint32_t)1U << (pin % 32U);
	} else {
		gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
		pinMask = (uint32_t)1U << (0U);

		PudIndex = (GPIO_O_GPGPUD - GPIO_O_GPGCTRL) / 4;
		InvIndex = (GPIO_O_GPGINV - GPIO_O_GPGCTRL) / 4;
		PddIndex = (GPIO_O_GPGPDD - GPIO_O_GPGPDD) / 4;
		OutInvIndex = (GPIO_O_GPGOUTINV - GPIO_O_GPGPDD) / 4;
	}
	gpioBaseAddr2 = (uint32_t *)(GPIOEXT_BASE +
								(pin / 32U) * GPIO_EXT_REGS_STEP);

	/* Enable pull-up if necessary */
	if ((pinType & BOOT_GPIO_PIN_TYPE_PULLUP) != 0U) {
		gpioBaseAddr1[PudIndex] &= ~pinMask;
	} else {
		gpioBaseAddr1[PudIndex] |= pinMask;
	}

	/* Invert polarity if necessary */
	if ((pinType & BOOT_GPIO_PIN_TYPE_INVERT) != 0U) {
		gpioBaseAddr1[InvIndex] |= pinMask;
	} else {
		gpioBaseAddr1[InvIndex] &= ~pinMask;
	}

	/* Pull-Down if necessary */
	if ((pinType & BOOT_GPIO_PIN_TYPE_PULLDOWN) != 0U) {
		gpioBaseAddr2[PddIndex] &= ~pinMask;
	} else {
		gpioBaseAddr2[PddIndex] |= pinMask;
	}

	/* Inverted Push-Pull if necessary */
	if ((pinType & BOOT_GPIO_PIN_TYPE_SCHMITT) != 0U) {
		gpioBaseAddr2[OutInvIndex] |= pinMask;
	} else {
		gpioBaseAddr2[OutInvIndex] &= ~pinMask;
	}
}
#endif

uint16_t boot_gpio_read_pin(uint16_t pin)
{

#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)

    switch (pin) {
    case 56:
        return (HWREG(IO_CFG_BASE + GPIO_O_AIODAT_RE) & 0x1000) ? 1U : 0U;
    case 57:
        return (HWREG(IO_CFG_BASE + GPIO_O_AIODAT_RE) & 0x4000) ? 1U : 0U;
    default:
        break;
    }

#endif

    volatile uint32_t *gpioDataReg;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));
    gpioDataReg = (uint32_t *)(GPIODATA_BASE + (pin / 32U) * GPIO_DATA_REGS_STEP);

    if (pin != 200) {
        return ((gpioDataReg[GPIO_GPxDAT_INDEX] >> (pin % 32U)) & (uint32_t)0x1U);
    } else {
        return (gpioDataReg[GPIO_GPxDAT_INDEX] & (uint32_t)0x1U);
    }
}

void boot_gpio_set_qualification_mode(uint32_t pin, GPIO_QualificationMode qualification)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           qSelIndex;
    uint32_t           shiftAmt;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    if (pin != 200) // Non GPOG configuration
    {
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);
        shiftAmt     = (uint32_t)GPIO_GPAQSEL1_GPIO1_S * (pin % 16U);
        qSelIndex    = GPIO_GPxQSEL_INDEX + ((pin % 32U) / 16U);
    } else {
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
        shiftAmt     = 0U;
        qSelIndex    = (GPIO_O_GPGQSEL1 - GPIO_O_GPGCTRL) / 4;
    }

    //
    // Write the input qualification mode to the register.
    //

    gpioBaseAddr[qSelIndex] &= ~((uint32_t)GPIO_GPAQSEL1_GPIO0_M << shiftAmt);
    gpioBaseAddr[qSelIndex] |= (uint32_t)qualification << shiftAmt;
}


void boot_gpio_set_clk_ref(GPIO_ClkRefMode mode)
{
	HWREG(IO_CFG_BASE + GPIO_O_CLK_REF_PIN_CTRL) = mode;
}

#endif		/* end of (GS32F00xx == 0x3000) */
