/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if (GS32F00xx == 0x3000)

#include "inc/hw_memmap.h"
#include "core_feature_cache.h"
#include "core_feature_base.h"

#include "boot_interrupt.h"
#include "boot_riscv_cache.h"
#include "boot_low_level_init_v3_0.h"
#include "boot_flash.h"
#include "boot_sysctl.h"
#include "boot_asysctl.h"

BOOT_TEXT void boot_low_level_init(uint32_t ahb_clk_freq)
{
	uint32_t flash_read_cycle;

	/* Enable I Cache */
	/* Check whether dcache real present or not */
	if (boot_ICache_present())
		boot_enable_ICache();

	/* Enable D Cache */
	/* Check whether dcache real present or not */
	if (boot_DCache_present())
		boot_enable_DCache();

	/**
	 * Do fence and fence.i to make sure previous
	 * ilm/dlm/icache/dcache control done
	 */
	__RWMB();
	__FENCE_I();

	/* Initialize interrupt. */
	boot_interrupt_init();

#if (GS32_PART_NUM == 0x5000 || GS32_PART_NUM == 0x4000)
	boot_sysctl_enable_io_mt48_49_56_57();
#else
	boot_sysctl_set_obs_sig_div(1);
#endif

	boot_sysctl_set_osc2_freq_range(BOOT_OSC2_FREQ_HIGH);
	boot_sysctl_set_osc2_coarse_internal_trim();
	boot_sysctl_set_osc2_fine_internal_trim();
	boot_sysctl_set_top_ldo_1v2_trim();
	boot_sysctl_set_ana_ldo_trim();
	boot_sysctl_set_osc1_trim();

	/* Set Flash bank wait read cycle. */
	if (ahb_clk_freq < 60000000U)
		flash_read_cycle = 1U;
	else if (ahb_clk_freq >= 60000000U && ahb_clk_freq < 90000000U)
		flash_read_cycle = 2U;
	else if (ahb_clk_freq >= 90000000U && ahb_clk_freq < 120000000U)
		flash_read_cycle = 3U;
	else if (ahb_clk_freq >= 120000000U && ahb_clk_freq < 180000000U)
		flash_read_cycle = 4U;
	else if (ahb_clk_freq >= 180000000U && ahb_clk_freq < 210000000U)
		flash_read_cycle = 5U;
	else if (ahb_clk_freq >= 210000000U && ahb_clk_freq < 260000000U)
		flash_read_cycle = 6U;
	else if (ahb_clk_freq >= 260000000U && ahb_clk_freq < 310000000U)
		flash_read_cycle = 7U;
	else
		flash_read_cycle = 8U;

	boot_flash_disable_register_write_protect(BOOT_FLASH_EFC0_ADDR);
	boot_flash_disable_register_write_protect(BOOT_FLASH_EFC1_ADDR);

	boot_flash_set_read_wait_timing(BOOT_FLASH_EFC0_ADDR, flash_read_cycle);
	boot_flash_set_read_wait_timing(BOOT_FLASH_EFC1_ADDR, flash_read_cycle);

	boot_flash_disable_fast_program_mode(BOOT_FLASH_EFC0_ADDR);
	boot_flash_disable_fast_program_mode(BOOT_FLASH_EFC1_ADDR);

	boot_flash_enable_register_write_protect(BOOT_FLASH_EFC0_ADDR);
	boot_flash_enable_register_write_protect(BOOT_FLASH_EFC1_ADDR);
}

#endif	/* GS32F00xx */
