/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "boot_riscv_eclic.h"
#include "riscv_encoding.h"
#include "core_feature_base.h"

BOOT_TEXT Core_IRegion_Info_t boot_get_core_iregion_info(void)
{
	unsigned long mcfg_info;
	Core_IRegion_Info_t iregion;

	mcfg_info = __RV_CSR_READ(CSR_MCFG_INFO);

	if (mcfg_info & MCFG_INFO_IREGION_EXIST) {
		iregion.iregion_base = (__RV_CSR_READ(CSR_MIRGB_INFO) >> 10) << 10;
		iregion.eclic_base = iregion.iregion_base + IREGION_ECLIC_OFS;
		iregion.systimer_base = iregion.iregion_base + IREGION_TIMER_OFS;
		iregion.smp_base = iregion.iregion_base + IREGION_SMP_OFS;
		iregion.idu_base = iregion.iregion_base + IREGION_IDU_OFS;
	} else {
		iregion.eclic_base = BOOT_FALLBACK_DEFAULT_ECLIC_BASE;
		iregion.systimer_base = BOOT_FALLBACK_DEFAULT_SYSTIMER_BASE;
	}

	return iregion;
}

BOOT_TEXT void boot_eclic_set_machine_mode(Eclic_Reg_t *eclic, uint8_t mth)
{
	eclic->MTH = mth;
}

BOOT_TEXT uint32_t boot_eclic_get_info_ctl_bits(Eclic_Reg_t *eclic)
{
	return ((uint32_t)((eclic->INFO & BOOT_CLIC_CLICINFO_CTLBIT_Msk) >> BOOT_CLIC_CLICINFO_CTLBIT_Pos));
}

BOOT_TEXT uint32_t boot_eclic_get_cfg_nl_bits(Eclic_Reg_t *eclic)
{
	return ((uint32_t)((eclic->CFG & BOOT_CLIC_CLICCFG_NLBIT_Msk) >> BOOT_CLIC_CLICCFG_NLBIT_Pos));
}

BOOT_TEXT void boot_eclic_set_cfg_nl_bits(Eclic_Reg_t *eclic, uint32_t nlbits)
{
	eclic->CFG &= ~BOOT_CLIC_CLICCFG_NLBIT_Msk;
	eclic->CFG |= (uint8_t)((nlbits << BOOT_CLIC_CLICCFG_NLBIT_Pos) & BOOT_CLIC_CLICCFG_NLBIT_Msk);
}

BOOT_TEXT void boot_eclic_set_trigger_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, BOOT_ECLIC_TRIGGER_t trig)
{
	eclic->CTRL[IRQn].INTATTR &= ~BOOT_CLIC_INTATTR_TRIG_Msk;
	eclic->CTRL[IRQn].INTATTR |= (uint8_t)(trig << BOOT_CLIC_INTATTR_TRIG_Pos);
}

BOOT_TEXT void boot_eclic_enable_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn)
{
	eclic->CTRL[IRQn].INTIE |= BOOT_CLIC_INTIE_IE_Msk;
}

BOOT_TEXT void boot_eclic_disable_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn)
{
	eclic->CTRL[IRQn].INTIE &= ~BOOT_CLIC_INTIE_IE_Msk;
}

BOOT_TEXT void boot_eclic_set_ShvIRQ(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint32_t shv)
{
	eclic->CTRL[IRQn].INTATTR &= ~BOOT_CLIC_INTATTR_SHV_Msk;
	eclic->CTRL[IRQn].INTATTR |= (uint8_t)(shv << BOOT_CLIC_INTATTR_SHV_Pos);
}

BOOT_TEXT uint8_t boot_eclic_get_ctrl_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn)
{
	return (eclic->CTRL[IRQn].INTCTRL);
}

BOOT_TEXT void boot_eclic_set_ctrl_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint8_t intctrl)
{
	eclic->CTRL[IRQn].INTCTRL = intctrl;
}

BOOT_TEXT void boot_eclic_set_level_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint8_t lvl_abs)
{
	uint8_t nlbits;
	uint8_t maxlvl;
	uint8_t lvl;
	uint8_t cur_ctrl;
	uint8_t intctlbits;

	nlbits = boot_eclic_get_cfg_nl_bits(eclic);
	intctlbits = (uint8_t)boot_eclic_get_info_ctl_bits(eclic);

	if (nlbits == 0)
		return;

	if (nlbits > intctlbits)
		nlbits = intctlbits;

	maxlvl = ((1 << nlbits) - 1);

	if (lvl_abs > maxlvl)
		lvl_abs = maxlvl;

	lvl = lvl_abs << (BOOT_ECLIC_MAX_NLBITS - nlbits);
	cur_ctrl = boot_eclic_get_ctrl_irq(eclic, IRQn);
	cur_ctrl = cur_ctrl << nlbits;
	cur_ctrl = cur_ctrl >> nlbits;
	boot_eclic_set_ctrl_irq(eclic, IRQn, (cur_ctrl | lvl));
}

BOOT_TEXT void boot_eclic_set_priority_irq(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uint8_t priority)
{
	uint8_t nlbits;
	uint8_t intctlbits;

	nlbits = boot_eclic_get_cfg_nl_bits(eclic);
	intctlbits = (uint8_t)boot_eclic_get_info_ctl_bits(eclic);

	if (nlbits < intctlbits) {
		uint8_t maxpri = ((1 << (intctlbits - nlbits)) - 1);
		if (priority > maxpri)
			priority = maxpri;

		priority = priority << (BOOT_ECLIC_MAX_NLBITS - intctlbits);
		uint8_t mask = ((uint8_t)(-1)) >> intctlbits;
		priority = priority | mask;
		uint8_t cur_ctrl = boot_eclic_get_ctrl_irq(eclic, IRQn);
		cur_ctrl = cur_ctrl >> (BOOT_ECLIC_MAX_NLBITS - nlbits);
		cur_ctrl = cur_ctrl << (BOOT_ECLIC_MAX_NLBITS - nlbits);
		boot_eclic_set_ctrl_irq(eclic, IRQn, (cur_ctrl | priority));
	}
}
